#include <matlabint.h>
#include <matlabint_workspace.h>
#include <matlabint_mesh.h>
#include <getfem_import.h>
/*
  $Id: gf_mesh.C,v 1.3 2003/07/25 09:04:30 pommier Exp $

  ChangeLog:
  $Log: gf_mesh.C,v $
  Revision 1.3  2003/07/25 09:04:30  pommier
  *** empty log message ***

  Revision 1.2  2003/07/23 18:31:14  pommier
  *** empty log message ***

  Revision 1.1  2003/05/22 13:18:03  pommier
  regroupement de tous les fichiers dans ./src , et mise en place des RPC

  Revision 1.20  2003/05/05 07:59:10  pommier
  ajout gf_slice , gf_slice_get

  Revision 1.19  2003/02/27 16:34:17  pommier
  improve friendlyness with gcc2.95

  Revision 1.18  2003/02/17 11:44:15  pommier
  doc updates

  Revision 1.17  2003/02/14 13:02:26  pommier
  divers modifs

  Revision 1.16  2003/02/11 11:01:45  pommier
  ajout des interfaces oo matlab

  Revision 1.15  2003/02/04 10:33:30  pommier
  modifs diverses pour les triangles/tetra avec trans geo non lin

  Revision 1.14  2003/01/30 16:11:00  pommier
  *** empty log message ***

  Revision 1.13  2003/01/28 13:50:58  pommier
  updates sgi + update eval_on_P1_tri_mesh

 */
using namespace matlabint;

std::string name_of_linear_qk_trans(int dim) {
  switch (dim) {
  case 1: return "GT_PK(1,1)";
  default: return std::string("GT_LINEAR_PRODUCT(") + name_of_linear_qk_trans(dim-1) + std::string(",GT_PK(1,1))");
  }
}

static void
cartesian_mesh(getfem::getfem_mesh *pmesh, matlabint::mexargs_in &in)
{
  matlabint::size_type dim = in.remaining();
  
  if (dim == 0) THROW_BADARG( "not enough input arguments");

  std::vector<mlab_vect> ppos(dim);
  std::vector<size_type> npts(dim);
  dal::uint32_type grid_npoints=1, grid_nconvex=1;
  for (size_type i = 0; i < dim; i++) {
    ppos[i] = in.pop().to_scalar_vector();
    npts[i] = ppos[i].size();
    grid_npoints *= npts[i];
    grid_nconvex *= (npts[i]-1);
  }
  
  /* add the points in 'fortran style' order */
  getfem::base_node pt(dim);
  for (dal::uint32_type i=0; i < grid_npoints; i++) {
    dal::uint32_type k = i;    
    for (matlabint::size_type j = 0; j < dim; j++) {
      pt[j] = ppos[j][k % (npts[j])];
      k /= (npts[j]);
    }

    size_type id_pt = pmesh->add_point(pt);
    if (id_pt != i) {
      THROW_ERROR( 
		"something has changed in getfem, you need to reconsider "
		"gf_mesh('cartesian')\nfor point " << i << 
		", the index is " << id_pt << endl);
    }
    /*    if (i == grid_npoints-1) {
      cerr << "nb de pts ajoutes: " << grid_npoints << " id dernier =" << id_pt << endl;
      }*/
  }
  

  std::vector<int> ipt(dim);
  std::vector<getfem::base_node> pts(1 << dim+1);
  
  bgeot::pgeometric_trans pgt = bgeot::geometric_trans_descriptor(name_of_linear_qk_trans(dim));

  /* add the convexes */
  for (dal::uint32_type i=0; i < grid_nconvex; i++) {
    dal::uint32_type k = i;

    /* find point location */
    for (matlabint::size_type j = 0; j < dim; j++) {
      ipt[j] = k % (npts[j]-1);
      k /= (npts[j]-1);
    }

    /* build the vertices list */
    for (matlabint::size_type j = 0; j < (unsigned(1)<<dim); j++) {
      pts[j].resize(dim);
      for (dal::uint32_type d=0; d < dim; d++) {
	if ((j >> d) & 1) {
	  pts[j][d] = ppos[d][ipt[d]+1];
	} else {
	  pts[j][d] = ppos[d][ipt[d]];
	}
      }
    }

    // we don't use the add_parall since the geometric transformatoin
    // is linear (the mesh is cartesian)
    //pmesh->add_parallelepiped_by_points(dim, pts.begin());
    pmesh->add_convex_by_points(pgt, pts.begin()); 
  }
}


static void
triangles_grid_mesh(getfem::getfem_mesh *pmesh, matlabint::mexargs_in &in)
{
  if (in.remaining() != 2) THROW_BADARG( "not enough input arguments");
  
  mlab_vect X = in.pop().to_scalar_vector();
  mlab_vect Y = in.pop().to_scalar_vector();
  if (X.size() < 1 || Y.size() < 1) THROW_BADARG( "bad dimensions");

  size_type ni = Y.size(), nj = X.size();
  for (size_type i=0; i < ni; i++) {
    for (size_type j=0; j < nj; j++) {
      getfem::base_node pt(2);
      pt[0] = X[j]; pt[1] = Y[i];
      //      cerr << "pt = " << pt << endl;
      pmesh->add_point(pt);
    }
  }
  for (size_type i=0; i < ni-1; i++) {
    for (size_type j=0; j < nj-1; j++) {
      //cerr << "i=" << i <<" j=" << j << endl;
      pmesh->add_triangle(i*nj + j, (i+1)*nj + j  , (i+1)*nj + j+1);
      pmesh->add_triangle(i*nj + j,     i*nj + j+1, (i+1)*nj + j+1);
    }
  }
}

static void
curved_mesh(getfem::getfem_mesh *dest_mesh, matlabint::mexargs_in &in)
{
  const getfem::getfem_mesh *src_mesh = in.pop().to_const_mesh();
  mlab_vect F = in.pop().to_scalar_vector(src_mesh->points().index().last()+1);

  int dim = src_mesh->dim();
  bgeot::base_node pt(dim+1);
  dest_mesh->clear();
  bgeot::size_type i, nbpt = src_mesh->points().index().last()+1;
  for (i = 0; i < nbpt; ++i)
  {
    std::copy(src_mesh->points()[i].begin(), src_mesh->points()[i].end(),
	      pt.begin());
    pt[dim] = F[i];
    dest_mesh->add_point(pt);
  }

  dal::bit_vector nn = src_mesh->convex_index();
  for (i << nn; i != bgeot::size_type(-1); i << nn)
  {
    dest_mesh->add_convex(src_mesh->trans_of_convex(i),
		       src_mesh->ind_points_of_convex(i).begin());
  }
}

static void
prismatic_mesh(getfem::getfem_mesh *dest_mesh, matlabint::mexargs_in &in)
{
  const getfem::getfem_mesh *src_mesh = in.pop().to_const_mesh();
  unsigned nblay = in.pop().to_integer(0,2500000);
  int dim = src_mesh->dim();
  bgeot::base_node pt(dim+1);

  dest_mesh->clear();
  bgeot::size_type i, nbpt = src_mesh->points().index().last()+1;
  
  for (i = 0; i < nbpt; ++i)
  {
    std::copy(src_mesh->points()[i].begin(), src_mesh->points()[i].end(),
	      pt.begin());
    pt[dim] = 0.0;
    for (size_type j = 0; j <= nblay; ++j, pt[dim] += 1.0 / nblay)
      dest_mesh->add_point(pt);
  }
  
  dal::bit_vector nn = src_mesh->convex_index();
  std::vector<bgeot::size_type> tab(5);
  for (i << nn; i != bgeot::size_type(-1); i << nn) // optimisable.
  {
    size_type nbp = src_mesh->nb_points_of_convex(i);
    if (2*nbp > tab.size()) tab.resize(2*nbp);
    for (size_type j = 0; j < nblay; ++j)
    {
      for (size_type k = 0; k < nbp; ++k)
	tab[k] = (nblay+1)*src_mesh->ind_points_of_convex(i)[k] + j;
      for (size_type k = 0; k < nbp; ++k)
	tab[k+nbp] = (nblay+1)*src_mesh->ind_points_of_convex(i)[k] + j + 1;
      bgeot::pgeometric_trans pgt = 
	//bgeot::product_geotrans(src_mesh->trans_of_convex(i), bgeot::simplex_geotrans(1,1));
      
	bgeot::geometric_trans_descriptor("GT_LINEAR_PRODUCT(" +
					  bgeot::name_of_geometric_trans(src_mesh->trans_of_convex(i)) +
					  ",GT_PK(1,1))");
      
      dest_mesh->add_convex(pgt, tab.begin());
      
      /*
      bgeot::convex_product_trans(
		         src_mesh->trans_of_convex(i),
			 bgeot::simplex_trans(1, 1)), tab.begin());
      */
    }
  }  
}

static void
pt2D_mesh(getfem::getfem_mesh *mesh, matlabint::mexargs_in &in)
{
  mlab_vect P = in.pop().to_scalar_vector(2, -1);
  mlab_vect T = in.pop().to_scalar_vector(-1, -1);
  if (T.getm() != 3 && T.getm() != 4) THROW_BADARG("wrong nb of rows for t, 3 or 4 rows were expected");
  id_type zone = 0;
  if (in.remaining()) zone = in.pop().to_integer(1,65000);

  getfem::base_node pt(2);
  size_type warn_cnt = 0;
  std::vector<id_type> id_tab(P.getn());
  for (size_type i = 0; i < P.getn(); ++i) {
    pt[0] = P[2*i]; pt[1] = P[2*i+1];
    id_tab[i] = mesh->add_point(pt);

    /* une hypothse bien commode pour la "compatibilit pdetool" */
    if (id_tab[i] != i && warn_cnt++ == 0) {
      DAL_WARNING(1, "The numbering of mesh points will be different, pt#" << 
                  i << " gets id#" << id_tab[i]);
    }
  }

  for (size_type i = 0; i < T.getn(); ++i) {
    id_type u = id_type(T(0,i)), v = id_type(T(1,i)), w = id_type(T(2,i));
    if (u > P.size() || v > P.size() || w > P.size())
      THROW_BADARG( "Bad triangulation.");
    if (zone == 0 || T.getm()==3 || zone == id_type(T(3,i)))
      mesh->add_triangle(id_tab[u-1], id_tab[v-1], id_tab[w-1]);
  }
}

/*MLABCOM

  FUNCTION M = gf_mesh([operation [, args]])

  General constructor for mesh object. Returns a getfem handle 
  to the newly created mesh object.
  
  * M = gf_mesh('empty',int dim) 
  Creates a new empty mesh.

  * M = gf_mesh('cartesian', vec X[, vec Y[, vec Z,..]])
  Builds quickly a cartesian mesh.

  * M = gf_mesh('triangles grid', vec X[, vec Y[, vec Z,..]])
  Builds quickly a regular planar mesh composed of triangles.

  * M = gf_mesh('curved', const_mesh I, vec F)
  builds a curved (n+1)-dimensions mesh from a n-dimensions mesh:
  The mesh m has one additional dimension. The additional coordinate
  is given by the vector F. This can be used to obtain meshes for
  shells.  (I may be a mesh_fem, in that case its linked mesh will be
  used)

  * M = gf_mesh('prismatic', const_mesh I, int K)
  Builds a prismatic mesh M from a mesh I. In the additional dimension
  there is K layers of elements built from 0 to 1.

  * M = gf_mesh('pt2D', P, T[, n])
  Converts a pdetool mesh exported in variables p and t into a GETFEM
  mesh i.  n is optional and is a zone number. If n is specified only
  the zone number 'n' is converted (in that case, T is expected to 
  have 4 rows, the fourth containing zone numbers).

  * M = gf_mesh('load', string FILENAME)
  Loads a mesh from an ascii file (which may have been created by
  gf_mesh_get(M,'save',FILENAME))

  * M = gf_mesh('from string', string s)
  Loads a mesh from a string description (i.e. a string returned by
  gf_mesh_get(M,'char')).

  * M = gf_mesh('import', string FORMAT, string FILENAME)
  Import a mesh, FORMAT may be:
    - 'gmsh' for a mesh created with gmsh ( http://www.geuz.org/gmsh )
    - 'gid'  for a mesh created with GiD  ( http://gid.cimne.upc.es )

  $Id: gf_mesh.C,v 1.3 2003/07/25 09:04:30 pommier Exp $
MLABCOM*/
void gf_mesh(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  matlabint_mesh *mi_mesh = new matlabint_mesh();
  out.pop().from_object_id(workspace().push_object(mi_mesh), MESH_CLASS_ID);
  getfem::getfem_mesh *pmesh = &mi_mesh->mesh();


  if (in.narg() < 1) {
    THROW_BADARG( "Wrong number of input arguments");
  }
  std::string cmd    = in.pop().to_string();
  if (check_cmd(cmd, "empty", in, out, 1, 1, 0, 1)) {
    size_type dim = in.pop().to_integer(1,255);
    getfem::base_node pt(dim);
    pmesh->sup_point(pmesh->add_point(pt)); /* just to initialize the dimension of the mesh
					       (this is not very nice, i know) */
  } else if (check_cmd(cmd, "cartesian", in, out, 1, 32, 0, 1)) {
    cartesian_mesh(pmesh, in);
  } else if (check_cmd(cmd, "triangles grid", in, out, 2, 2, 0, 1)) {
    triangles_grid_mesh(pmesh, in);
  } else if (check_cmd(cmd, "curved", in, out, 2, 2, 0, 1)) {
    curved_mesh(pmesh, in);
  } else if (check_cmd(cmd, "prismatic", in, out, 2, 2, 0, 1)) {
    prismatic_mesh(pmesh, in);
  } else if (check_cmd(cmd, "pt2D", in, out, 2, 3, 0, 1)) {
    pt2D_mesh(pmesh, in);
  } else if (check_cmd(cmd, "load", in, out, 1, 1, 0, 1)) {
    std::string fname = in.pop().to_string();
    pmesh->read_from_file(fname);
  } else if (check_cmd(cmd, "from string", in, out, 1, 1, 0, 1)) {
    //#ifdef USING_BROKEN_GCC295 /* with memory leaks */
    //      std::strstream ss; ss << in.pop().to_string().c_str();
    //#else
    //    std::stringstream ss(in.pop().to_string(), std::stringstream::in);
    //ndif
    std::stringstream ss(in.pop().to_string());
    pmesh->read_from_file(ss);
  } else if (check_cmd(cmd, "import", in, out, 2, 2, 0, 1)) {
    std::string fmt = in.pop().to_string();
    std::string fname = in.pop().to_string();
    if (cmd_strmatch(fmt,"gmsh")) {
      getfem::import_gmsh_msh_file(fname, *pmesh);
    } else if (cmd_strmatch(fmt,"gid")) {
      getfem::import_gid_msh_file(fname, *pmesh);
    } else {
      THROW_BADARG("the mesh format \"" << fmt << "\" is unknown");
    }
  } else bad_cmd(cmd);
}
