#include <matlabint.h>
#include <matlabint_pfi.h>

using namespace matlabint;

static void check_not_exact(getfem::pintegration_method im) {
  if (im->is_exact()) THROW_ERROR("this has no meaning for exact integration methods");
}
/*MLABCOM
  FUNCTION I = gf_integ_get(F, ...)
    General function for querying information about FEM integration method objects.
    
    * gf_integ_get(IM,'is_exact')
    Returns true if the integration is an exact one (i.e. polynomial integration)

    * gf_integ_get(IM,'dim')

    * gf_integ_get(IM,'nbpts')
    Returns the number of integration points for the convex and each face (only
    for approximate methods)

    * gf_integ_get(IM,'pts')
    * gf_integ_get(IM,'coeffs')
    Returns the list of integration points (only for approximate methods) and 
    the associated coefficients.

    * gf_integ_get(IM,'face_pts',F)
    * gf_integ_get(IM,'face_coeffs',F)
    Lists the integration points for the face F, and the associated coefficients    

    * S=gf_integ_get(IM,'char')
    Ouput a (unique) string representation of the IM. This can be used to 
    perform comparisons between two different IM objects.
    
MLABCOM*/

void gf_integ_get(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 2) {
    THROW_BADARG( "Wrong number of input arguments");
  }
  getfem::pintegration_method im = in.pop().to_fem_interpolation();
  getfem::papprox_integration pai = im->is_exact() ? 0 : im->approx_method();
  size_type imdim = im->is_exact() ? 
    im->exact_method()->dim() : 
    im->approx_method()->dim();

  std::string cmd = in.pop().to_string();
  if (check_cmd(cmd, "is_exact", in, out, 0, 0, 0, 1)) {
    out.pop().from_scalar(im->is_exact() ? 1. : 0.);
  } else if (check_cmd(cmd, "dim", in, out, 0, 0, 0, 1)) {
    out.pop().from_scalar(imdim);
  } else if (check_cmd(cmd, "nbpts", in, out, 0, 0, 0, 1)) {
    check_not_exact(im);
    mlab_vect w = out.pop().create_vector(1, 1+pai->structure()->nb_faces());
    w(0,0) = pai->nb_points_on_convex();
    for (size_type i=0; i < pai->structure()->nb_faces(); ++i) 
      w(0,i+1) = pai->nb_points_on_face(i);
  } else if (check_cmd(cmd, "pts", in, out, 0, 0, 0, 1)) {
    check_not_exact(im);
    out.pop().from_vector_container(pai->integration_points());
  } else if (check_cmd(cmd, "face_pts", in, out, 1, 1, 0, 1)) {
    check_not_exact(im); 
    size_type f = in.pop().to_integer(1, pai->structure()->nb_faces())-1;
    mlab_vect w = out.pop().create_vector(imdim, pai->nb_points_on_face(f));
    for (size_type j=0; j < pai->nb_points_on_face(f); ++j)
      for (size_type i=0; i < imdim; ++i)
	w(i,j)=pai->point_on_face(f,j)[i];
  } else if (check_cmd(cmd, "coeffs", in, out, 0, 0, 0, 1)) {
    check_not_exact(im);
    out.pop().from_vector(im->approx_method()->integration_coefficients());
  } else if (check_cmd(cmd, "face_coeffs", in, out, 1, 1, 0, 1)) {
    check_not_exact(im); 
    size_type f = in.pop().to_integer(1, pai->structure()->nb_faces())-1;
    mlab_vect w = out.pop().create_vector(1, pai->nb_points_on_face(f));
    for (size_type j=0; j < pai->nb_points_on_face(f); ++j)
      w(0,j)=pai->coeff_on_face(f,j);
  } else if (check_cmd(cmd, "char", in, out, 0, 0, 0, 1)) {
    std::string s = getfem::name_of_int_method(im);    
    out.pop().from_string(s.c_str());
  } else bad_cmd(cmd);
}

