#define MATLABINT_WORKSPACE_C

#include <bgeot_config.h>
#include <matlabint_workspace.h>

namespace matlabint
{
  workspace_stack *pwstack = NULL;

  workspace_stack& workspace() { 
    if (pwstack == NULL) pwstack = new workspace_stack(); 
    return *pwstack;
  }

  void workspace_stack::set_dependance(getfem_object *user, getfem_object *used)
  {
    used->used_by.push_back(user->get_id());
  }

  /* check if object 'id' can be deleted 
     (all objects which depend on it should be marked as anonymous)
  */
  bool workspace_stack::is_deletable_object(id_type id, dal::bit_vector& v) const {
    const getfem_object *o = obj[id];  

    //mexPrintf("is_deletable: id=%d, o = %p\n", id, o);
    if (!o) THROW_INTERNAL_ERROR;
    if (!o->is_anonymous()) return false; /* this object was never requested for deletion */
    
    v[id] = true;
    
    /* look for every object which uses object 'id' */
    for (std::vector<id_type>::const_iterator it = o->used_by.begin();
	 it != o->used_by.end(); it++) {
      /* if it has not been marqued for delete */
      if (!v[*it]) {
	/* mark it and try to find if it is deletable */
	if (!is_deletable_object(*it, v)) return false;
      }
    }
    return true;
  }

  /* inserts a new object (and gives it an id) */
  id_type workspace_stack::push_object(getfem_object *o) {
    id_type obj_id = obj.add(o);
    o->set_workspace(current_workspace);
    o->set_id(obj_id);
    return obj_id;
  }

  /* at least mark the objet for future deletion (object becomes anonymous)
     and if possible, destroy the object (and all the objects which use this one
     if they are all anonymous) */
  void workspace_stack::delete_object(id_type id) {
    if (obj.index()[id]) {
      getfem_object *o = obj[id]; 
      //      mexPrintf("delete_object: id=%d, o = %p\n", id, o);

      if (!o) THROW_INTERNAL_ERROR;

      /* mark the object as an anonymous one */
      o->set_workspace(anonymous_workspace);
      
      /* list of objects to delete */
      dal::bit_vector dellst; 
      
      /* check if the object is deletable */
      if (is_deletable_object(id, dellst)) {
	//	mexPrintf("object %d is deletable (%d objects will be deleted)\n", id, dellst.card());
	/* suppress the dependancies on an object which is on the deletion list */
	for (obj_ct::tas_iterator it = obj.tas_begin();
	     it != obj.tas_end(); it++) {
	  int cnt = 0;
	  for (unsigned i = 0; i < (*it)->used_by.size(); i++) {
	    if (dellst[(*it)->used_by[i]]) {
	      (*it)->used_by[i] = workspace_stack::invalid_id;
	      cnt ++;
	    }
	  }
	  if (cnt) {
	    int j = 0;
	    for (unsigned i = 0; i < (*it)->used_by.size(); i++) {
	      if ((*it)->used_by[i] != workspace_stack::invalid_id) {
		(*it)->used_by[j] = (*it)->used_by[i]; j++;
	      }
	    }
	    (*it)->used_by.resize(j);

	    /* anonymous objects floating around which are no used by anyone 
	       are added to the deletion list */
	    if (j == 0 && (*it)->is_anonymous()) dellst.add(it.index());
	  }
	}
	bgeot::size_type n;
	dal::bit_vector nn = dellst;
	for (n << nn; n != size_type(-1); n << nn) obj[n]->clear();

	while ((n << dellst) != bgeot::ST_NIL) {
	  getfem_object *o = obj[n];
	  //mexPrintf("suppressing object %d (%s)\n", o->get_id(), o->type_name().c_str());
	  delete o; 
	  //mexPrintf("object successfully deleted -- obj.index()[n]=%d\n", obj.index()[n]);
	  obj.sup(n);
	  //mexPrintf("object successfully removed from index");
	  
	}
      }
    } else {
      std::stringstream s;
      s << "object number " << id << " no longer exists : can't delete it";
      throw matlabint_error(s.str());
    }
  }

  /* create a new workspace on top of the stack */
  void workspace_stack::push_workspace(std::string n) { 
    id_type new_workspace = wrk.add(workspace_data(n, current_workspace));
    current_workspace = new_workspace;
  }

  /* move the object in the parent workspace, in order to prevent
     the object from being deleted when the current workspace will
     be 'poped' */
  void workspace_stack::send_object_to_parent_workspace(id_type obj_id) {
    getfem_object *o = obj[obj_id];
    if (!o) { THROW_ERROR("this object does not exist\n"); }
    if (!wrk.index()[o->get_workspace()]) THROW_INTERNAL_ERROR;
    
    o->set_workspace(wrk[current_workspace].parent_workspace);
  }

  void workspace_stack::send_all_objects_to_parent_workspace() {
    for (obj_ct::tas_iterator it = obj.tas_begin(); 
	 it != obj.tas_end(); ++it) {
      if ((*it)->get_workspace() == current_workspace) {
	(*it)->set_workspace(wrk[current_workspace].parent_workspace);
      }
    }
  }
  /* delete every object in the workspace, but *does not* delete the workspace itself */
  void workspace_stack::clear_workspace(id_type wid) {
    if (wid == anonymous_workspace) THROW_INTERNAL_ERROR;
    //    obj_ct::iterator o_it;
    //    for (o_it = obj.begin(); o_it != obj.end(); o_it++) {
    dal::bit_vector bv = obj.index();
    size_type oid;
    for (oid << bv; oid != size_type(-1); oid << bv) {
      id_type owid = obj[oid]->get_workspace();
      //mexPrintf("id=%d ow = %d [%d]\n", oid, owid, (void*)(*o_it), w);
      if (owid != anonymous_workspace && !wrk.index_valid(owid))
	THROW_INTERNAL_ERROR;
      if (owid == wid) {
	delete_object(oid);
      }
    }
  }

  /* deletes the current workspace and returns to the parent workspace */
  void workspace_stack::pop_workspace(bool keep_all) {
    if (!wrk.index()[current_workspace]) THROW_INTERNAL_ERROR;
    if (current_workspace == base_workspace) THROW_INTERNAL_ERROR;
    
    if (keep_all) send_all_objects_to_parent_workspace();
    else clear_workspace();
    id_type tmp = current_workspace;
    current_workspace = wrk[current_workspace].parent_workspace;
    wrk.sup(tmp);
  }
}
