/* -*- c++ -*- (enables emacs c++ mode)                                    */
/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Interface (matlabint)                                  */
/* File    : matlabint.h : global definitions for Matlab interface.        */
/*     									   */
/*                                                                         */
/* Date : September 26, 2001.                                              */
/* Author : Yves Renard, Yves.Renard@gmm.insa-tlse.fr                      */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */


#ifndef __MATLABINT_H
#define __MATLABINT_H

#include <matlabint_std.h>
#include <set>
#include <getfem_mesh.h>
#include <getfem_mesh_fem.h>
#include <gmm.h>
#include <mex.h>

namespace matlabint
{
  typedef gmm::row_matrix<gmm::wsvector<getfem::scalar_type> >
  gf_sparse_by_row;
  typedef gmm::col_matrix<gmm::wsvector<getfem::scalar_type> >
  gf_sparse_by_col;
  typedef gmm::csc_matrix_ref<const double *, const int *, const int *> 
  gf_sparse_matlab_const_ref;

  typedef bgeot::dim_type dim_type;

  class matlabint_mesh;
  class matlabint_mesh_fem;

  /* fake vector class for matlab arrays (necessary for template functions) */
  class mlab_vect {
    unsigned sz;
    double *data;
    unsigned m,n,p; /* original number of rows & columns of matlab array 
		       p is a pseudo third dimension
		    */
  public:
    typedef double* iterator;
    typedef const double* const_iterator;

    double& operator[](unsigned i) { if (i >= sz) THROW_INTERNAL_ERROR; return data[i]; }
    const double& operator[](unsigned i) const { if (i >= sz) THROW_INTERNAL_ERROR; return data[i]; }
    double& operator()(unsigned i, unsigned j, unsigned k=0) { if (i+j*m+k*m*n >= sz) THROW_INTERNAL_ERROR; 
    return data[i+j*m+k*m*n]; }
    const double& operator()(unsigned i, unsigned j, unsigned k=0) const { if (i+j*m+k*m*n >= sz) THROW_INTERNAL_ERROR;return data[i+j*m+k*m*n]; }

    mlab_vect(): sz(0),data(0),m(0),n(0),p(0) {}
    void assign(const mxArray *mx) { 
      if (!mxIsDouble(mx)) THROW_INTERNAL_ERROR;
      data = mxGetPr(mx); sz = mxGetNumberOfElements(mx); 
      int ndim = mxGetNumberOfDimensions(mx);
      if (ndim < 3) {
	m = mxGetM(mx); n = mxGetN(mx); p = 1;
      } else if (ndim >= 3) {
	const int *dim = mxGetDimensions(mx);
	m = dim[0]; n = dim[1]; p = 1;
	for (int i=2; i < ndim; ++i) p*=dim[i];
      } else THROW_INTERNAL_ERROR;
      if (n*m*p != sz) THROW_INTERNAL_ERROR;
    }
    mlab_vect(const mxArray *mx) { 
      assign(mx);
    }
    mlab_vect(double *v, int _sz) { data = v; sz = _sz; m=_sz; n=1; p = sz ? sz/(m*n) : 0; }
    unsigned size() const { return sz; }
    unsigned getm() const { return m; }
    unsigned getn() const { return n; }
    unsigned getp() const { return p; }

    void reshape(unsigned _n, unsigned _m, unsigned _p=1) {
      if (sz != _n*_m*_p) THROW_INTERNAL_ERROR;
      n = _n; m = _m; p = _p;
    }

    getfem::base_node col_to_bn(unsigned j, unsigned k=0) const {
      getfem::base_node P(m);
      for (unsigned i=0; i < m; i++) P[i] = operator()(i,j,k);
      return P;
    }
    getfem::base_matrix row_col_to_bm(unsigned k=0) const {
      getfem::base_matrix M(m,n);
      for (unsigned i=0; i < m; i++) 
	for (unsigned j=0; j < n; j++) M(i,j) = operator()(i,j,k);
      return M;
    }
    bool in_range(double vmin, double vmax);

    iterator begin() { return data; }
    iterator end() { return data+sz; }
    const_iterator begin() const { return data; }
    const_iterator end() const { return data+sz; }

    /* copies the matlab vector into a new VECT object */
    template<class VECT> VECT to_vector() const {
      VECT v(sz);
      std::copy(begin(), end(), v.begin());
      return v;
    }
  };
}

/* intermde: interface gmm pour les mlab_vect */
namespace gmm {
  template<> struct linalg_traits<matlabint::mlab_vect> {
    typedef matlabint::mlab_vect this_type;
    typedef linalg_false is_reference;
    typedef abstract_vector linalg_type;
    typedef double value_type;
    typedef double& reference;
    typedef this_type::iterator iterator;
    typedef this_type::const_iterator const_iterator;
    typedef abstract_plain storage_type;
    typedef plain_access<iterator,const_iterator> access_type;
    typedef plain_clear<iterator> clear_type;
    static size_type size(const this_type &v) { return v.size(); }
    static iterator begin(this_type &v) { return v.begin(); }
    static const_iterator begin(const this_type &v) { return v.begin(); }
    static iterator end(this_type &v) { return v.end(); }
    static const_iterator end(const this_type &v) { return v.end(); }
    static const void* origin(const this_type &v) { return &v; }
    static void do_clear(this_type &v) { clear_type()(origin(v), begin(v), end(v)); }
  };
} /* fin de l'intermde */


namespace matlabint {

  /* input argument of the gf_* mex-files */
  class mexarg_in {
    void check_int_values(int vmin=INT_MIN, int vmax=INT_MAX);
    double _to_scalar(bool isint);
  public:
    const mxArray *arg;
    int argnum;

    mexarg_in(const mxArray *_arg, int _num) { arg = _arg; argnum = _num; }
    bool                        is_string() { return (mxIsChar(arg) == 1 && mxGetM(arg) == 1); }
    bool                        is_object_id(id_type *pid=0, id_type *pcid=0);
    bool                        is_mesh_fem();
    int                         to_integer(int min_val=INT_MIN, int max_val=INT_MAX);
    double                      to_scalar(double min_val=-1e300, double max_val=1e300);
    std::string                 to_string();
    id_type                     to_object_id(id_type *pid=0, id_type *pcid=0);
    bgeot::base_poly *          to_poly();
    getfem::mesh_fem *          to_mesh_fem();
    const getfem::getfem_mesh * to_const_mesh();
    matlabint_mesh *            to_matlabint_mesh();
    matlabint_mesh_fem *        to_matlabint_mesh_fem();
    getfem::getfem_mesh *       to_mesh();
    getfem::pintegration_method to_fem_interpolation();
    getfem::pfem                to_fem();
    getfem::pmat_elem_type      to_mat_elem_type();
    bgeot::pgeometric_trans     to_pgt();
    bgeot::pconvex_structure    to_convex_structure();

    /* do not perform any check on the number of dimensions of the matlab array */
    mlab_vect                   to_scalar_vector();

    /* expect the argument to be a row or column vector of given dimension */
    mlab_vect                   to_scalar_vector(int expected_dim);

    /* expect the argument to be a matrix (or possibly a 3D array)
       if any of the arguments has a value of -1, the corresponding dimension
       is not checked
     */
    mlab_vect                   to_scalar_vector(int expected_n, int expected_m, int expected_k=1);
    mlab_vect                   to_int_vector();
    mlab_vect                   to_int_vector(int expected_dim);
    mlab_vect                   to_int_vector(int expected_n, int expected_m, int expected_k=1);

    dal::bit_vector             to_bit_vector(const dal::bit_vector *subsetof = NULL, int shiftvals=-1);
    getfem::base_node           to_base_node() { return to_base_node(-1); }
    getfem::base_node           to_base_node(int expected_dim);
    void                        to_sparse(gf_sparse_matlab_const_ref& M);
  };

  /* output argument of the gf_* mex-files */
  class mexarg_out {
  public:
    mxArray *& arg;
    int argnum;
    mexarg_out(mxArray * &_arg, int _num) : arg(_arg), argnum(_num) {}

    void from_object_id(id_type id, id_type class_id);
    void from_object_id(std::vector<id_type> ids, id_type class_id);
    void from_integer(int i);
    void from_scalar(double v);
    void from_string(const char *s);
    template<class STR_CONT> void from_string_container(const STR_CONT& s);
    void from_bit_vector(const dal::bit_vector& bv, int shift=+1);
    void from_sparse(const gf_sparse_by_row& M, double threshold=1e-12);
    void from_sparse(const gf_sparse_by_col& M, double threshold=1e-12);
    void from_tensor(const getfem::base_tensor& t);
    mlab_vect create_vector(unsigned dim);
    mlab_vect create_vector(unsigned n, unsigned m); 
    mlab_vect create_vector(unsigned n, unsigned m, unsigned p); 
    template<class VECT> void from_vector(VECT& v) {
      create_vector(v.size());
      std::copy(v.begin(), v.end(), mxGetPr(arg));
    }
    template<class VEC_CONT> void from_vector_container(const VEC_CONT& vv);
  };

  template<class STR_CONT> void 
  mexarg_out::from_string_container(const STR_CONT& s)
  {
    arg = mxCreateCellMatrix(s.size(), 1);
    typename STR_CONT::const_iterator it;
    size_type cnt = 0;
    for (it = s.begin(); it != s.end(); ++it, ++cnt) {
      mxArray *str = mxCreateString((*it).c_str());
      mxSetCell(arg, cnt, str);
    }
  }

  /* output a matrix from a container of vectors (of same dimensions) */
  template<class VEC_CONT> void 
  mexarg_out::from_vector_container(const VEC_CONT& vv)
  {
    size_type n = vv.size();
    size_type m = (n == 0) ? 0 : vv[0].size();
    mlab_vect w  = create_vector(m,n);
    for (size_type j=0; j < n; ++j)
      for (size_type i=0; i < m; ++i)
	w(i,j) = vv[j][i];
  }

  mxArray *mx_create_object_id(int nid, id_type *ids, id_type cid);
  inline mxArray *mx_create_object_id(id_type id, id_type cid) { 
    return mx_create_object_id(1,&id,cid); 
  }


  /* handles the list of input arguments */
  class mexargs_in {
    const mxArray **in;
    dal::bit_vector idx;
    int nb_arg;
    bool use_cell;
    mexargs_in(const mexargs_in& );
    mexargs_in& operator=(const mexargs_in& );
  public:
    mexargs_in(int n, const mxArray *p[], bool use_cell);
    ~mexargs_in();
    void check() const { if (idx.card() == 0) THROW_INTERNAL_ERROR; }
    const mxArray *pop_mx(size_type decal=0, int *out_idx = NULL) {
      size_type i = idx.first_true(); 
      check();
      if (decal >= idx.card()) THROW_INTERNAL_ERROR;
      while (decal>0) { i++; check(); if (idx.is_in(i)) decal--; }
      idx.sup(i); 
      if (out_idx) *out_idx = i;
      return in[i];
    }
    mexarg_in pop(size_type decal=0) { 
      int i;
      const mxArray *m = pop_mx(decal, &i);
     return mexarg_in(m,i+1); 
    }
    void restore(size_type i) { idx.add(i); }
    mexarg_in front() const { check(); return mexarg_in(in[idx.first_true()],idx.first_true()); }
    int narg() const { return nb_arg; }
    int remaining() const { return idx.card(); }
    void get_array(const mxArray **& m) {
      if (remaining()) {
	m = new const mxArray *[remaining()];
	for (size_type i=0; remaining(); i++) {
	  m[i] = pop_mx();
	}
      } else m = NULL;
    }
    void pop_all() { idx.clear(); }
  };

  /* handles the list of output arguments */
  class mexargs_out {
    mxArray **out;
    int nb_arg;
    int idx;
    bool use_cell;
    mxArray **pout_cell;

    mexargs_out(const mexargs_out& );
    mexargs_out& operator=(const mexargs_out& );
  public:
    mexargs_out(int _nb_arg, mxArray *p[], bool _use_cell);
    ~mexargs_out();
    void check() const { if ((idx >= nb_arg || idx < 0) && !(idx==0)) THROW_INTERNAL_ERROR; }
    mexarg_out pop() { check(); idx++; return mexarg_out(out[idx-1], idx); }
    mexarg_out front() const { check(); return mexarg_out(out[idx], idx+1); }
    int narg() const { return nb_arg; }
    int remaining() const { return std::max(nb_arg,1) - idx; }
    bool fixed_size() const { return !use_cell; }
    mxArray **get_array() const { return out+idx; }
    void pop_all() { idx = nb_arg; }
    void return_packed_obj_ids(const std::vector<id_type>& ids, id_type class_id);
  };


  bool cmd_strmatch(const std::string& a, const char *s);
  bool check_cmd(const std::string& cmdname, const char *s, 
		 const mexargs_in& in, const mexargs_out& out,
		 int min_argin=0, int max_argin=-1,
		 int min_argout=0, int max_argout=-1);
  
  inline void bad_cmd(std::string& cmd) { 
    THROW_BADARG("Bad command name: " << cmd); }

  typedef void (*mex_func_type)(mexargs_in&, mexargs_out&);

  void catch_errors(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[],
		    mex_func_type pf, const char *local_workspace=NULL);
  void call_matlab_function(const std::string& fname, mexargs_in& in, mexargs_out& out);
  void check_cv_fem(const getfem::mesh_fem& mf, size_type cv);

}  /* end of namespace matlabint.                                          */

#endif /* __MATLABINT_H                                                    */
