#include <map>
#include <matlabint.h>
#include <matlabint_misc.h>
#include <matlabint_workspace.h>
#include <matlabint_poly.h>
#include <matlabint_mesh.h>
#include <matlabint_mesh_fem.h>
#include <matlabint_matelemtype.h>
#include <matlabint_matelem.h>
#include <matlabint_pfem.h>
#include <matlabint_pfi.h>
#include <matlabint_pgt.h>
#include <matlabint_convex_structure.h>

#include <matlabint_misc.h>
#ifdef MAINTAINER_MODE
# ifdef HAVE_CSIGNAL
#  include <csignal>
# else 
#  include <signal.h>
# endif
#endif
#include <exception> // NE PAS METTRE CE FICHIER EN PREMIER !!!! ou bien ennuis avec dec cxx 6.3 garantis

namespace matlabint {

  /* associate the class ID found in the matlab structures referencing
     getfem object to a class name which coincides with the class name
     given by matlab to the structure (hum..) */
  const char *name_of_getfemint_class_id(unsigned cid) {
    static const char *cname[GETFEMINT_NB_CLASS] = {
      "gfMesh", "gfMeshFem", "gfGeoTrans", 
      "gfFem", "gfInteg","gfEltm","gfCvStruct","gfPoly"
    };

    if (cid >= GETFEMINT_NB_CLASS) return "not_a_getfem_class";
    else return cname[cid];
  }


  static const char*
  mxClassID2string(mxClassID id) {
    switch (id) {
    case mxUNKNOWN_CLASS: return "UNKNOWN";
    case mxCELL_CLASS:    return "CELL";
    case mxSTRUCT_CLASS:  return "STRUCT";
    case mxOBJECT_CLASS:  return "OBJECT";
    case mxCHAR_CLASS:    return "CHAR";
      //    case mxLOGICAL_CLASS: return "LOGICAL";
    case mxDOUBLE_CLASS:  return "DOUBLE";
    case mxSINGLE_CLASS:  return "SINGLE";
    case mxINT8_CLASS:    return "INT8";
    case mxUINT8_CLASS:   return "UINT8";
    case mxINT16_CLASS:   return "INT16";
    case mxUINT16_CLASS:  return "UINT16";
    case mxINT32_CLASS:   return "INT32";
    case mxUINT32_CLASS:  return "UINT32";
    case mxINT64_CLASS:   return "INT64";
    case mxUINT64_CLASS:  return "UINT64";
    case mxSPARSE_CLASS:  return "SPARSE";
      // case mxFUNCTION_CLASS: return "FUNCTION_CLASS"; // not supported by matlab 5.3
    case mxOPAQUE_CLASS: return "OPAQUE_CLASS";
    default: return "unknown class...";
    }
  }

  /*  std::string
  mxGetClassName(const mxArray *m) {
    return mxClassID2string(mxGetClassID(m));
    }*/
  double
  mexarg_in::_to_scalar(bool isint)
  {
    double dv;
    if (mxGetNumberOfElements(arg) != 1) {
      THROW_BADARG("Argument " << argnum << 
		   "has dimensions " << mxGetN(arg) << "x" << mxGetM(arg) <<
		   " but a [1x1] " << std::string(isint ? "integer" : "scalar") << 
		   " was expected");
    }
    switch (mxGetClassID(arg)) {
    case mxDOUBLE_CLASS: {
      dv = *mxGetPr(arg);
    } break;
    case mxSINGLE_CLASS: {
      dv = (double)( *((float*)mxGetData(arg)));
    } break;
    case mxINT32_CLASS: {
      dv = (double)( *((dal::int32_type*)mxGetData(arg)));
    } break;
    case mxUINT32_CLASS: {
      dv = (double)( *((dal::uint32_type*)mxGetData(arg)));
    } break;
    case mxINT16_CLASS: {
      dv = (double)( *((dal::int16_type*)mxGetData(arg)));
    } break;
    case mxUINT16_CLASS: {
      dv = (double)( *((dal::uint16_type*)mxGetData(arg)));
    } break;
    default: {
      THROW_BADARG("Argument " << argnum << 
		" is not an scalar value");
    } break;
    }
    return dv;
  }

  double
  mexarg_in::to_scalar(double minval, double maxval)
  {
    double dv = _to_scalar(false);
    if (dv < minval || dv > maxval) {
      THROW_BADARG("Argument " << argnum << 
		" is out of bounds : " << dv << " not in [" << 
		minval << "..." << maxval << "]");
    }
    return dv;
  }

  int
  mexarg_in::to_integer(int minval, int maxval)
  {
    double dv = _to_scalar(true);
    if (dv != floor(dv)) {
      THROW_BADARG("Argument " << argnum << 
		" is not an integer value");
    }
    if (dv < minval || dv > maxval) {
      THROW_BADARG("Argument " << argnum << 
		" is out of bounds : " << dv << " not in [" << 
		minval << "..." << maxval << "]");
    }
    return (int)dv;
  }



  bool
  mexarg_in::is_object_id(id_type *pid, id_type *pcid) 
  {
    if (mxGetClassID(arg) == mxOBJECT_CLASS ||
	mxGetClassID(arg) == mxSTRUCT_CLASS) {
      mxArray *fid = mxGetField(arg, 0, "id");
      mxArray *fcid = mxGetField(arg, 0, "cid");
      if (fid && mxGetClassID(fid) == mxUINT32_CLASS &&
	  fcid && mxGetClassID(fcid) == mxUINT32_CLASS)
	if (pid)  *pid  = *(static_cast<dal::uint32_type*>(mxGetData(fid)));
	if (pcid) *pcid = *(static_cast<dal::uint32_type*>(mxGetData(fcid)));
	return true;
    }
    return false;
  }

  bool
  mexarg_in::is_mesh_fem() 
  {
    id_type id, cid;
    if (is_object_id(&id, &cid) && cid == MESHFEM_CLASS_ID) {
      getfem_object *o = workspace().object(id, "MESH FEM");
      return (object_is_mesh_fem(o));
    } else return false;
  }

  std::string 
  mexarg_in::to_string()
  {
    /* string => row vector. */
    if (!is_string()) 
      THROW_BADARG("Argument " << argnum << " must be a string.");
    int bsz = mxGetN(arg) + 1;
    char *buf = (char*) mxCalloc(bsz, sizeof(char));
    if(mxGetString(arg, buf, bsz)) { 
      mexWarnMsgTxt("Not enough space. String is truncated.");
    }
    std::string s(buf);
    mxFree(buf);
    return s;
  }

  id_type
  mexarg_in::to_object_id(id_type *pid, id_type *pcid) 
  {
    id_type id,cid;
    if (!is_object_id(&id, &cid)) {
      THROW_BADARG("wrong type for argument " << argnum <<
		   ": expecting a getfem object, got a " << mxGetClassName(arg));
    }
    if (pid) *pid = id;
    if (pcid) *pcid = cid;
    return id;
  }


  /*
    check if the argument is a valid handle to an intset
    and returns it
  */
  /*
  dal::bit_vector * mexarg_in::to_intset()
  {
    getfem_object *o = workspace().object(to_object_id());
    if (!object_is_intset(o)) {
      THROW_BADARG("Argument " << argnum << " should be an intset descriptor");
    }
    return &object_to_intset(o)->intset();
  }
  */


  /*
    check if the argument is a valid handle to a base_poly
    and returns it
  */
  bgeot::base_poly *
  mexarg_in::to_poly()
  {
    id_type id, cid;
    to_object_id(&id,&cid);
    if (cid != POLY_CLASS_ID) {
      THROW_BADARG("argument " << argnum << " should be a polynom descriptor, its class is " << name_of_getfemint_class_id(cid));
    }
    getfem_object *o = workspace().object(id,name_of_getfemint_class_id(cid));
    return &object_to_poly(o)->poly();
  }


  /*
    check if the argument is a valid handle to a mesh_fem,
    and returns it
  */
  matlabint_mesh_fem *
  mexarg_in::to_matlabint_mesh_fem()
  {
    id_type id, cid;
    to_object_id(&id,&cid);
    if (cid != MESHFEM_CLASS_ID) {
      THROW_BADARG("argument " << argnum << " should be a mesh_fem descriptor, its class is " << name_of_getfemint_class_id(cid));
    }
    getfem_object *o = workspace().object(id,name_of_getfemint_class_id(cid));
    return object_to_mesh_fem(o);
  }
  getfem::mesh_fem *
  mexarg_in::to_mesh_fem() {
    return &to_matlabint_mesh_fem()->mesh_fem();
  }

  /*
    check if the argument is a valid mesh handle
    if a mesh_fem handle is given, its associated
    mesh is used
  */
  const getfem::getfem_mesh *
  mexarg_in::to_const_mesh()
  {
    id_type id, cid;
    to_object_id(&id,&cid);
    if (cid != MESHFEM_CLASS_ID && cid != MESH_CLASS_ID) {
      THROW_BADARG("argument " << argnum << " should be a mesh or mesh_fem descriptor, its class is " << name_of_getfemint_class_id(cid));
    }
    const getfem::getfem_mesh *mesh = NULL;
    getfem_object *o = workspace().object(id,name_of_getfemint_class_id(cid));
    if (object_is_mesh(o)) {
      mesh = &object_to_mesh(o)->mesh();
    } else if (object_is_mesh_fem(o)) {
      mesh = &object_to_mesh_fem(o)->mesh_fem().linked_mesh();
    } else THROW_INTERNAL_ERROR;
    return mesh;
  }


  /*
    check if the argument is a valid mesh handle
    (the returned arg is not const, so we can't use the mesh from mesh_fem objects)
  */
  matlabint_mesh *
  mexarg_in::to_matlabint_mesh()
  {
    id_type id, cid;
    to_object_id(&id,&cid);
    if (cid != MESH_CLASS_ID) {
      THROW_BADARG("argument " << argnum << " should be a mesh descriptor, its class is " << name_of_getfemint_class_id(cid));
    }
    getfem_object *o = workspace().object(id,name_of_getfemint_class_id(cid));
    return object_to_mesh(o);
  }

  getfem::getfem_mesh *
  mexarg_in::to_mesh() {
    return &to_matlabint_mesh()->mesh();
  }

  getfem::pintegration_method
  mexarg_in::to_fem_interpolation()
  {
    id_type id,cid;
    to_object_id(&id,&cid);
    if (cid != INTEG_CLASS_ID)
      THROW_BADARG("Argument " << argnum << 
		   " should be an integration method descriptor");
    if (!exists_pfi(id)) {
      THROW_BADARG("Argument " << argnum << 
		   " is not a valid integration method handle");
    }
    return addr_pfi(id);
  }

  getfem::pmat_elem_type
  mexarg_in::to_mat_elem_type()
  {
    id_type id,cid;
    to_object_id(&id,&cid);
    if (cid != ELTM_CLASS_ID)
      THROW_BADARG("Argument " << argnum << 
		   " should be a elementary matrix descriptor.");
    if (!exists_matelemtype(id))
      THROW_BADARG("Argument " << argnum << 
		   " is not a valid elementary matrix handle");
    return addr_matelemtype(id);
  }

  getfem::pfem
  mexarg_in::to_fem()
  {
    id_type id,cid;
    to_object_id(&id,&cid);
    if (cid != FEM_CLASS_ID)
      THROW_BADARG("Argument " << argnum << 
		   " should be a fem descriptor");
    
    if (!exists_pfem(id))
      THROW_BADARG("Argument " << argnum << 
		   " is not a valid fem handle");
    return addr_pfem(id);
  }


  bgeot::pgeometric_trans
  mexarg_in::to_pgt() {
    id_type id,cid;
    to_object_id(&id,&cid);
    if (cid != GEOTRANS_CLASS_ID)
      THROW_BADARG("Argument " << argnum << 
		   " is not a geometric transformation handle");
    if (!matlabint::exists_pgt(id))
      THROW_BADARG("Argument " << argnum << 
		   " refers to a geometric transformation that does not exists");
    return matlabint::addr_pgt(id);
  }

  bgeot::pconvex_structure
  mexarg_in::to_convex_structure() {
    id_type id,cid;
    to_object_id(&id,&cid);
    if (cid != CVSTRUCT_CLASS_ID)
      THROW_BADARG("Argument " << argnum << 
		   " is not a convex structure handle");
    if (!matlabint::exists_convex_structure(id))
      THROW_BADARG("Argument " << argnum << 
		   " refers to a convex structure that does not exists");
    return matlabint::addr_convex_structure(id);
  }

  mlab_vect
  mexarg_in::to_scalar_vector() {
    if(!(mxIsDouble(arg))) {
      THROW_BADARG("Argument " << argnum << 
		   " should be a DOUBLE REAL data array");
    }
    return mlab_vect(arg);
  }

  /* check that the supplied argument IS a vector,
     with the right number of elements */
  mlab_vect
  mexarg_in::to_scalar_vector(int expected_dim) {
    mlab_vect v = to_scalar_vector();
    if (v.getn() != 1 && v.getm() != 1 && v.size() != 0) {
      THROW_BADARG("Argument " << argnum << 
		   " should be a vector, not a matrix");
    }
    if (expected_dim != -1 && (int)v.size() != expected_dim) {
      THROW_BADARG("Argument " << argnum << 
		   " has wrong dimensions: expected " << expected_dim << 
		   ", found " << v.size());
    }
    return v;
  }

  /* check that the supplied array has a good number of rows
     and/or a good number of columns (and or a good number of third dimension) */
  mlab_vect
  mexarg_in::to_scalar_vector(int expected_m, int expected_n, int expected_p) {
    mlab_vect v = to_scalar_vector();
    if (expected_m != -1 && (int)v.getm() != expected_m)
      THROW_BADARG("Argument " << argnum << 
		   " has a wrong number of rows (" << v.getm() <<
		   ") , " << expected_m << " rows were expected");
    if (expected_n != -1 && (int)v.getn() != expected_n)
      THROW_BADARG("Argument " << argnum << 
		   " has a wrong number of columns (" << v.getn() <<
		   ") , " << expected_n << " columns were expected");

    if (expected_p != -1 && (int)v.getp() != expected_p)
      THROW_BADARG("Argument " << argnum << 
		   " was expected to be a three-dimensional array, with " << 
		   expected_p << " elements in its third dimension (got " << 
		   v.getp() << ")");

    return v;
    /*
    if (m == 1 || n == 1) {
      inout_dim = 1; len = (m == 1) ? n : m;
      if (len > nbdof && nbdof && (len % nbdof == 0)) {
	inout_dim = len / nbdof;
	len = nbdof;
      }
    } else {
      inout_dim = m;
      len = n;
    }

    if (len != nbdof) {
      THROW_BADARG("Argument " << argnum << 
      " has a bad size (" << len << 
      ") , it should be " << nbdof);
    }

    if (dim != inout_dim && dim != 0) {
    THROW_BADARG("Argument " << argnum << 
		"is supposed to have " << dim << " rows");
    }
    return mlab_vect(arg);*/
  }

  void
  mexarg_in::check_int_values(int vmin, int vmax) {
    mlab_vect v = to_scalar_vector();
    for (unsigned i=0; i < v.size(); i++) {
      if (floor(v[i]) != v[i]) {
	THROW_BADARG("Argument " << argnum << 
		     " should be a DOUBLE REAL data array containing only "
		     "INTEGER values --- at index " << i+1 << 
		     " the scalar value " << v[i] << " was found");
      }
      if (v[i] < vmin || v[i] > vmax) {
	THROW_BADARG("Argument " << argnum << 
		     " has an out-of-bound value at index " << i+1 << 
		     " : " << v[i] << " is not in the range [" << vmin << 
		     ":" << vmax << "]");
      }
    }
  }


  mlab_vect
  mexarg_in::to_int_vector() {
    check_int_values();
    return to_scalar_vector();
  }

  mlab_vect
  mexarg_in::to_int_vector(int expected_dim) {
    check_int_values();
    return to_scalar_vector(expected_dim);    
  }

  mlab_vect
  mexarg_in::to_int_vector(int expected_m, int expected_n, int expected_p) {
    check_int_values();
    return to_scalar_vector(expected_m, expected_n, expected_p);    
  }

  getfem::base_node
  mexarg_in::to_base_node(int expected_dim) {
    mlab_vect w = to_scalar_vector(expected_dim,1);
    getfem::base_node bn(w.size()); std::copy(w.begin(), w.end(), bn.begin());
    return bn;
  }

//   void
//   mexarg_in::to_sparse(gf_sparse_by_row& M) {
//     if (!mxIsSparse(arg)) {
//       THROW_BADARG("Argument " << argnum << " was expected to be a sparse matrix");
//     }
//     smatrix_from_matlab(arg, M);
//   }
//   void
//   mexarg_in::to_sparse(gf_sparse_by_col& M) {
//     if (!mxIsSparse(arg)) {
//       THROW_BADARG("Argument " << argnum << " was expected to be a sparse matrix");
//     }
//     smatrix_from_matlab(arg, M);
//   }

  void
  mexarg_in::to_sparse(gf_sparse_matlab_const_ref& M) {
    if (!mxIsSparse(arg)) {
      THROW_BADARG("Argument " << argnum << " was expected to be a sparse matrix");
    }
    M = gf_sparse_matlab_const_ref(mxGetPr(arg), mxGetIr(arg), mxGetJc(arg),
				   mxGetM(arg), mxGetN(arg));
  }
  
  bool
  mlab_vect::in_range(double vmin, double vmax) {
    for (size_type i = 0; i < sz; i++) {
      if (data[i] < vmin || data[i] > vmax) {
	return false;
      }
    }
    return true;
  }
  

  /* converts the mxArray into a bit vector , shift all its values by
     'shift' and checking that they are a subset of 'subsetof' ( if the pointer
     is non-nul ) */
  dal::bit_vector
  mexarg_in::to_bit_vector(const dal::bit_vector *subsetof, int shift) {
    dal::bit_vector bv;
    mlab_vect v = to_int_vector(-1);
    for (size_type i = 0; i < v.size(); i++) {
      int idx = (int)v[i] + shift;
      if (idx < 0 || idx > 1000000000) {
	THROW_BADARG("Argument " << argnum << 
		     " should only contain values greater or equal to "
		     << -shift << " ([found " << v[i] << ")");
      } else if (subsetof && !subsetof->is_in(idx)) {
	THROW_BADARG("Argument " << argnum << 
		     " is not a valid set");
      }
      bv.add(idx);
    }
    return bv;
  }

  mxArray *
  mx_create_object_id(int nid, id_type *ids, id_type cid) {
    mxArray *arg;
    int j=1;
    static const char *fields[] = {"id","cid"};
    arg = mxCreateStructArray(1, &nid, 2, fields);
    for (size_type i=0; i < nid; ++i) {
      mxArray *mxid = mxCreateNumericArray(1, &j, mxUINT32_CLASS, mxREAL);
      *(static_cast<dal::uint32_type*>(mxGetData(mxid))) = ids[i];
      mxSetField(arg,i,fields[0], mxid);
      mxArray *mxcid = mxCreateNumericArray(1, &j, mxUINT32_CLASS, mxREAL);
      *(static_cast<dal::uint32_type*>(mxGetData(mxcid))) = cid;
      mxSetField(arg,i,fields[1], mxcid);
    }
    return arg;
  }

  void
  mexarg_out::from_object_id(id_type id, id_type cid)
  { arg = mx_create_object_id(id, cid); }

  void
  mexarg_out::from_object_id(std::vector<id_type> ids, id_type cid)
  { arg = mx_create_object_id(ids.size(), &ids[0], cid); }

  void
  mexarg_out::from_integer(int i)
  {
    from_scalar((double)i);
  }

  void
  mexarg_out::from_scalar(double v)
  {
    arg = mxCreateDoubleMatrix(1,1,mxREAL);
    *mxGetPr(arg) = v;
  }

  void 
  mexarg_out::from_string(const char *s) {
    arg = mxCreateString(s);
  }


  void
  mexarg_out::from_bit_vector(const dal::bit_vector& bv, int shift)
  {
    mlab_vect w = create_vector(1, bv.card());
    size_type j = 0;
    for (dal::bit_vector::const_iterator it = bv.begin(); it != bv.end(); ++it) {
      if (*it) {
	w(0,j++) = double(it.index() + shift);
      }
    }
    if (j != bv.card()) THROW_INTERNAL_ERROR;
  }


  void
  mexarg_out::from_sparse(const gf_sparse_by_row& M, double threshold) {
    arg = convert_to_mxSparse(M);
  }

  void
  mexarg_out::from_sparse(const gf_sparse_by_col& M, double threshold) {
    arg = convert_to_mxSparse(M);
  }

  void
  mexarg_out::from_tensor(const getfem::base_tensor& t) {
    std::vector<int> tab(t.order());
    std::copy(t.sizes().begin(), t.sizes().end(), tab.begin());
    arg = mxCreateNumericArray(t.order(), &(tab.begin()[0]),mxDOUBLE_CLASS, mxREAL);
    double *q = (double *)(mxGetData(arg));
    std::copy(t.begin(), t.end(), q);        
  }

  mlab_vect
  mexarg_out::create_vector(unsigned dim)
  {
    arg = mxCreateDoubleMatrix(1, dim, mxREAL);
    //    mexPrintf("vector of size %d created [%p]\n", dim, arg);
    return mlab_vect(arg);
  }

  mlab_vect
  mexarg_out::create_vector(unsigned n,unsigned m,unsigned p)
  {
    int sz[3];
    sz[0] = n; sz[1] = m; sz[2] = p;
    arg = mxCreateNumericArray(3, sz,mxDOUBLE_CLASS, mxREAL);
    return mlab_vect(arg);
  }
  /* creates a 'matrix' (from the matlab point of view.. for getfem 
     it is still a vector, stored in fortran style) */
  mlab_vect
  mexarg_out::create_vector(unsigned dim, unsigned nbdof)
  {
    arg = mxCreateDoubleMatrix(dim, nbdof, mxREAL);
    return mlab_vect(arg);
  }

  /* very tolerant case-insensitive string comparison: 
     spaces are matched with underscores.*/
  bool cmd_strmatch(const std::string& a, const char *s) {
    if (a.length() != strlen(s)) return false;
    for (size_type i=0; s[i]; ++i) {
      if ((a[i] == ' ' || a[i] == '_') && (s[i] == ' ' || s[i] == '_')) continue;
      if (toupper(a[i]) != toupper(s[i])) return false;
    }
    return true;
  }

  bool check_cmd(const std::string& cmdname, const char *s, 
		 const mexargs_in& in, const mexargs_out& out,
		 int min_argin, int max_argin,
		 int min_argout, int max_argout) {
    if (cmd_strmatch(cmdname,s)) {
      if ((int)in.remaining() < min_argin) {
	THROW_BADARG("Not enough input arguments for command '"<< 
		     cmdname << "' (got " << in.narg() << 
		     ", expected at least " << min_argin + in.narg()- in.remaining() << ")");
      }
      if ((int)in.remaining() > max_argin && max_argin != -1) {
	THROW_BADARG("Too much input arguments for command '"<< 
		     cmdname << "' (got " << in.narg() << 
		     ", expected at most " << max_argin + in.narg()- in.remaining()<< ")");
      }
      if (out.narg() < min_argout && out.fixed_size()) {
	THROW_BADARG("Not enough output arguments for command '"<< 
		     cmdname << "' (got " << out.remaining() << 
		     ", expected at least " << min_argout << ")");
      }
      if (out.narg() > max_argout && max_argout != -1 && out.fixed_size()) {
	THROW_BADARG("Too much output arguments for command '"<< 
		     cmdname << "' (got " << out.narg() << 
		     ", expected at most " << max_argout << ")");
      }
      return true;
    }
    return false;
  }


  mexargs_in::mexargs_in(int n, const mxArray *p[], bool _use_cell) {
    nb_arg = n; use_cell = _use_cell;
    if (!use_cell) {
      in = p; if (n) { idx.add(0, n-1); }
    } else {
      assert(n == 1);
      nb_arg = mxGetNumberOfElements(p[0]);
      in = new const mxArray*[nb_arg];
      for (int i = 0; i < nb_arg; i++) { p[i] = mxGetCell(p[0], i); idx.add(i); }
    }
  }
  
  mexargs_in::~mexargs_in() {
    if (in && use_cell) delete[] in;
  }

  mexargs_out::mexargs_out(int n, mxArray *p[], bool _use_cell)
  {
    idx = 0; use_cell = _use_cell;
    if (!use_cell) {
      out = p; nb_arg = n; 
    } else {
      out = new mxArray*[n];
      use_cell = true;
    }
  }

  mexargs_out::~mexargs_out() {
    if (use_cell) {
      *pout_cell = mxCreateCellMatrix(1, idx);
      for (int j=0; j < idx; j++) {
	assert(out[j]!=0);
	mxSetCell(*pout_cell, j, out[j]);
      }
      delete[] out;
    }
  }

  void 
  mexargs_out::return_packed_obj_ids(const std::vector<id_type>& id, id_type class_id) {
    std::vector<id_type> uid(id);
    std::sort(uid.begin(), uid.end());
    /* remove duplicates */
    uid.erase(std::unique(uid.begin(), uid.end()), uid.end());
    /* remove id_type(-1) */
    std::vector<id_type>::iterator it =
      std::find(uid.begin(), uid.end(), id_type(-1));
    if (it != uid.end())
      uid.erase(it);
    pop().from_object_id(uid, class_id);
    if (remaining()) {
      std::map<id_type,id_type> m;
      for (size_type i=0; i < uid.size(); ++i) m[uid[i]]=i+1;
      mlab_vect v = pop().create_vector(1,id.size());
      for (size_type i=0; i < id.size(); ++i) 
	v[i] = (id[i] != id_type(-1)) ? double(m[id[i]]) : id_type(-1);
    }
  }

  class matlabint_exc_cb : public dal::exception_callback {
    int lvl;
    virtual void callback(const std::string& msg) {
      static int cnt = 0;
      cerr << "matlabint callback: exception raised:" << msg << endl;
      if (msg.find("Internal error") != std::string::npos || lvl) {
	cerr << "lauching debugger..\n";
	cnt++;
	if (cnt < 10) {
	  attach_gdb();
	} else {
	  cerr << "debugger not launched in case of a fork bomb!\n";
	}
      }
    }
  public:
    matlabint_exc_cb() { lvl = 0; }
    void set_level(int i) { lvl = i; }
  };
#ifdef MAINTAINER_MODE
  extern "C" void  catch_segv(int) {
    static int segcnt = 0;
    cerr << "getfem/matlab caught a SEGFAULT!\n";
    segcnt++;
    if (segcnt < 2) {
      cerr << "lauching debugger..\n";
      attach_gdb();
      exit(1);
    } else {
      cerr << "ignoring..(you don't want this to turn into a fork-bomb, don't you?)\n";
    }
  }

  static mxArray* find_matlab_variable(const char *name) {
    mxArray *m;
    m = mexGetVariable("global", name);
    if (!m) m = mexGetVariable("caller", name);
    if (!m) m = mexGetVariable("base", name);
    return m;
  }
#endif


  void catch_errors(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[],
		    mex_func_type pf, const char *local_workspace)
  {
    bool use_local_workspace = (local_workspace != NULL);
#  ifdef MAINTAINER_MODE
    static matlabint_exc_cb * pexc_cb = 0;
    /*
      a marche mais c'est pas pratique...
    */
      void (*old)(int);
      old = ::signal(SIGSEGV, catch_segv);
     
    
      if (pexc_cb == 0) {
	cerr << "maintainer: set_exception_callback\n";
	pexc_cb = new matlabint_exc_cb();
	dal::set_exception_callback(pexc_cb);
      }
    
    mxArray *mx = find_matlab_variable("GETFEM_DEBUG_LVL");
    if (mx) { 
      cerr << "set level to " << int(mxGetScalar(mx)) << endl;
      pexc_cb->set_level(int(mxGetScalar(mx))); 
    }

#  endif
    try {
      if (use_local_workspace) workspace().push_workspace(local_workspace);

      /* detect if arguments were packed in a cell array */
      bool use_cell = (nrhs == 1 && mxIsCell(prhs[0]) && nrhs == 1);

      mexargs_in argin(nrhs, prhs, use_cell);
      mexargs_out argout(use_cell ? 50 : nlhs, plhs, use_cell);

      /* call the function that will do the real work */
      pf(argin,argout);
      //      ::signal(SIGSEGV, (void (*)(int) )old);    
    } 

    //    sigaction(SIGSEGV, &sa_old, NULL);
    catch (matlabint_bad_arg e) {
      cerr << "matlabint_bad_arg exception caught\n";
      if (use_local_workspace) workspace().pop_workspace();
      mexErrMsgTxt(e.what());      
    }
    catch (matlabint_error e) {
      cerr << "matlabint_error exception caught\n";
      if (use_local_workspace) workspace().pop_workspace();
      mexErrMsgTxt(e.what());      
    }
    catch (std::logic_error e) {
      cerr << "logic_error exception caught\n";
      if (use_local_workspace) workspace().pop_workspace();
      mexErrMsgTxt(e.what());
    }
    catch (std::runtime_error e) {
      cerr << "runtime_error exception caught\n";
      if (use_local_workspace) workspace().pop_workspace();
      mexErrMsgTxt(e.what());
    }
    catch(std::bad_alloc) {
      if (use_local_workspace) workspace().pop_workspace();
      mexErrMsgTxt("getfem caught a bad_alloc exception\n");
    }
    catch(std::bad_cast) {
      if (use_local_workspace) workspace().pop_workspace();
      mexErrMsgTxt("getfem caught a bad_cast exception\n");
    }
    catch(std::bad_typeid) {
      if (use_local_workspace) workspace().pop_workspace();
      mexErrMsgTxt("getfem caught a bad_typeid exception\n");
    }
    catch(std::bad_exception) {
      if (use_local_workspace) workspace().pop_workspace();
      mexErrMsgTxt("getfem caught a bad_exception exception\n");
    }
    /*
    catch(ios_base::failure) {
      mexErrMsgTxt("getfem caught a ios_base::failure exception\n");
    }
    */
    catch (...) {
      if (use_local_workspace) workspace().pop_workspace();
      mexErrMsgTxt("getfem caught an unknown exception\n");
    }
    if (use_local_workspace) workspace().pop_workspace(true);
#  ifdef MAINTAINER_MODE
    ::signal(SIGSEGV, old);
#  endif    
  }

  void
  call_matlab_function(const std::string& fname, mexargs_in& in, mexargs_out& out) {
    mexSetTrapFlag(1);

    const mxArray **plhs;
    int nlhs = in.remaining();
    in.get_array(plhs);
    /* merci a la declaration de mexcallmatlab pour le const_cast.. */
    int status = mexCallMATLAB(out.remaining(),out.get_array(),
			       nlhs, const_cast<mxArray **>(plhs), fname.c_str());
    delete[] plhs;

    in.pop_all(); out.pop_all();
    /* If input argument to mexsettrapflag is non-zero, control
       remains in MEX-file. */
    if (status != 0) {
      /* Display contents of MATLAB function lasterr to see error
	 issued in MATLAB */
      mxArray *error_msg[1];
  
      mexCallMATLAB(1,error_msg,0, (mxArray **)NULL, "lasterr"); 
      char *buf = mxArrayToString(error_msg[0]);
      mexPrintf("The last error message issued in MATLAB was: \n%s\n", buf);
      mxFree(buf);    
      mexErrMsgTxt("mexCallMATLAB failed.\n");
    }
  }

  void check_cv_fem(const getfem::mesh_fem& mf, size_type cv) {
    if (!mf.convex_index()[cv]) THROW_ERROR( "convex " << cv+1 << " has no FEM");
  }

} /* namespace matlabint */
