function [hsurf, hcontour, hquiver, hmesh, hdefmesh]=gf_plot(mf,U,varargin)
% function h=gf_plot(mf,U,varargin)
% this function is used to plot a general pde solution
%
% The options are specified as pairs of "option name"/"option value"
%  
%  'zplot',{'off'|'on'}       : only for qdim=1, mdim=2
%  'norm', {'off'|'on'}       : if qdim >= 2, color-plot the norm of the field
%  'dir',[]	              : or the scalar product of the field with 'dir' 
%                               (can be a vector, or 'x', 'y' etc..)
%  'refine',8		      : nb of refinments for curved edges and surface plots
%  'interpolated',{'off'|'on'}: if triangular patch are interpolated
%  'pcolor',{'on'|'off'}      : if the field is scalar, a color plot of its values is plotted
%  'quiver',{'on'|'off'}      : if the field is vector, represent arrows 	       
%  'quiver_density',50        : density of arrows in quiver plot
%  'quiver_scale',1           : scaling of arrows (0=>no scaling)
%  'mesh',{'off'|'on'}	      : show the mesh ?
%  'meshopts',{cell(0)}	         : cell array of options passed to gf_plot_mesh for the mesh 
%  'deformed_mesh', {'off'|'on'} : shows the deformed mesh (only when qdim == mdim)
%  'deformed_meshopts', {cell(0)}: cell array of options passed to gf_plot_mesh 
%                                  for the deformed mesh 
%  'deformation',[]	      : plots on the deformed object (only when qdim == mdim)
%  'deformation_mf',[]        : plots on the deformed object (only when qdim == mdim)
%  'deformation_scale','10%'  : indicate the amplitude of the deformation. Can be 
%                               a percentage of the mesh width if given as a string, 
%                               or an absolute value if given as a number
%  'cvlst',[]		      : list of convexes to plot (empty=>all convexes)
%  'title',[]                 : set the title
%  'contour',[]               : list of contour values
%%%%%%%%%%%%
  if nargin<2,
    error('Too few input arguments')
  end

  mf=struct(mf);
  hsurf=[];
  hcontour={};
  hmesh=[];
  hquiver=[];
  
  qdim = gf_mesh_fem_get(mf, 'qdim');
  mdim = gf_mesh_get(mf, 'dim'); mdim3=mdim*3;

  if (mdim ~= 2 & mdim ~= 3),
    error('only 2D and 3D mesh are handled by this function');
  end;
  
  opt = struct('zplot','off',...   % only for qdim=1, mdim=2
	       'norm','off',...     % if qdim >= 2, color-plot the norm of the field
	       'dir',[],...       % or the scalar product of the field with 'dir' (can be a vector, or 'x', 'y' etc..)
	       'refine',8,...         % nb of refinments for curved edges and surface plots
	       'interpolated','off',... %if triangular patch are interpolated
	       'pcolor','on',... % if the field is scalar, a color plot of its values is plotted
	       'quiver','on',... % if the field is vector, represent arrows 	       
	       'quiver_density',50,... % density of arrows in quiver plot
	       'quiver_scale',1,... % scaling of arrows (0=>no scaling)
	       'mesh','off',...  % show the mesh ?
	       'meshopts',{cell(0)},... % cell array of options passed to gf_plot_mesh for the mesh 
	       'deformed_mesh', 'off',... % shows the deformed mesh (only when qdim == mdim)
	       'deformed_meshopts', {cell(0)},... % cell array of options passed to gf_plot_mesh for the deformed mesh 
	       'deformation',[],... % plots on the deformed object (only when qdim == mdim)
	       'deformation_mf',[],... % plots on the deformed object (only when qdim == mdim)
	       'deformation_scale','10%',... % indicate the amplitude of the deformation. Can be a percentage of the mesh width if given as a string, or an absolute value if given as a number
	       'cvlst',[],... % list of convexes to plot
	       'title',[],...
	       'contour',[]); % list of contour values

  opt = getopt(opt,varargin);

  disp(opt);
  if (ison(opt.zplot))
    if (mdim ~= 2),
      error('zplot allowed only on 2D scalar mesh_fems');
    else opt.interpolated = 'on'; % or patch won't work
    end;
  end;
  is_scalarplot = (ison(opt.norm) + ~isempty(opt.dir));
  if (is_scalarplot > 1),
    error('only one occurence of the options ''norm'' and ''dir'' is allowed');
  end;
  if (ischar(opt.dir)),
    v = zeros(1,qdim);
    if (strcmp(opt.dir,'x'))
      v(1)=1;
    elseif (strcmp(opt.dir,'y'))
      v(2)=1;
    elseif (strcmp(opt.dir,'z'))
      v(3)=1;
    else error('wrong direction');
    end;
    opt.dir=v;
  end;
  scalarplot_dir=opt.dir(:);
  if (qdim == 1) is_scalarplot = 1; scalarplot_dir=1; end;
  if (~isempty(opt.contour) & ~is_scalarplot),
    error('contour plot has no meaning for a vector field');
  end;
  mfdef = mf;
  if (~isempty(opt.deformation_mf)) mfdef = struct(opt.deformation_mf); end;
  dqdim = gf_mesh_fem_get(mfdef,'qdim');
  if (~isempty(opt.deformation) | ison(opt.deformed_mesh)),
    if (mdim ~= dqdim),
      error(sprintf('can''t plot the deformation of an %dD-object by a %dD-field',mdim,dqdim));
    end;
  end;

  if (isempty(opt.cvlst)), opt.cvlst = gf_mesh_get(mf,'cvid'); end;

  nbdof=gf_mesh_fem_get(mf,'nbdof');
  if (nbdof <= 0), error('invalid finite element mf argument'); end

  if (length(U) ~= nbdof),
    error(['wrong dimensions for U, should' ...
	   ' have ' int2str(nbdof) 'columns']); 
  end

  % interpolate the solution on a refined P1 mesh
  P = gf_compute(mf, U, 'eval on triangulated surface', opt.refine, opt.cvlst);
  % get vector values
  vV = P((mdim3+1):(mdim3+qdim*3),:);

  if (~isempty(opt.deformation) | mfdef.id ~= mf.id),
    if (~isempty(opt.deformation)), Udef = opt.deformation; 
    else Udef = U; end;
    Pdef = gf_compute(mfdef, Udef, 'eval on triangulated surface', opt.refine, opt.cvlst);
    vVdef = Pdef((mdim3+1):(mdim3+dqdim*3),:);
    d = Pdef(1:mdim3,:)-P(1:mdim3,:);
    if (size(vVdef,2) ~= size(vV,2) | max(abs(d(:))) > 1e-12)
      error('the deformation mesh_fem and the data mesh_fem do not seem to share the same mesh');
    end;
    clear Pdef; clear d;
  else
    Udef = U; vVdef = vV;
  end;
  
  if (isnumeric(opt.deformation_scale)), dscale = opt.deformation_scale;
  elseif (ischar(opt.deformation_scale) & ...
	   numel(opt.deformation_scale) & opt.deformation_scale(end)=='%'),
      dscale = str2num(opt.deformation_scale(1:end-1));
      mwidth = max([max(P(1:mdim3,:),[],2) - min(P(1:mdim3,:),[],2)]);
      vVmax = max(abs(vVdef(:)));
      if (vVmax),	
	dscale = dscale * 0.01 * mwidth / vVmax;
      end;
  else error('wrong value for deformation_scale: should be a number or a percentage in a string');
  end;

  
  % save graphical context
  cax = newplot;
  cfig = get(cax,'Parent');
  hold_state = ishold;
  ax_nextplot = lower(get(cax,'NextPlot'));
  fig_nextplot = lower(get(cfig,'NextPlot'));

  
  if (ison(opt.zplot) | mdim == 3), view(3); else view(2); end;


  if (~isempty(opt.deformation)),
    P(1:mdim3,:) = P(1:mdim3,:) + dscale * vVdef;
  end;

  if (ison(opt.deformed_mesh)),
    hdefmesh={gf_plot_mesh(mf,'cvlst',opt.cvlst,'deformation',Udef*dscale, opt.deformed_meshopts{:})};
    hold on; 
  end;
  if (ison(opt.mesh)), 
    hmesh={gf_plot_mesh(mf,'cvlst',opt.cvlst,opt.meshopts{:})};
    hold on; 
  end;


  if (is_scalarplot),
    % and compute scalar values if necessary
    for i=1:3
      z = vV(((i-1)*qdim+1):(i*qdim),:);
      if (ison(opt.norm)),
	sV(i,:) = sqrt(sum(z.*conj(z),1));
      else
	sV(i,:) = scalarplot_dir'*z;
      end;
    end;
    
    if (~ison(opt.interpolated)),
      siV=mean(sV);
    else
      siV=sV;
    end;
    
    % plot the 'surfacic' part
    if (ison(opt.pcolor)),
      if (~ison(opt.zplot)),
	if (mdim == 2),
	  hsurf=patch(P(1:mdim:(mdim*3),:),P(2:mdim:(mdim*3),:), ...
		      siV, 'Parent', cax, 'Erasemode','normal','Edgecolor','none');
	else      
	  hsurf=patch(P(1:mdim:(mdim*3),:),P(2:mdim:(mdim*3),:), P(3:mdim:(mdim*3),:), ...
		      siV, 'Parent', cax, 'Erasemode','normal','Edgecolor','none');
	end;
      else
	hsurf=patch(P(1:2:6,:),P(2:2:6,:), siV, siV, 'Parent', cax, 'Erasemode','normal','Edgecolor','none');
      end;
    end;
    
    % basic contour plot (should work also on 3D surfaces)
    contour_colors = [.9 0 0; 0 .8 0; .0 0 .6; .6 .6 0; .7 0 .7; 0 .7 .9]; 
    contour_linestyle = get(cax,'LineStyleOrder');
    hcontour=cell(numel(opt.contour),1);
    for cnum=1:numel(opt.contour),
      c=opt.contour(cnum);
      C=sV;
      a=sum(C>c); a=find(a~=0 & a ~= 3); % find triangle which cross the contour line
      LX=[]; LY=[]; LZ=[];
      hcontour{cnum}=[];
      for t=a,
	g = [0 0 0];
	v = C(:,t);
	if (abs(v(2)-v(1))>eps), g(3) = (c-v(1))/(v(2)-v(1)); end;
	if (abs(v(3)-v(1))>eps), g(2) = (c-v(1))/(v(3)-v(1)); end;
	if (abs(v(3)-v(2))>eps), g(1) = (c-v(2))/(v(3)-v(2)); end;
	ok = (g>0 & g<1);
	X=P(1:mdim:mdim3,t);
	Y=P(2:mdim:mdim3,t); LX = []; LY=[];
	if (mdim==3), Z=P(3:mdim:mdim3,t); LZ = []; end;
	if (ok(1)),
	  LX=X(2) + g(1)*(X(3)-X(2));
	  LY=Y(2) + g(1)*(Y(3)-Y(2));
	  if (mdim==3) LZ=Z(2) + g(1)*(Z(3)-Z(2));end;
	end;
	if (ok(2)),
	  LX=[LX X(1) + g(2)*(X(3)-X(1))];
	  LY=[LY Y(1) + g(2)*(Y(3)-Y(1))];
	  if (mdim==3) LZ=[LZ Z(1) + g(2)*(Z(3)-Z(1))];end;
	end;
	if (ok(3)),
	  LX=[LX X(1) + g(3)*(X(2)-X(1))];
	  LY=[LY Y(1) + g(3)*(Y(2)-Y(1))];
	  if (mdim==3) LZ=[LZ Z(1) + g(3)*(Z(2)-Z(1))];end;
	end;
	
	if (numel(LX)==2), 
	  if (mdim == 2)
	    arg={'XData',LX,'YData',LY};
	  else arg={'XData',LX,'YData',LY,'ZData',LZ}; end;
	  hh=line(arg{:});
	  if (~isempty(hh)), hcontour{cnum} = [hcontour{cnum} hh]; end;
	end;
      end;
      if (~isempty(hcontour{cnum})),
        set(hcontour{cnum},...
            'Color',contour_colors(mod(cnum,size(contour_colors,1))+1,:),...
            'LineStyle',contour_linestyle(mod(cnum,numel(contour_linestyle))+1),...
            'LineWidth',1);
      end;
    end;
  else
    % vector plot
    xy=gf_mesh_fem_get(mf, 'dof nodes');
    ptlst=1:qdim:nbdof;
    bmin = min(xy); bmax = max(xy);
    xyscale = max(bmax-bmin);
    qradius2 = (xyscale/opt.quiver_density)^2;
    vscale  = max(max(abs(U)));
    qlst = [];
    rm=[];
    while (numel(ptlst)>0)
      ii = ptlst(1); 
      qlst = [qlst ii];
      x = xy(1,ii); y = xy(2, ii); 
      if (mdim == 2),
	rm = (find((xy(1,:)-x).^2 + (xy(2,:)-y).^2 < qradius2));
      elseif (mdim == 3),
	z = xy(3,ii);
	rm = (find((xy(1,:)-x).^2 + (xy(2,:)-y).^2 + (xy(3,:)-z).^2 < qradius2));
      end;
      if (numel(rm)==0) error('internal error in gf_plot'); end;
      ptlst = setdiff(ptlst, rm);
    end;
    if (qdim == 2),
      hquiver=quiver(xy(1,qlst),xy(2,qlst),U(qlst),U(qlst+1),opt.quiver_scale);
      set(hquiver,'Color', [0 .4 0]);
    else
      hquiver=quiver3(xy(1,qlst),xy(2,qlst),xy(3,qlst),U(qlst),U(qlst+1),U(qlst+2),...
		      opt.quiver_scale);
      set(hquiver,'Color', [0 .4 0]);
    end;
  end
  
  
  if (~hold_state),
    set(cax,'DataAspectRatio', [1 1 1]);
    set(cax,'XLimMode','auto');
    set(cax,'YLimMode','auto');
    set(cax,'ZLimMode','auto');
  end;
  
  % restore graphical context
  set(cax,'NextPlot',ax_nextplot);
  set(cfig,'NextPlot',fig_nextplot);


function r=ison(v)
  r = strcmpi(v,'on');