#include <map>
#include <matlabint_misc.h>
#include <matlabint_mesh.h>
#include <matlabint_convex_structure.h>
#include <matlabint_pgt.h>
#include <getfem_export.h>

using namespace matlabint;

void check_empty_mesh(const getfem::getfem_mesh *pmesh)
{
  if (pmesh->dim() == bgeot::dim_type(-1) || pmesh->dim() == 0) {
    THROW_ERROR( "mesh object has an invalid dimension");
  }
}

/* this function assumes that the edges are sorted by convex number 
   it tries to reorder the edges of each convex such that they form a closed loop
   of course, the result is only interesting for edges of 2D faces..
*/
/*
function removed because stupid and useless


void loop_edge_list(bgeot::edge_list &el) {
  bgeot::edge_list::iterator it;
  it = el.begin();
  while (it != el.end()) {
    bgeot::edge_list::iterator it2 = it+1, itn = it+1;
    // brute search, won't be efficient for convexes with 500 edges 
    while (it2 != el.end() && (*it2).cv == (*it).cv) {
      if ((*it).j == (*it2).i) {
	if (it2 != itn) std::swap(*itn,*it2); break;
      } else if ((*it).j == (*it2).j) {
	if (it2 != itn) {
	  std::swap((*itn).i,(*it2).j);
	  std::swap((*itn).j,(*it2).i); 
	} else std::swap((*itn).i,(*itn).j); break;
      }
      ++it2;
    }
    it = itn;
  }
}
*/
void
get_structure_or_geotrans_of_convexes(const getfem::getfem_mesh& m, 
				      mexargs_in& in, mexargs_out& out, 
				      id_type class_id)
{  
  dal::bit_vector cvlst;
  if (in.remaining()) cvlst = in.pop().to_bit_vector(&m.convex_index());
  else cvlst = m.convex_index();
  std::vector<id_type> ids; ids.reserve(cvlst.card());
  size_type cv;
  for (cv << cvlst; cv != size_type(-1); cv << cvlst)
    if (class_id == CVSTRUCT_CLASS_ID)
      ids.push_back(ind_convex_structure(m.structure_of_convex(cv)));
    else ids.push_back(ind_pgt(m.trans_of_convex(cv)));
  out.return_packed_obj_ids(ids, class_id);
}

/* apply geomtric transformation to edge lists */
void
transform_edge_list(const getfem::getfem_mesh &m, unsigned N, const bgeot::edge_list &el, mlab_vect &w)
{
  bgeot::edge_list::const_iterator it;
  size_type cv = size_type(-1);
  getfem::getfem_mesh::ref_convex cv_ref;
  bgeot::pgeometric_trans pgt = NULL;
  size_type ecnt = 0;
  for (it = el.begin(); it != el.end(); it++, ecnt++) {
    if (cv != (*it).cv) {
      cv = (*it).cv;
      cv_ref = m.convex(cv);
      pgt = m.trans_of_convex(cv);
    }
    /* build the list of points on the edge, on the reference element */
    /* get local point numbers in the convex */
    bgeot::size_type iA = m.local_ind_of_convex_point(cv, (*it).i);
    bgeot::size_type iB = m.local_ind_of_convex_point(cv, (*it).j);

    if (iA >= cv_ref.nb_points() ||
	iB >= cv_ref.nb_points() || iA == iB) {
      mexPrintf("iA=%d , iB=%d\n",iA,iB);
      THROW_INTERNAL_ERROR;
    }
    //getfem::base_node A = m.points_of_convex(cv)[iA]; //cv_ref.points()[iA];
    //getfem::base_node B = m.points_of_convex(cv)[iB]; //cv_ref.points()[iB];
    getfem::base_node A = pgt->convex_ref()->points()[iA];
    getfem::base_node B = pgt->convex_ref()->points()[iB];
    //cerr << "A=" << A << ", B=" << B << endl;
    for (size_type i = 0; i < N; i++) {
      getfem::base_node pt;
      
      pt = A +  (B-A)*(i/(double)(N-1));
      //cerr << "pt original=" << pt;
      pt = pgt->transform(pt, m.points_of_convex(cv));
      //cerr << ", transf = " << pt << endl;
      std::copy(pt.begin(), pt.end(), &w(0,i, ecnt));
    }
  }
}

typedef dal::dynamic_tree_sorted<convex_face> mesh_faces_list;

void
faces_from_pid(const getfem::getfem_mesh &m, mexargs_in &in, mexargs_out &out)
{
  dal::bit_vector pts = in.pop().to_bit_vector(&m.points().index());
  mesh_faces_list lst;
  dal::bit_vector convex_tested;

  for (dal::bit_vector::const_iterator it_pts = pts.begin(); it_pts != pts.end(); ++it_pts) {
    if (!(*it_pts)) continue;

    size_type ip = it_pts.index();
    /* iterator over all convexes attached to point ip */
    bgeot::mesh_convex_ind_ct::const_iterator 
      cvit = m.convex_to_point(ip).begin(),
      cvit_end = m.convex_to_point(ip).end();

    //    cerr << "point " << ip+1 << endl;

    for ( ; cvit != cvit_end; cvit++) {
      size_type ic = *cvit;
      if (!convex_tested.is_in(ic)) {
	convex_tested.add(ic);
	/* loop over the convex faces */
	for (size_type f = 0; f < m.structure_of_convex(ic)->nb_faces(); f++) {
	  bgeot::ind_ref_mesh_point_ind_ct pt = m.ind_points_of_face_of_convex(ic, f);
	  bool ok = true;
	  for (bgeot::ind_ref_mesh_point_ind_ct::const_iterator it = pt.begin(); it != pt.end(); it++) {
	    //	    cerr << "testing cv=" << ic+1 << " face=" << f+1 << " point[" << (*it)+1 << "] ? " << pts.is_in(*it) << endl;
	    if (!pts.is_in(*it)) { 
	      ok = false; break; 
	    }
	  }
	  if (ok) {
	    lst.add(convex_face(ic,f));
	    //	    cerr << "added!\n";
	  }
	}
      }
    }
  }
  mlab_vect w = out.pop().create_vector(2,lst.size());
  for (size_type j=0; j < lst.size(); j++) { w(0,j) = lst[j].cv+1; w(1,j) = lst[j].f+1; }
}


struct mesh_faces_by_pts_list_elt  {
  std::vector<size_type> ptid; // point numbers of faces
  int cnt; // number of convexes sharing that face
  int cv, f;
  inline bool operator < (const mesh_faces_by_pts_list_elt &e) const
  {
//     cerr << " p1 = " << ptid[0] << " " << ptid[1] << " p2=" << e.ptid[0] << " " << e.ptid[1] << endl;
//     cerr << " p1<p2 ? " << ((ptid < e.ptid) ? true : false) << endl;
    if (ptid.size() < e.ptid.size()) return true;
    if (ptid.size() > e.ptid.size()) return false;
    return ptid < e.ptid;
  }

  mesh_faces_by_pts_list_elt(size_type _cv, size_type _f, std::vector<size_type>& p) : cv(_cv), f(_f) {
    cnt = 0;
    if (p.size() == 0) THROW_INTERNAL_ERROR;
    std::sort(p.begin(), p.end());

//     cerr << "ajout cv=" << cv << " f=" << f << " pt=["; 
//     std::copy(p.begin(), p.end(), std::ostream_iterator<size_type,char>(cerr, ","));
//     cerr << "]\n";
    ptid = p; 
  }
  mesh_faces_by_pts_list_elt() {}
};
typedef dal::dynamic_tree_sorted<mesh_faces_by_pts_list_elt> mesh_faces_by_pts_list;


void
outer_faces(const getfem::getfem_mesh &m, mexargs_in &in, mexargs_out &out)
{
  mesh_faces_by_pts_list lst;
  dal::bit_vector convex_tested;
  dal::bit_vector cvlst;

  if (in.remaining()) cvlst = in.pop().to_bit_vector(&m.convex_index());
  else cvlst = m.convex_index();

  
  for (dal::bit_vector::const_iterator it_cv = cvlst.begin(); it_cv != cvlst.end(); ++it_cv) {
    if (!(*it_cv)) continue;

    size_type ic = it_cv.index();

    if (m.structure_of_convex(ic)->dim() == m.dim()) {
      for (size_type f = 0; f < m.structure_of_convex(ic)->nb_faces(); f++) {
	bgeot::ind_ref_mesh_point_ind_ct pt = m.ind_points_of_face_of_convex(ic, f);
	
	
	std::vector<size_type> p(pt.begin(), pt.end());
	size_type idx = lst.add_norepeat(mesh_faces_by_pts_list_elt(ic,f,p));
	//       cerr << " <-- idx = " << idx << endl;
	lst[idx].cnt++;
      }
    } else { /* les objets de dim inferieure sont considrs comme "exterieurs" 
		(c'ets plus pratique pour faire des dessins)
	      */
      bgeot::ref_mesh_point_ind_ct pt = m.ind_points_of_convex(ic);		
      std::vector<size_type> p(pt.begin(), pt.end());
      size_type idx = lst.add_norepeat(mesh_faces_by_pts_list_elt(ic,size_type(-1),p));
      lst[idx].cnt++;
    }
  }
  size_type fcnt = 0;
  for (size_type i = 0; i < lst.size(); i++) {
    fcnt += (lst[i].cnt == 1 ? 1 : 0);
  }

  mlab_vect w = out.pop().create_vector(2,fcnt);
  for (size_type j=0, fcnt=0; j < lst.size(); j++) { 
    if (lst[j].cnt == 1) {
      w(0,fcnt) = lst[j].cv+1; 
      w(1,fcnt) = lst[j].f+1; 
      fcnt++;
    }
  }
}


bgeot::base_node 
normal_of_face(const getfem::getfem_mesh& mesh, size_type cv, bgeot::dim_type f, size_type node) {
  if (!mesh.convex_index().is_in(cv)) THROW_BADARG("convex " << cv+1 << " not found in mesh");
  if (f >= mesh.structure_of_convex(cv)->nb_faces()) 
    THROW_BADARG("convex " << cv+1 << " has only " << 
		 mesh.structure_of_convex(cv)->nb_faces() << 
		 ": can't find face " << f+1);
  if (node >= mesh.structure_of_convex(cv)->nb_points_of_face(f))
    THROW_BADARG("invalid node number: " << node);
  bgeot::base_node N = mesh.normal_of_face_of_convex(cv, f, node);
  N /= bgeot::vect_norm2(N);
  return N;
}

/*MLABCOM

  FUNCTION [...] = gf_mesh_get(mesh M, [operation [, args]])

  General mesh inquiry function. All these functions accept also a 
  mesh_fem argument instead of a mesh M (in that case, the mesh_fem
  linked mesh will be used).

  * I = gf_mesh_get(M, 'dim')
  Returns the dimension of the mesh (2 for a 2D mesh, etc..)

  * I = gf_mesh_get(M, 'nbpts')
  Returns the number of points of the mesh.

  * I = gf_mesh_get(M, 'nbcvs')
  Returns the number of convexes of the mesh.

  * [PT] = gf_mesh_get(M, 'pts' [,pids])
  Returns the list of points of mesh M, each point being stored in a column of
  PT. If PIDLST is specified, only those points are listed. Otherwise, PT will
  have gf~mesh~get(M, 'max pid') rows, which might be greater than
  gf~mesh~get(M, 'nbpts') (if you destroyed some convexes or points in the mesh
  for example). The columns corresponding to inexistant points will be filled
  with NaN. You can use gf~mesh~get(M, 'pid') to filter such invalid points.

  If the optional argument pid was specified, then PT will contain the
  respective coordinates of points listed in pids.

  * [PTID] = gf_mesh_get(M, 'pid')
  Return the list of points numbers composing M (their numbering is
  not supposed to be contiguous from 1 to gf_mesh_get(M,'nbpts'),
  especially if you destroyed some convexes) in a row vector.

  * [CVID] = gf_mesh_get(M, 'cvid')
  Returns the list of convex numbers composing M (their numbering is
  not supposed to be contiguous from 1 to gf_mesh_get('nbcvs'),
  especially if you destroyed some convexes) in a row vector.

  * I = gf_mesh_get(M, 'max pid')
  Returns the maximum ID of all points in the mesh (this is the same
  value as MAX(gf_mesh_get(M, 'pid')), but it won't be equal to 
  gf_mesh_get(M, 'nbpts') if some points have been destroyed and the
  mesh was not 'repacked'.

  * I = gf_mesh_get(M, 'max cvid')
  Returns the maximum ID of all convexes in the mesh (see 'max pid').

  * [PID,IDX] = gf_mesh_get(M, 'pid from cvid'[,CVLST])
  IDX is a row vector, length(IDX) = length(CVLIST)+1. PID is a row
  vector containing the concatenated list of points of each convex in
  cvlst. Each entry of IDX is the position of the corresponding convex
  point list in PID. Hence, for example, the list of points of the
  second convex is PID(IDX(2):IDX(3)-1).

  If you specified convex numbers which do not exist in CVLST, their
  point list will be empty.

  A good way to use this function in order to quickly find the point
  list of any convex is to call
     [PID,IDX] = gf_mesh_get(M, 'pid from cvid', 1:gf_mesh_get(M,'max cvid'));


  * [V] = gf_mesh_get(M, 'pid from coords', mat PT)
  PT is an array containing a list of point coordinates. On return, V is a row
  vector containing the mesh id of the points which are part of the mesh, and
  -1 for those which where not found in the mesh.

  * [V] = gf_mesh_get(M, 'cvid from pid', vec PTID)
  Returns the list of convexes that share the points numbers given in PTID in a
  row vector (possibly empty).


  * [V] = gf_mesh_get(M, 'faces from pid', vec PTID)
  Returns the list of convexes faces of which every vertex is in PTID.
  On return, the first row of V contains the convex number, and the
  second row contains the face number.

  * CVFACELST = gf_mesh_get(M, 'outer faces' [, CVLIST])
  Returns the list of faces which are not shared by two convexes (i.e. the
  face on the boundary of the mesh). The search can be restricted to the
  optionnal argument CVLIST.

  * [E,C] = gf_mesh_get(M, 'edges' [, CVLIST][,'merge convex'])

  Return the list of edges of mesh M for the convexes listed in the
  row vector CVLIST. E is a 2 x nb_edges matrix containing point
  indices. If CVLIST is omitted, then the edges of all convexes are
  returned. If CVLIST has two rows then the first row is supposed to
  contain convex numbers, and the second face numbers, of which the
  edges will be returned.  If 'merge convex' is indicated, all common
  edges of convexes are merged in a single edge.  If the optional
  output argument C is specified, it will contain the convex number
  associated with each edge.


  * [E,C] = gf_mesh_get(M, 'curved edges', int N, [, CVLIST])

  More sophisticated version of gf_mesh_get(M, 'edges') designed for
  curved elements. This one will return N (N>=2) points of the
  (curved) edges. With N==2, this is equivalent to gf_mesh_get(M,
  'edges'). Since the points are no more always part of the mesh,
  their coordinates are returned instead of points number, in the
  array E which is a [ mesh_dim x 2 x nb_edges ] array.  If the
  optional output argument C is specified, it will contain the convex
  number associated with each edge.

  * [T] = gf_mesh_get(M, 'triangulated surface', int Nrefine [,CVLIST])

  Similar function to 'curved edges' : split (if necessary, i.e. if
  the geometric transformation if non-linear) each face into
  sub-triangles and return their coordinates in T (see also
  gf_compute(mf,U,'eval on P1 tri mesh'))

  * [N] = gf_mesh_get(M, 'normal of face', int CV, int F[, int FPTNUM])
    [N] = gf_mesh_get(M, 'normal of faces', mat CVFACES)
  This command evaluates the normal of convex faces. The first form
  returns the normal of convex CV for its face F, evaluated at the
  FPTNUMth point of the face. If FPTNUM is not specified, then the
  normal is evaluated at each geometrical node of the face.  The
  second form returns the normal for a set of faces of convex, each
  normal being computed at the center of the face (CVFACES is supposed
  to contain convex numbers at its first row and convex face number in
  its second row)

  * [CVS,CV2STRUC] = gf_mesh_get(M, 'cvstruct',[CVLST])
  Returns an array of all the convex structure used in the mesh (optionnaly
  restricted to the convexes of CVLST), and a second optionnal output vector
  CV2STRUCT which maps the convexes indice in CVLST to the indice of its
  structure in CVS.

  * [GT,GT2STRUCT] = gf_mesh_get(M, 'geotrans',[CVLST])
  Returns an array of the geometric transformations (similar to gf_mesh_get(M,
  'cvstruct')

  * gf_mesh_get(M, 'save', string FILENAME)
  Save the mesh object to an ascii file. This mesh can be restored with
  gf_mesh('load', FILENAME);

  * S=gf_mesh_get(M, 'char')
  Output a string description of the mesh M.

  * ms=gf_mesh_get(M, 'memsize')
  Returns the amount of memory (in bytes) used by the mesh object.

  $Id: gf_mesh_get.C,v 1.24 2003/02/27 16:34:17 pommier Exp $
MLABCOM*/


void gf_mesh_get(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 2) {
    THROW_BADARG( "Wrong number of input arguments");
  }

  const getfem::getfem_mesh *pmesh = in.pop().to_const_mesh();
  std::string cmd                  = in.pop().to_string();
  if (check_cmd(cmd, "dim", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(pmesh->dim());
  } else if (check_cmd(cmd, "nbpts", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(pmesh->nb_points());
  } else if (check_cmd(cmd, "nbcvs", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(pmesh->nb_convex());
  } else if (check_cmd(cmd, "pts", in, out, 0, 1, 0, 1)) {
    double nan = mxGetNaN();
    if (!in.remaining()) {
      dal::bit_vector bv = pmesh->points().index();
      mlab_vect w = out.pop().create_vector(pmesh->dim(), bv.last_true()+1);
      for (size_type j = 0; j < bv.last_true()+1; j++) {
	for (size_type i = 0; i < pmesh->dim(); i++) {
	  w(i,j) = (bv.is_in(j)) ? (pmesh->points()[j])[i] : nan;
	}
      }
    } else {
      dal::bit_vector pids = in.pop().to_bit_vector();
      mlab_vect w = out.pop().create_vector(pmesh->dim(), pids.card());
      size_type j,cnt=0;
      for (j << pids; j != size_type(-1); j << pids) {
	if (!pmesh->points().index().is_in(j))
	  THROW_ERROR("point " << j+1 << " is not part of the mesh");
	for (size_type i = 0; i < pmesh->dim(); i++) {
	  w(i,cnt) = (pmesh->points()[j])[i];
	}
	cnt++;
      }
    }
  } else if (check_cmd(cmd, "pid", in, out, 0, 0, 0, 1)) {
    out.pop().from_bit_vector(pmesh->points().index());
  } else if (check_cmd(cmd, "cvid", in, out, 0, 0, 0, 1)) {
    out.pop().from_bit_vector(pmesh->convex_index());
  } else if (check_cmd(cmd, "max pid", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(pmesh->points().index().card() ? pmesh->points().index().last_true()+1 : 0);
  } else if (check_cmd(cmd, "max cvid", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(pmesh->convex_index().card()? pmesh->convex_index().last_true()+1 : 0);
  }
 else if (check_cmd(cmd, "pid from cvid", in, out, 0, 1, 0, 2)) {
    dal::bit_vector cvlst;
    if (in.remaining()) cvlst = in.pop().to_bit_vector();
    else cvlst = pmesh->convex_index();
    
    size_type pcnt = 0;
    /* phase one: count the total number of pids */
    for (dal::bit_vector::const_iterator it = cvlst.begin();
	 it != cvlst.end(); ++it) {
      if (!*it) continue;
      size_type cv = it.index();
      if (pmesh->convex_index().is_in(cv)) {
	pcnt += pmesh->structure_of_convex(cv)->nb_points();
      }
    }
    /* phase two: allocation */
    mlab_vect pid = out.pop().create_vector(1, pcnt);
    bool fill_idx = out.remaining();
    mlab_vect idx; if (fill_idx) idx = out.pop().create_vector(1, cvlst.card() + 1);

    pcnt = 0;
    size_type cvcnt = 0;
    /* phase three: build the list */
    for (dal::bit_vector::const_iterator it = cvlst.begin();
	 it != cvlst.end(); ++it) {
      if (!*it) continue; 
      size_type cv = it.index();
      if (fill_idx) idx(0,cvcnt) = double(pcnt+1);
      if (pmesh->convex_index().is_in(cv)) {
	for (bgeot::ref_mesh_point_ind_ct::const_iterator pit = 
	       pmesh->ind_points_of_convex(cv).begin();
	     pit != pmesh->ind_points_of_convex(cv).end(); ++pit) {
	  pid(0,pcnt++) = double((*pit) + 1);
	}
      }
      cvcnt++;
    }
    if (fill_idx) idx(0,idx.getn()-1) = double(pcnt+1); /* for the last convex */
  } else if (check_cmd(cmd, "edges", in, out, 0, 2, 0, 2)) {
    bgeot::edge_list el;
    build_edge_list(*pmesh, el, in);
    mlab_vect w   = out.pop().create_vector(2, el.size());
    /* copy the edge list to the matlab array */
    for (size_type j = 0; j < el.size(); j++) { 
      w(0,j) = (el[j].i + 1);
      w(1,j) = (el[j].j + 1);
    }
    if (out.remaining()) {
      mlab_vect cv = out.pop().create_vector(1,el.size());
      for (size_type j = 0; j < el.size(); j++) { 
	cv[j] = el[j].cv+1;
      }
    }
  } else if (check_cmd(cmd, "curved edges", in, out, 1, 2, 0, 2)) {    
    bgeot::edge_list el;
    size_type N = in.pop().to_integer(2,10000);
    build_edge_list(*pmesh, el, in);
    mlab_vect w   = out.pop().create_vector(pmesh->dim(), N, el.size());
    transform_edge_list(*pmesh, N, el, w);
    if (out.remaining()) {
      mlab_vect cv = out.pop().create_vector(1,el.size());
      for (size_type j = 0; j < el.size(); j++) { 
	cv[j] = el[j].cv+1;
      }
    }
  } else if (check_cmd(cmd, "pid from coords", in, out, 1, 1, 0, 1)) {
    check_empty_mesh(pmesh);
    mlab_vect v   = in.pop().to_scalar_vector(pmesh->dim(), -1);
    mlab_vect w   = out.pop().create_vector(1, v.getn());
    for (size_type j=0; j < v.getn(); j++) {
      getfem::size_type id = pmesh->search_point(v.col_to_bn(j));
      if (id == getfem::size_type(-1)) w[j] = double(-1);
      else w[j] = double(id + 1);
    }
  } else if (check_cmd(cmd, "cvid from pid", in, out, 1, 1, 0, 1)) {
    check_empty_mesh(pmesh);
    dal::bit_vector pts = in.pop().to_bit_vector(&pmesh->points().index());
    dal::bit_vector cvchecked;

    std::vector<size_type> cvlst;

    /* loop over the points */
    for (dal::bit_vector::const_iterator it_pt = pts.begin(); it_pt != pts.end(); ++it_pt) {
      if (!*it_pt) continue;
      size_type ip = it_pt.index();

      /* iterators over the convexes attached to point ip */
      bgeot::mesh_convex_ind_ct::const_iterator 
	cvit = pmesh->convex_to_point(ip).begin(),
	cvit_end = pmesh->convex_to_point(ip).end();    

      for ( ; cvit != cvit_end; cvit++) {
	size_type ic = *cvit;

	//	cerr << "cv = " << ic+1 << endl;

	if (!cvchecked.is_in(ic)) {
	  bool ok = true;
	  
	  bgeot::ref_mesh_point_ind_ct cvpt = pmesh->ind_points_of_convex(ic);
	  /* check that each point of the convex is in the list */
	  for (bgeot::ref_mesh_point_ind_ct::const_iterator it = cvpt.begin();
	       it != cvpt.end(); ++it) {
	    //	    cerr << "  test pt " << (*it)+1 << ": is_in=" << pts.is_in(*it) << endl;
	    if (!pts.is_in(*it)) {
	      ok = false; break;
	    }
	  }
	  if (ok) cvlst.push_back(ic);
	  cvchecked.add(ic);
	}
      }
    }
    mlab_vect w = out.pop().create_vector(1,cvlst.size());
    for (size_type j=0; j < cvlst.size(); j++) w[j] = cvlst[j]+1;
  } else if (check_cmd(cmd, "faces from pid", in, out, 1, 1, 0, 1)) {
    check_empty_mesh(pmesh);
    faces_from_pid(*pmesh, in, out);
  } else if (check_cmd(cmd, "outer faces", in, out, 0, 1, 0, 1)) {
    check_empty_mesh(pmesh);
    outer_faces(*pmesh, in, out);
  } else if (check_cmd(cmd, "triangulated surface", in, out, 1, 2, 0, 1)) {
    int Nrefine = in.pop().to_integer(1, 1000);
    std::vector<convex_face> cvf;
    if (in.remaining() && !in.front().is_string()) {
      mlab_vect v = in.pop().to_int_vector(-1, -1);
      build_convex_face_lst(*pmesh, cvf, &v);
    } else build_convex_face_lst(*pmesh, cvf, 0);
    eval_on_triangulated_surface(pmesh, Nrefine, cvf, out, NULL, mlab_vect());
  } else if (check_cmd(cmd, "normal of face", in, out, 2, 3, 0, 1)) {
    size_type cv = in.pop().to_integer()-1;
    size_type f  = in.pop().to_integer()-1;
    size_type node = 0; 
    if (in.remaining()) node = in.pop().to_integer(0,10000)-1; 
    bgeot::base_node N = normal_of_face(*pmesh, cv, f, node);
    out.pop().from_vector(N);
  } else if (check_cmd(cmd, "normal of faces", in, out, 1, 1, 0, 1)) {
    mlab_vect v            = in.pop().to_int_vector(2,-1);
    mlab_vect w            = out.pop().create_vector(pmesh->dim(), v.getn());    
    for (size_type j=0; j < v.getn(); j++) {
      size_type cv = size_type(v(0,j))-1;
      size_type f  = size_type(v(1,j))-1;
      bgeot::base_node N = normal_of_face(*pmesh, cv, f, 0);
      for (size_type i=0; i < pmesh->dim(); ++i) w(i,j)=N[i];
    }
  } else if (check_cmd(cmd, "cvstruct", in, out, 0, 1, 0, 2)) {
    get_structure_or_geotrans_of_convexes(*pmesh, in, out, CVSTRUCT_CLASS_ID);
  } else if (check_cmd(cmd, "geotrans", in, out, 0, 1, 0, 2)) {
    get_structure_or_geotrans_of_convexes(*pmesh, in, out, GEOTRANS_CLASS_ID);
  } else if (check_cmd(cmd, "save", in, out, 1, 1, 0, 0)) {
    std::string fname = in.pop().to_string();
    //    mexPrintf("saving to %s\n", fname.c_str());
    pmesh->write_to_file(fname);
  } else if (check_cmd(cmd, "char", in, out, 0, 0, 0, 1)) {
    std::stringstream s;
    pmesh->write_to_file(s);
#ifdef USING_BROKEN_GCC295 /* with memory leaks */
    out.pop().from_string(s.str());
#else
    out.pop().from_string(s.str().c_str());
#endif
  } else if (check_cmd(cmd, "memsize", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(pmesh->memsize());
  } else bad_cmd(cmd);
}

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  catch_errors(nlhs, plhs, nrhs, prhs, gf_mesh_get);
}
