#include <matlabint.h>
#include <matlabint_pgt.h>

using namespace matlabint;

/*MLABCOM
  FUNCTION I = gf_geotrans_get(GT, ...)
    General function for querying information about geometric transformations 
    objects.

    * gf_geotrans_get(GT,'dim')
    * gf_geotrans_get(GT,'is_linear')
    * gf_geotrans_get(GT,'nbpts')
    * gf_geotrans_get(GT,'pts')
    * gf_geotrans_get(GT,'normals')
    * gf_geotrans_get(GT,'poly_str')

    * S=gf_geotrans_get(GT,'char')
    Ouput a (unique) string representation of the GT. This can be used to 
    perform comparisons between two different GT objects.

    
MLABCOM*/

void gf_geotrans_get(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 2) {
    THROW_BADARG( "Wrong number of input arguments");
  }
  bgeot::pgeometric_trans pgt = in.pop().to_pgt();
  std::string cmd = in.pop().to_string();
  if (check_cmd(cmd, "dim", in, out, 0, 0, 0, 1)) {
    out.pop().from_scalar(pgt->dim());
  } else if (check_cmd(cmd, "is_linear", in, out, 0, 0, 0, 1)) {
    out.pop().from_scalar(pgt->is_linear() ? 1. : 0.);
  } else if (check_cmd(cmd, "nbpts", in, out, 0, 0, 0, 1)) {
    out.pop().from_scalar(pgt->nb_points());
  } else if (check_cmd(cmd, "pts", in, out, 0, 0, 0, 1)) {
    out.pop().from_vector_container(pgt->convex_ref()->points());
  } else if (check_cmd(cmd, "normals", in, out, 0, 0, 0, 1)) {
    out.pop().from_vector_container(pgt->normals());
  } else if (check_cmd(cmd, "poly_str", in, out, 0, 0, 0, 1)) {
    std::vector<std::string> s(pgt->poly_vector().size());
    for (size_type i=0; i < s.size(); ++i) {
      std::stringstream ss; ss << pgt->poly_vector()[i];
      s[i] = ss.str();
    }
    out.pop().from_string_container(s);
  } else if (check_cmd(cmd, "char", in, out, 0, 0, 0, 1)) {
    std::string s = bgeot::name_of_geometric_trans(pgt);    
    out.pop().from_string(s.c_str());
  } else bad_cmd(cmd);
}

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  catch_errors(nlhs, plhs, nrhs, prhs, gf_geotrans_get);
}
