#include <matlabint.h>
#include <matlabint_pfem.h>

using namespace matlabint;

/*MLABCOM
  FUNCTION I = gf_fem_get(F, ...)
    General function for querying information about FEM objects.
    
    * gf_fem_get(F,'nbdof')
    * gf_fem_get(F,'dim')
    * gf_fem_get(F,'target_dim')
    * gf_fem_get(F,'pts')
    * gf_fem_get(F,'is_equivalent')
    * gf_fem_get(F,'is_lagrange')
    * gf_fem_get(F,'is_polynomial')
    * gf_fem_get(F,'estimated_degree')
    * gf_fem_get(F,'base_value',X)
    * gf_fem_get(F,'grad_base_value',X)
    * gf_fem_get(F,'hess_base_value',X)

    * S=gf_fem_get(F,'char')
    Ouput a (unique) string representation of the FEM. This can be used to 
    perform comparisons between two different FEM objects.

    
MLABCOM*/

void gf_fem_get(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 2) {
    THROW_BADARG( "Wrong number of input arguments");
  }
  getfem::pfem fem = in.pop().to_fem();
  std::string cmd = in.pop().to_string();
  if (check_cmd(cmd, "nbdof", in, out, 0, 0, 0, 1)) {
    out.pop().from_scalar(fem->nb_dof());
  } else if (check_cmd(cmd, "dim", in, out, 0, 0, 0, 1)) {
    out.pop().from_scalar(fem->dim());
  } else if (check_cmd(cmd, "target_dim", in, out, 0, 0, 0, 1)) {
    out.pop().from_scalar(fem->target_dim());
  } else if (check_cmd(cmd, "pts", in, out, 0, 0, 0, 1)) {
    out.pop().from_vector_container(fem->node_convex().points());
  } else if (check_cmd(cmd, "is_equivalent", in, out, 0, 0, 0, 1)) {
    out.pop().from_scalar(fem->is_equivalent());
  } else if (check_cmd(cmd, "is_lagrange", in, out, 0, 0, 0, 1)) {
    out.pop().from_scalar(fem->is_lagrange());
  } else if (check_cmd(cmd, "is_polynomial", in, out, 0, 0, 0, 1)) {
    out.pop().from_scalar(fem->is_polynomial());
  } else if (check_cmd(cmd, "estimated_degree", in, out, 0, 0, 0, 1)) {
    out.pop().from_scalar(fem->estimated_degree());
  } else if (check_cmd(cmd, "base_value", in, out, 1, 1, 0, 1)) {
    getfem::base_tensor t;
    getfem::base_node x = in.pop().to_base_node(fem->dim());
    fem->base_value(x,t);
    out.pop().from_tensor(t);
  } else if (check_cmd(cmd, "grad_base_value", in, out, 1, 1, 0, 1)) {
    getfem::base_tensor t;
    getfem::base_node x = in.pop().to_base_node(fem->dim());
    fem->grad_base_value(x,t);
    out.pop().from_tensor(t);
  } else if (check_cmd(cmd, "hess_base_value", in, out, 1, 1, 0, 1)) {
    getfem::base_tensor t;
    getfem::base_node x = in.pop().to_base_node(fem->dim());
    fem->hess_base_value(x,t);
    out.pop().from_tensor(t);
  } else if (check_cmd(cmd, "char", in, out, 0, 0, 0, 1)) {
    std::string s = getfem::name_of_fem(fem);    
    out.pop().from_string(s.c_str());
  } else bad_cmd(cmd);
}

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  catch_errors(nlhs, plhs, nrhs, prhs, gf_fem_get);
}
