#include <matlabint_misc.h>
#include <getfem_derivatives.h>
#include <getfem_export.h>
#include <getfem_assembling.h>
#include <bgeot_convex_hull.h>
using namespace matlabint;

static void
error_for_non_lagrange_elements(getfem::mesh_fem &mf, bool warning_only = false)
{
  dal::bit_vector nn = mf.linked_mesh().convex_index();

  size_type cv, cnt=0, total=0, cnt_no_fem=0;
  for (cv << nn; cv != getfem::ST_NIL; cv << nn) {
    if (!mf.convex_index()[cv]) cnt_no_fem++;
    else if (!mf.fem_of_element(cv)->is_lagrange()) cnt++; 
    total++;
  }
  if (cnt) {
    if (!warning_only) {
      THROW_ERROR( "Error: " << cnt << " elements on " << total << " are NOT lagrange elements -- Unable to compute a derivative");
    } else {
      mexPrintf("Warning: %d elements on %d are NOT lagrange elements\n", cnt, total);
    }
  }
  if (cnt_no_fem) {
    if (!warning_only) {
      THROW_ERROR( "Error: " << cnt_no_fem << " elements on " << total << " have NO FEM!");
    } else {
      mexPrintf("Warning: %d elements on %d have NO FEM", cnt_no_fem, total);
    }
  }
}



void
mesh_edges_deformation(getfem::mesh_fem *mf, mlab_vect &U, unsigned N, 
		       mexargs_in &in, mexargs_out &out)
{
  size_type mdim = mf->linked_mesh().dim();
  if (mf->get_qdim() != mdim) {
    THROW_BADARG( "Error, the mesh is of dimension " << mdim << 
		  " while its Qdim is " << mf->get_qdim());
  }
  bgeot::edge_list el;
  getfem::getfem_mesh &m = mf->linked_mesh();

  build_edge_list(m, el, in);
  
  mlab_vect w   = out.pop().create_vector(mdim, N, el.size());

  bgeot::edge_list::const_iterator it = el.begin();
  size_type nbe = 0;
  while (it != el.end()) {
    bgeot::edge_list::const_iterator nit = it;
    size_type ecnt = 0;

    /* count consecutives edges of one convex */
    while (nit != el.end() && (*nit).cv == (*it).cv) {
      ecnt++; nit++;
    }
    size_type cv = (*it).cv;
    check_cv_fem(*mf, cv);

    bgeot::pgeometric_trans pgt = m.trans_of_convex(cv);

    bgeot::vsvector<getfem::base_node> pt(ecnt * N);

    /* for every edge of the convex, push the points of its refined edge
       on the vector 'pt' */
    size_type pcnt = 0;
    for (bgeot::edge_list::const_iterator eit = it; eit != nit; eit++) {
      /* build the list of points on the edge, on the reference element */
      /* get local point numbers in the convex */
      bgeot::size_type iA = m.local_ind_of_convex_point(cv, (*eit).i);
      bgeot::size_type iB = m.local_ind_of_convex_point(cv, (*eit).j);
      
      getfem::base_node A = pgt->convex_ref()->points()[iA];
      getfem::base_node B = pgt->convex_ref()->points()[iB];
      for (size_type i = 0; i < N; i++) {
	pt[pcnt++] = A +  (B-A)*(i/(double)(N-1));
      }
    }
    if (pcnt != ecnt * N) THROW_INTERNAL_ERROR;

    /* now, evaluate the field U on every point of pt  once */
    getfem::base_matrix pt_val;
    interpolate_on_convex_ref(mf, cv, pt, U, pt_val);

    if (pt_val.size() != ecnt * N * mdim) THROW_INTERNAL_ERROR;

    /* evaluate the point location on the real mesh, adds it 
       the 'deformation' field pt_val interpolated from U,
       and write the result in the destination vector */
    for (ecnt = 0; it != nit; it++, ecnt++, nbe++) {
      for (size_type i = 0; i < N; i++) {
	size_type ii = ecnt*N+i;
	getfem::base_node def_pt = pgt->transform(pt[ii], m.points_of_convex(cv));
	for (size_type k = 0; k < mdim; k++) {
	  def_pt[k] += pt_val(k,ii);
	}
	std::copy(def_pt.begin(), def_pt.end(), &w(0, i, nbe));
      }
    }
  }
}


/*MLABCOM
  FUNCTION [x] = gf_compute(meshfem MF, vec U, operation [, args])

  Various computations involving the solution U of the finite element problem.

  * N = gf_compute(MF, U, 'L2 norm' [,CVLST])
  Computes the L2 norm of U. If CVLST is indicated, the norm will be
  computed only on the listed convexes.

  * N = gf_compute(MF, U, 'H1 semi norm' [,CVLST])
  Computes the L2 norm of grad(U).

  * N = gf_compute(MF, U, 'H1 norm' [,CVLST])
  Computes the H1 norm of U.

  * DU = gf_compute(MF, U, 'gradient', mesh_fem MFGRAD)

  Computes the gradient of the field U defined on meshfem MF. The
  gradient is interpolated on the meshfem MFGRAD, and returned in DU.
  For example, if U is defined on a P2 mesh_fem, DU should be
  evaluated on a P1-discontinuous mesh_fem. MF and MFGRAD should share
  the same mesh. If they also have the same Qdim, then
  size(DU)==mdim*nbdof(MFGRAD), with mdim being the dimension of the
  common mesh. But if qdim(MFGRAD)==1 and qdim(mfu)~=1, then DU is given
  as a 3D array of dimensions mdim*qdim(MF)*nbdof(MFGRAD).

  * U2 = gf_compute(MF, U, 'interpolate on', MF2)
  Interpolates a field defined on mesh_fem MF on another (lagrangian)
  mesh_fem MF2. If MF and MF2 share the same mesh object, the 
  interpolation will be much faster.


  * [U2[,MF2,[,X[,Y[,Z]]]]] = gf_compute(MF,U,'interpolate on Q1 grid', 
                               {'regular h', hxyz | 'regular N',Nxyz |
           			   X[,Y[,Z]]}

  Creates a cartesian Q1 mesh fem and interpolates U on it. The
  returned field U2 is organized in a matrix such that in can be drawn
  via the MATLAB command 'pcolor'. The first dimension is the Qdim of
  MF (i.e.  1 if U is a scalar field)

  example (mf_u is a 2D mesh_fem): 
   >> Uq=gf_compute(mf_u, U, 'interpolate on Q1 grid', 'regular h', [.05, .05]);
   >> pcolor(squeeze(Uq(1,:,:)));

  * E = gf_compute(MF, U, 'mesh edges deformation', N [,vec or 
                   mat CVLIST])

  Evaluates the deformation of the mesh caused by the field U (for a
  2D mesh, U must be a [2 x nb_dof] matrix). N is the refinment level
  (N>=2) of the edges.  CVLIST can be used to restrict the computation
  to the edges of the listed convexes ( if CVLIST is a row vector ),
  or to restrict the computations to certain faces of certain convexes
  when CVLIST is a two-rows matrix, the first row containing convex
  numbers and the second face numbers.

  * UP = gf_compute(MF, U, 'eval on triangulated surface', int Nrefine,
                    [vec CVLIST])
  Utility function designed for 2D triangular meshes : returns a list
  of triangles coordinates with interpolated U values. This can be
  used for the accurate visualization of data defined on a
  discontinous high order element. On output, the six first rows of UP
  contains the triangle coordinates, and the others rows contain the
  interpolated values of U (one for each triangle vertex) CVLIST may
  indicate the list of convex number that should be consider, if not
  used then all the mesh convexes will be used. U should be a row
  vector.

  $Id: gf_compute.C,v 1.21 2003/02/19 18:29:15 pommier Exp $
MLABCOM*/

void gf_compute(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 3) {
    THROW_BADARG( "Wrong number of input arguments");
  }

  getfem::mesh_fem *mf   = in.pop().to_mesh_fem();
  mlab_vect U            = in.pop().to_scalar_vector(-1, mf->nb_dof());
  std::string cmd        = in.pop().to_string();

  if (check_cmd(cmd, "L2 norm", in, out, 0, 1, 0, 1)) {
    dal::bit_vector bv = in.remaining() ? 
      in.pop().to_bit_vector(&mf->convex_index()) : mf->convex_index();
    out.pop().from_scalar(getfem::asm_L2_norm(*mf, U, bv));
  } else if (check_cmd(cmd, "H1 semi norm", in, out, 0, 1, 0, 1)) {
    dal::bit_vector bv = in.remaining() ? 
      in.pop().to_bit_vector(&mf->convex_index()) : mf->convex_index();
    out.pop().from_scalar(getfem::asm_H1_semi_norm(*mf, U, bv));
  } else if (check_cmd(cmd, "H1 norm", in, out, 0, 1, 0, 1)) {
    dal::bit_vector bv = in.remaining() ? 
      in.pop().to_bit_vector(&mf->convex_index()) : mf->convex_index();
    out.pop().from_scalar(getfem::asm_H1_norm(*mf, U, bv));
  } else if (check_cmd(cmd, "gradient", in, out, 1, 1, 0, 1)) {
    getfem::mesh_fem *mf_grad = in.pop().to_mesh_fem();
    error_for_non_lagrange_elements(*mf_grad);
    size_type qm = (mf_grad->get_qdim() == mf->get_qdim()) ? 1 : mf->get_qdim();
    mlab_vect DU;
    if (qm == 1) {
      DU = out.pop().create_vector(mf->linked_mesh().dim(), mf_grad->nb_dof());
    } else {
      DU = out.pop().create_vector(mf->linked_mesh().dim(), qm, mf_grad->nb_dof());
    }
    /* compute_gradient also check that the meshes are the same */
    getfem::compute_gradient(*mf, *mf_grad, U, DU);
  } else if (check_cmd(cmd, "eval on triangulated surface", in, out, 1, 2, 0, 1)) {
    int Nrefine = in.pop().to_integer(1, 1000);
    std::vector<convex_face> cvf;
    if (in.remaining() && !in.front().is_string()) {
      mlab_vect v = in.pop().to_int_vector(-1, -1);
      build_convex_face_lst(mf->linked_mesh(), cvf, &v);
    } else build_convex_face_lst(mf->linked_mesh(), cvf, 0);
    if (U.getn() != mf->nb_dof()) {
      THROW_BADARG("Wrong number of columns (need transpose?)");
    }  
    eval_on_triangulated_surface(&mf->linked_mesh(), Nrefine, cvf, out, mf, U);
  } else if (check_cmd(cmd, "interpolate on", in, out, 1, 1, 0, 1)) {
    getfem::mesh_fem *mf_dest = in.pop().to_mesh_fem();
    error_for_non_lagrange_elements(*mf_dest, true);
    mlab_vect U2 = out.pop().create_vector(1, mf_dest->nb_dof());
    getfem::interpolation_solution(*mf, *mf_dest,
				   U, U2);
  } else if (check_cmd(cmd, "interpolate on Q1 grid", in, out, 1, 100, 0, 100)) {
    in.restore(0); in.restore(1); /* push back the mf_u and u */
    call_matlab_function("gf_compute_Q1grid_interp", in,out);
  } else if (check_cmd(cmd, "mesh edges deformation", in, out, 1, 2, 0, 1)) {
    unsigned N = in.pop().to_integer(2,10000);
    mesh_edges_deformation(mf, U, N, in, out);
  } else  bad_cmd(cmd);
}

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  catch_errors(nlhs, plhs, nrhs, prhs, gf_compute);
}
