/* -*- c++ -*- (enables emacs c++ mode)                                    */
/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Interface (matlabint)                                  */
/* File    : matlabint.h : global definitions for Matlab interface.        */
/*     									   */
/*                                                                         */
/* Date : September 26, 2001.                                              */
/* Author : Yves Renard, Yves.Renard@gmm.insa-tlse.fr                      */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */


#ifndef __MATLABINT_H
#define __MATLABINT_H

#include <matlabint_std.h>
#include <getfem_mesh.h>
#include <getfem_mesh_fem.h>
#include <bgeot_smatrix.h>


namespace matlabint
{
  typedef bgeot::smatrix<getfem::scalar_type> gf_sparse_matrix;

  class matlabint_mesh;

  /* fake class for storing the dimension of unkowns with the mesh_fem
     object */
  class mesh_fem_int : public getfem::mesh_fem {
    int u_dim;
  public:
    mesh_fem_int(getfem::getfem_mesh &m, unsigned _u_dim) : getfem::mesh_fem(m), u_dim(_u_dim) {}
    unsigned get_u_dim() const { return u_dim; }
  };



  /* fake vector class for matlab arrays (necessary for template functions) */
  class mlab_vect {
    unsigned sz;
    double *data;
    unsigned m,n,p; /* original number of rows & columns of matlab array 
		       p is a pseudo third dimension
		    */
  public:
    double& operator[](unsigned i) { if (i >= sz) THROW_INTERNAL_ERROR; return data[i]; }
    const double& operator[](unsigned i) const { if (i >= sz) THROW_INTERNAL_ERROR; return data[i]; }
    double& operator()(unsigned i, unsigned j, unsigned k=0) { if (i+j*m+k*m*n >= sz) THROW_INTERNAL_ERROR; 
    return data[i+j*m+k*m*n]; }
    const double& operator()(unsigned i, unsigned j, unsigned k=0) const { if (i+j*m+k*m*n >= sz) THROW_INTERNAL_ERROR;return data[i+j*m+k*m*n]; }

    mlab_vect(): sz(0),data(0),m(0),n(0),p(0) {}
    mlab_vect(const mxArray *mx) { 
      if (!mxIsDouble(mx)) THROW_INTERNAL_ERROR;
      data = mxGetPr(mx); sz = mxGetNumberOfElements(mx); 
      int ndim = mxGetNumberOfDimensions(mx);
      if (ndim < 3) {
	m = mxGetM(mx); n = mxGetN(mx); p = 1;
      } else if (ndim == 3) {
	const int *dim = mxGetDimensions(mx);
	m = dim[0]; n = dim[1]; p = dim[2];
      } else THROW_INTERNAL_ERROR;
      if (n*m*p != sz) THROW_INTERNAL_ERROR;
    }
    mlab_vect(double *v, int _sz) { data = v; sz = _sz; m=_sz; n=1; p = sz ? sz/(m*n) : 0; }
    unsigned size() const { return sz; }
    unsigned getm() const { return m; }
    unsigned getn() const { return n; }
    unsigned getp() const { return p; }

    void reshape(unsigned _n, unsigned _m, unsigned _p=1) {
      if (sz != _n*_m*_p) THROW_INTERNAL_ERROR;
      n = _n; m = _m; p = _p;
    }

    getfem::base_node col_to_bn(unsigned j, unsigned k=0) const {
      getfem::base_node P(m);
      for (unsigned i=0; i < m; i++) P[i] = operator()(i,j,k);
      return P;
    }
    getfem::base_matrix row_col_to_bm(unsigned k=0) const {
      getfem::base_matrix M(m,n);
      for (unsigned i=0; i < m; i++) 
	for (unsigned j=0; j < n; j++) M(i,j) = operator()(i,j,k);
      return M;
    }
    bool in_range(double vmin, double vmax);
  };

  class mexarg_in {
    void check_int_values(int vmin=INT_MIN, int vmax=INT_MAX);
  public:
    const mxArray *arg;
    int argnum;

    mexarg_in(const mxArray *_arg, int _num) { arg = _arg; argnum = _num; }
    bool                        is_string() { return (mxIsChar(arg) == 1 && mxGetM(arg) == 1); }
    int                         to_integer(int min_val=INT_MIN, int max_val=INT_MAX);
    double                      to_scalar();
    std::string                 to_string();
    id_type                     to_object_id();
    dal::bit_vector *           to_intset();
    bgeot::base_poly *          to_poly();
    mesh_fem_int *              to_mesh_fem();
    const getfem::getfem_mesh * to_const_mesh();
    matlabint_mesh *            to_matlabint_mesh();
    getfem::getfem_mesh *       to_mesh();
    getfem::pintegration_method to_fem_interpolation();
    getfem::pfem                to_fem();
    getfem::pmat_elem_type      to_mat_elem_type();
    bgeot::pconvex_structure    to_cvs();
    bgeot::pgeometric_trans     to_pgt();

    /* do not perform any check on the number of dimensions of the matlab array */
    mlab_vect                   to_scalar_vector();

    /* expect the argument to be a row or column vector of given dimension */
    mlab_vect                   to_scalar_vector(int expected_dim);

    /* expect the argument to be a matrix (or possibly a 3D array)
       if any of the arguments has a value of -1, the corresponding dimension
       is not checked
     */
    mlab_vect                   to_scalar_vector(int expected_n, int expected_m, int expected_k=1);
    mlab_vect                   to_int_vector();
    mlab_vect                   to_int_vector(int expected_dim);
    mlab_vect                   to_int_vector(int expected_n, int expected_m, int expected_k=1);

    dal::bit_vector             to_bit_vector(const dal::bit_vector *subsetof = NULL, int shiftvals=-1);
  };

  class mexarg_out {
  public:
    mxArray *& arg;
    int argnum;
    mexarg_out(mxArray * &_arg, int _num) : arg(_arg), argnum(_num) {}

    void from_object_id(id_type id);
    void from_integer(int i);
    void from_scalar(double v);
    void from_bit_vector(const dal::bit_vector& bv, int shift=+1);
    void from_sparse(const gf_sparse_matrix& M);
    void from_tensor(const getfem::base_tensor& t);
    mlab_vect create_vector(unsigned dim);
    mlab_vect create_vector(unsigned n, unsigned m); 
    mlab_vect create_vector(unsigned n, unsigned m, unsigned p); 
  };


  /* handles the list of input arguments */
  class mexargs_in {
    const mxArray **in;
    dal::bit_vector idx;
    int nb_arg;
    bool use_cell;
    mexargs_in(const mexargs_in& );
    mexargs_in& operator=(const mexargs_in& );
  public:
    mexargs_in(int n, const mxArray *p[], bool use_cell);
    ~mexargs_in();
    void check() const { if (idx.card() == 0) THROW_INTERNAL_ERROR; }
    const mxArray *pop_mx(size_type decal=0, int *out_idx = NULL) {
      size_type i = idx.first_true(); 
      check();
      if (decal >= idx.card()) THROW_INTERNAL_ERROR;
      while (decal>0) { i++; check(); if (idx.is_in(i)) decal--; }
      idx.sup(i); 
      if (out_idx) *out_idx = i;
      return in[i];
    }
    mexarg_in pop(size_type decal=0) { 
      int i;
      const mxArray *m = pop_mx(decal, &i);
     return mexarg_in(m,i+1); 
    }
    void restore(size_type i) { idx.add(i); }
    mexarg_in front() const { check(); return mexarg_in(in[idx.first_true()],idx.first_true()); }
    int narg() const { return nb_arg; }
    int remaining() const { return idx.card(); }
    void get_array(const mxArray **& m) {
      if (remaining()) {
	m = new const mxArray *[remaining()];
	for (size_type i=0; remaining(); i++) {
	  m[i] = pop_mx();
	}
      } else m = NULL;
    }
    void pop_all() { idx.clear(); }
  };

  /* handles the list of output arguments */
  class mexargs_out {
    mxArray **out;
    int nb_arg;
    int idx;
    bool use_cell;
    mxArray **pout_cell;

    mexargs_out(const mexargs_out& );
    mexargs_out& operator=(const mexargs_out& );
  public:
    mexargs_out(int _nb_arg, mxArray *p[], bool _use_cell);
    ~mexargs_out();
    void check() const { if ((idx >= nb_arg || idx < 0) && !(idx==0)) THROW_INTERNAL_ERROR; }
    mexarg_out pop() { check(); idx++; return mexarg_out(out[idx-1], idx); }
    mexarg_out front() const { check(); return mexarg_out(out[idx], idx+1); }
    int narg() const { return nb_arg; }
    int remaining() const { return std::max(nb_arg,1) - idx; }
    bool fixed_size() const { return !use_cell; }
    mxArray **get_array() const { return out+idx; }
    void pop_all() { idx = nb_arg; }
  };

  bool check_cmd(std::string& cmdname, const char *s, 
		 const mexargs_in& in, const mexargs_out& out,
		 int min_argin=0, int max_argin=-1,
		 int min_argout=0, int max_argout=-1);
  
  inline void bad_cmd(std::string& cmd) { 
    DAL_THROW(matlabint_bad_arg, "Bad command name: " << cmd); }

  typedef void (*mex_func_type)(mexargs_in&, mexargs_out&);

  void catch_errors(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[],
		    mex_func_type pf, const char *local_workspace=NULL);
  void call_matlab_function(const std::string& fname, mexargs_in& in, mexargs_out& out);

}  /* end of namespace matlabint.                                          */

#endif /* __MATLABINT_H                                                    */
