#include <matlabint_misc.h>
#include <matlabint_mesh.h>

using namespace matlabint;

void check_empty_mesh(const getfem::getfem_mesh *pmesh)
{
  if (pmesh->dim() == bgeot::dim_type(-1) || pmesh->dim() == 0) {
    DAL_THROW(matlabint_error, "mesh object has an invalid dimension");
  }
}

/*MLABCOM
  FUNCTION [x] = gf_mesh_set(mesh M, operation [, args])

    General function for modification of a mesh object.

    * IDX = gf_mesh_set(M, 'add point', PT)

    Inserts new points in the mesh. PT should be an [n x m] matrix ,
    where n is the mesh dimension, and m is the number of points that
    will be added to the mesh. On output, IDX contains the indices 
    of these new points.

    Remark: if some points are already part of the mesh, they won't
    be inserted again, be IDX will contains the previously assigned
    indices of the points.

    * gf_mesh_set(M, 'del point', IDX)

    Remove one or more points from the mesh. IDX should contain the 
    point indexes, such as the one returned bu the 'add point' command.

    * IDX = gf_mesh_set(M, 'add convex', cvstruct CVS, mat CVPTS)

    Add a new convex of structure CVS, and whose points are given by the
    columns of CVPTS. On return, IDX contains the convex number.
    CVPTS might be a three dimensionnal array in order to insert more
    than one convex (or a two dimensional array correctly shaped).

    * gf_mesh_set(M, 'del convex', IDX)

    Remove one or more convexes from the mesh. IDX should contain the 
    convexes indexes, such as the ones returned bu the 'add convex' command.

    * gf_mesh_set(M, 'translate', vec V)
    Translates each point of the mesh from V.

    * gf_mesh_set(M, 'transform', mat M)
    Applies the matrix M to each point of the mesh.

  MLABCOM*/
void gf_mesh_set(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 2) {
    DAL_THROW(matlabint_bad_arg, "Wrong number of input arguments");
  }

  getfem::getfem_mesh *pmesh = in.pop().to_mesh();
  std::string cmd            = in.pop().to_string();
  if (check_cmd(cmd, "add point", in, out, 1, 1, 0, 1)) {
    check_empty_mesh(pmesh);
    mlab_vect v = in.pop().to_scalar_vector(pmesh->dim(), -1);
    mlab_vect w = out.pop().create_vector(1, v.getn());
    for (size_type j=0; j < v.getn(); j++) {
      w[j] = (double) pmesh->add_point(v.col_to_bn(j)) + 1;
    }
  } else if (check_cmd(cmd, "del point", in, out, 1, 1, 0, 0)) {
    check_empty_mesh(pmesh);
    mlab_vect v = in.pop().to_int_vector();
    
    for (size_type j=0; j < v.size(); j++) {
      id_type id = (int)v[j]-1;
      if (pmesh->point_is_valid(id)) {
	DAL_THROW(matlabint_error, "Can't remove point " << id+1 
		  << ": a convex is still attached to it.");
      }
      pmesh->sup_point(id);
    }
  } else if (check_cmd(cmd, "add convex", in, out, 1, 1, 0, 1)) {
    check_empty_mesh(pmesh);
    bgeot::pgeometric_trans pgt = in.pop().to_pgt();
    mlab_vect v = in.pop().to_scalar_vector(pmesh->dim(), pgt->nb_points(), -1);
    mlab_vect w = out.pop().create_vector(1, v.getp());

    std::vector<matlabint::id_type> qp(pgt->nb_points());
    /* loop over convexes */
    for (size_type k=0; k < v.getp(); k++) {
      /* loop over convex points */
      for (size_type j=0; j < v.getn(); j++) {
	qp[j] = pmesh->add_point(v.col_to_bn(j,k));
      }
      id_type cv_id = pmesh->add_convex(pgt, qp.begin());
      w[k] = (double)(cv_id+1);
    }
  } else if (check_cmd(cmd, "del convex", in, out, 1, 1, 0, 0)) {
    check_empty_mesh(pmesh);
    mlab_vect v = in.pop().to_int_vector();
    
    for (size_type j=0; j < v.size(); j++) {
      id_type id = (int)v[j]-1;
      pmesh->sup_convex(id-1);
    }
  } else if (check_cmd(cmd, "translate", in, out, 1, 1, 0, 0)) {
    check_empty_mesh(pmesh);
    mlab_vect v = in.pop().to_scalar_vector(pmesh->dim(),1);
    pmesh->translation(v.col_to_bn(0));
  } else if (check_cmd(cmd, "transform", in, out, 1, 1, 0, 0)) {
    check_empty_mesh(pmesh);
    mlab_vect v = in.pop().to_scalar_vector(pmesh->dim(),pmesh->dim());
    pmesh->transformation(v.row_col_to_bm());
  } else bad_cmd(cmd);
}

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  catch_errors(nlhs, plhs, nrhs, prhs, gf_mesh_set);
}
