#include <matlabint_misc.h>
#include <matlabint_mesh.h>
#include <getfem_export.h>

using namespace matlabint;

void check_empty_mesh(const getfem::getfem_mesh *pmesh)
{
  if (pmesh->dim() == bgeot::dim_type(-1) || pmesh->dim() == 0) {
    DAL_THROW(matlabint_error, "mesh object has an invalid dimension");
  }
}




void
transform_edge_list(const getfem::getfem_mesh &m, unsigned N, const getfem::edge_list &el, mlab_vect &w)
{
  getfem::edge_list::const_iterator it;
  size_type cv = size_type(-1);
  getfem::getfem_mesh::ref_convex cv_ref;
  bgeot::pgeometric_trans pgt = NULL;
  size_type ecnt = 0;
  for (it = el.begin(); it != el.end(); it++, ecnt++) {
    if (cv != (*it).cv) {
      cv = (*it).cv;
      cv_ref = m.convex(cv);
      pgt = m.trans_of_convex(cv);
    }
    /* build the list of points on the edge, on the reference element */
    /* get local point numbers in the convex */
    bgeot::size_type iA = m.local_ind_of_convex_point(cv, (*it).i);
    bgeot::size_type iB = m.local_ind_of_convex_point(cv, (*it).j);

    if (iA >= cv_ref.nb_points() ||
	iB >= cv_ref.nb_points() || iA == iB) {
      mexPrintf("iA=%d , iB=%d\n",iA,iB);
      THROW_INTERNAL_ERROR;
    }
    getfem::base_node A = cv_ref.points()[iA];
    getfem::base_node B = cv_ref.points()[iB];
    for (size_type i = 0; i < N; i++) {
      getfem::base_node pt;
      
      pt = A +  (B-A)*(i/(double)(N-1));

      pt = pgt->transform(pt, m.points_of_convex(cv));

      std::copy(pt.begin(), pt.end(), &w(0,i, ecnt));
    }
  }
}

struct mesh_faces_list_elt  {
  size_type cv;
  size_type f;
  inline bool operator < (const mesh_faces_list_elt &e) const
  {
    if (cv < e.cv) return true; if (cv > e.cv) return false; 
    if (f < e.f) return true; else if (f > e.f) return false;
    return false;
  }
  mesh_faces_list_elt(size_type _cv, size_type _f) : cv(_cv), f(_f) {}
  mesh_faces_list_elt() {}
};

typedef dal::dynamic_tree_sorted<mesh_faces_list_elt> mesh_faces_list;

void
faces_from_pid(const getfem::getfem_mesh &m, mexargs_in &in, mexargs_out &out)
{
  dal::bit_vector pts = in.pop().to_bit_vector(&m.points().index());
  mesh_faces_list lst;
  dal::bit_vector convex_tested;

  for (dal::bit_vector::const_iterator it_pts = pts.begin(); it_pts != pts.end(); ++it_pts) {
    if (!(*it_pts)) continue;

    size_type ip = it_pts.index();
    /* iterator over all convexes attached to point ip */
    bgeot::mesh_convex_ind_ct::const_iterator 
      cvit = m.convex_to_point(ip).begin(),
      cvit_end = m.convex_to_point(ip).end();

    //    cerr << "point " << ip+1 << endl;

    for ( ; cvit != cvit_end; cvit++) {
      size_type ic = *cvit;
      if (!convex_tested.is_in(ic)) {
	convex_tested.add(ic);
	/* loop over the convex faces */
	for (size_type f = 0; f < m.structure_of_convex(ic)->nb_faces(); f++) {
	  bgeot::ind_ref_mesh_point_ind_ct pt = m.ind_points_of_face_of_convex(ic, f);
	  bool ok = true;
	  for (bgeot::ind_ref_mesh_point_ind_ct::const_iterator it = pt.begin(); it != pt.end(); it++) {
	    //	    cerr << "testing cv=" << ic+1 << " face=" << f+1 << " point[" << (*it)+1 << "] ? " << pts.is_in(*it) << endl;
	    if (!pts.is_in(*it)) { 
	      ok = false; break; 
	    }
	  }
	  if (ok) {
	    lst.add(mesh_faces_list_elt(ic,f));
	    //	    cerr << "added!\n";
	  }
	}
      }
    }
  }
  mlab_vect w = out.pop().create_vector(2,lst.size());
  for (size_type j=0; j < lst.size(); j++) { w(0,j) = lst[j].cv+1; w(1,j) = lst[j].f+1; }
}


struct mesh_faces_by_pts_list_elt  {
  std::vector<size_type> ptid; // point numbers of faces
  int cnt; // number of convexes sharing that face
  int cv, f;
  inline bool operator < (const mesh_faces_by_pts_list_elt &e) const
  {
//     cerr << " p1 = " << ptid[0] << " " << ptid[1] << " p2=" << e.ptid[0] << " " << e.ptid[1] << endl;
//     cerr << " p1<p2 ? " << ((ptid < e.ptid) ? true : false) << endl;
    if (ptid.size() < e.ptid.size()) return true;
    if (ptid.size() > e.ptid.size()) return false;
    return ptid < e.ptid;
  }

  mesh_faces_by_pts_list_elt(size_type _cv, size_type _f, std::vector<size_type>& p) : cv(_cv), f(_f) {
    cnt = 0;
    if (p.size() == 0) THROW_INTERNAL_ERROR;
    std::sort(p.begin(), p.end());

//     cerr << "ajout cv=" << cv << " f=" << f << " pt=["; 
//     std::copy(p.begin(), p.end(), std::ostream_iterator<size_type,char>(cerr, ","));
//     cerr << "]\n";
    ptid = p; 
  }
  mesh_faces_by_pts_list_elt() {}
};
typedef dal::dynamic_tree_sorted<mesh_faces_by_pts_list_elt> mesh_faces_by_pts_list;


void
outer_faces(const getfem::getfem_mesh &m, mexargs_in &in, mexargs_out &out)
{
  mesh_faces_by_pts_list lst;
  dal::bit_vector convex_tested;
  dal::bit_vector cvlst;

  if (in.remaining()) cvlst = in.pop().to_bit_vector(&m.convex_index());
  else cvlst = m.convex_index();
  
  for (dal::bit_vector::const_iterator it_cv = cvlst.begin(); it_cv != cvlst.end(); ++it_cv) {
    if (!(*it_cv)) continue;

    size_type ic = it_cv.index();

    for (size_type f = 0; f < m.structure_of_convex(ic)->nb_faces(); f++) {
      bgeot::ind_ref_mesh_point_ind_ct pt = m.ind_points_of_face_of_convex(ic, f);
      

      std::vector<size_type> p(pt.begin(), pt.end());
      size_type idx = lst.add_norepeat(mesh_faces_by_pts_list_elt(ic,f,p));
//       cerr << " <-- idx = " << idx << endl;
      lst[idx].cnt++;
    }
  }
  size_type fcnt = 0;
  for (size_type i = 0; i < lst.size(); i++) {
    fcnt += (lst[i].cnt == 1 ? 1 : 0);
  }

  mlab_vect w = out.pop().create_vector(2,fcnt);
  for (size_type j=0, fcnt=0; j < lst.size(); j++) { 
    if (lst[j].cnt == 1) {
      w(0,fcnt) = lst[j].cv+1; 
      w(1,fcnt) = lst[j].f+1; 
      fcnt++;
    }
  }
}

/*MLABCOM

  FUNCTION [...] = gf_mesh_get(mesh M, [operation [, args]])

  General mesh inquiry function. All these functions accept also a 
  mesh_fem argument instead of a mesh M (in that case, the mesh_fem
  linked mesh will be used).

  * I = gf_mesh_get(M, 'dim')
  Returns the dimension of the mesh (2 for a 2D mesh, etc..)

  * I = gf_mesh_get(M, 'nbpts')
  Returns the number of points of the mesh.

  * I = gf_mesh_get(M, 'nbcvs')
  Returns the number of convexes of the mesh.

  * [PT] = gf_mesh_get(M, 'pts coords')

  Return the list of points of mesh M, each point is stored in a
  column of PT. CAUTION: if you destroyed some convexes or points in
  the mesh, the columns correponding to these destroyed points will be
  filled with NaN. You can use 'gf_mesh_get(M, 'pts id')' to filter
  such invalidated points.

  * [PTID] = gf_mesh_get(M, 'pid')
  Return the list of points numbers composing M (their numbering is
  not supposed to be contiguous from 1 to gf_mesh_get('nbpts'),
  especially if you destroyed some convexes).

  * [CVID] = gf_mesh_get(M, 'cvid')
  Return the list of convex numbers composing M (their numbering is
  not supposed to be contiguous from 1 to gf_mesh_get('nbcvs'),
  especially if you destroyed some convexes).

  * [CVLST] = gf_mesh_get(M, 'cvs')
  Returns the list of convexes numbers in the first row of CVLST. On the second
  row is the number of faces of the convex. On the third row is the number of
  points of the convex.

  * I = gf_mesh_get(M, 'max pid')
  Returns the maximum ID of all points in the mesh (this is the same
  value as MAX(gf_mesh_get(M, 'pts id')), but it won't be equal to 
  gf_mesh_get(M, 'nbpts') if some points have been destroyed and the
  mesh was not 'repacked'.

  * I = gf_mesh_get(M, 'max cvid')
  Returns the maximum ID of all convexes in the mesh (this is the same
  value as MAX((gf_mesh_get(M, 'cvs'))(1,:)), but it won't be equal to 
  gf_mesh_get(M, 'nbcvs') if some convexes have been destroyed and the
  mesh was not 'repacked'.

  * [IDX,PID] = gf_mesh_get(M, 'pid from cv'[,CVLST])
  IDX is a row vector, length(IDX) = length(CVLIST)+1. PID is a row
  vector containing the concatenated list of points of each convex in
  cvlst. Each entry of IDX is the position of the corresponding convex
  point list in PID. Hence, for example, the list of points of the
  second convex is PID(IDX(2):IDX(3)-1).

  If you specified convex numbers with do not exist in CVLST, their
  point list will be empty.

  A good way to use this function in order to quickly find the point
  list of any convex is to call
     [IDX,PID] = gf_mesh_get(M, 'pid from cv', 1:gf_mesh_get(M,'max cvid'));


  * [V] = gf_mesh_get(M, 'pid from coords', mat PT)
  PT is an array containing a list of point coordinates. On return, V is a row
  vector containing the mesh id of the points which are part of the mesh, and
  -1 for those which where not found in the mesh.

  * [V] = gf_mesh_get(M, 'cvid from pid', vec PTID)
  Returns the list of convexes that share the points numbers given in PTID in a
  row vector (possibly empty).


  * [V] = gf_mesh_get(M, 'faces from pid', vec PTID)
  Returns the list of convexes faces of which every vertex is in PTID.
  On return, the first row of V contains the convex number, and the
  second row contains the face number.

  * CVFACELST = gf_mesh_get(M, 'outer faces' [, CVLIST])
  Returns the list of faces with are not shared by two convexes (i.e. the
  face on the boundary of the mesh). The search can be restricted to the
  optionnal argument CVLIST.

  * [E] = gf_mesh_get(M, 'edges' [, CVLIST][,'merge convex'])

  Return the list of edges of mesh M for the convexes listed in the
  row vector CVLIST. E is a 3 x nb_edges matrix containing point
  indices in its two first rows, and convex number in the third
  row. If CVLIST is omitted, then the edges of all convexes are
  returned. If CVLIST has two rows then the first row is supposed to
  contain convex numbers, and the second face numbers, of which the
  edges will be returned.  If 'merge convex' is indicated, all common
  edges of convexes are merged in a single edge.


  * [E] = gf_mesh_get(M, 'curved edges', int N, [, CVLIST])

  More sophisticated version of gf_mesh_get(M, 'edges') designed for curved 
  elements. This one will return N (N>=2) points of the (curved) edges. With N==2,
  this is equivalent to  gf_mesh_get(M, 'edges'). Since the points are no more
  always part of the mesh, their coordinates are returned instead of points number,
  in the array E which is a [ mesh_dim x 2 x nb_edges ] array.

  * [CVFACES] = gf_mesh_get(M, 'boundary faces' [, CVLIST])
  
  

  * gf_mesh_get(M, 'save', string FILENAME)
  Save the mesh object to an ascii file. This mesh can be restored with
  gf_mesh('load', FILENAME);

MLABCOM*/


void gf_mesh_get(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 2) {
    DAL_THROW(matlabint_bad_arg, "Wrong number of input arguments");
  }

  const getfem::getfem_mesh *pmesh = in.pop().to_const_mesh();
  std::string cmd                  = in.pop().to_string();
  if (check_cmd(cmd, "dim", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(pmesh->dim());
  } else if (check_cmd(cmd, "nbpts", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(pmesh->nb_points());
  } else if (check_cmd(cmd, "nbcvs", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(pmesh->nb_convex());
  } else if (check_cmd(cmd, "pts coords", in, out, 0, 0, 0, 1)) {
    dal::bit_vector bv = pmesh->points().index();
    mlab_vect w   = out.pop().create_vector(pmesh->dim(), bv.last_true()+1);
    for (size_type j = 0; j < bv.last_true()+1; j++) {
      for (size_type i = 0; i < pmesh->dim(); i++) {
	w(i,j) = (bv.is_in(j)) ? (pmesh->points()[j])[i] : NAN; /* not sure if NAN is portable .. */
      }
    }
  } else if (check_cmd(cmd, "pid", in, out, 0, 0, 0, 1)) {
    out.pop().from_bit_vector(pmesh->points().index());
  } else if (check_cmd(cmd, "cvid", in, out, 0, 1, 0, 1)) {
    out.pop().from_bit_vector(pmesh->convex_index());
  } else if (check_cmd(cmd, "cvs", in, out, 0, 1, 0, 1)) {
    dal::bit_vector cvlst;
    if (in.remaining()) cvlst = in.pop().to_bit_vector(&pmesh->convex_index());
    else cvlst = pmesh->convex_index();

    mlab_vect w   = out.pop().create_vector(3, pmesh->nb_convex());
    size_type cv, j = 0;
    for (cv << cvlst; cv != size_type(-1); cv << cvlst) {
      w(0,j) = cv+1;
      w(1,j) = pmesh->structure_of_convex(cv)->nb_faces();
      w(2,j) = pmesh->structure_of_convex(cv)->nb_points();
      j++;
    }
  } else if (check_cmd(cmd, "max pid", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(pmesh->points().index().last_true());
  } else if (check_cmd(cmd, "max cvid", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(pmesh->convex_index().last_true());
  }
 else if (check_cmd(cmd, "pid from cvid", in, out, 0, 1, 2, 2)) {
    dal::bit_vector cvlst;
    if (in.remaining()) cvlst = in.pop().to_bit_vector();
    else cvlst = pmesh->convex_index();
    
    size_type pcnt = 0;
    /* phase one: count the total number of pids */
    for (dal::bit_vector::const_iterator it = cvlst.begin();
	 it != cvlst.end(); ++it) {
      size_type cv = it.index();
      if (pmesh->convex_index().is_in(cv)) {
	pcnt += pmesh->structure_of_convex(cv)->nb_points();
      }
    }
    /* phase two: allocation */
    mlab_vect idx = out.pop().create_vector(1, cvlst.card() + 1);
    mlab_vect pid = out.pop().create_vector(1, pcnt);

    pcnt = 0;
    /* phase three: build the list */
    for (dal::bit_vector::const_iterator it = cvlst.begin();
	 it != cvlst.end(); ++it) {
      size_type cv = it.index();
      idx(0,cv) = double(pcnt+1);
      if (pmesh->convex_index().is_in(cv)) {
	for (bgeot::ref_mesh_point_ind_ct::const_iterator pit = 
	       pmesh->ind_points_of_convex(cv).begin();
	     pit != pmesh->ind_points_of_convex(cv).end(); ++pit) {
	  pid(0,pcnt++) = double((*pit) + 1);
	}
      }
    }
    idx(0,idx.getn()-1) = double(pcnt+1); /* for the last convex */
  } else if (check_cmd(cmd, "edges", in, out, 0, 2, 0, 1)) {
    getfem::edge_list el;
    build_edge_list(*pmesh, el, in);
    /* copy the edge list to the matlab array */
    mlab_vect w   = out.pop().create_vector(2, el.size());
    for (size_type j = 0; j < el.size(); j++) { 
      w(0,j) = (el[j].i + 1);
      w(1,j) = (el[j].j + 1);
    }
  } else if (check_cmd(cmd, "curved edges", in, out, 1, 2, 0, 1)) {    
    getfem::edge_list el;
    size_type N = in.pop().to_integer(2,10000);
    build_edge_list(*pmesh, el, in);
    mexPrintf("hop!\n");
    mlab_vect w   = out.pop().create_vector(pmesh->dim(), N, el.size());
    mexPrintf("hop!\n");
    transform_edge_list(*pmesh, N, el, w);
  } else if (check_cmd(cmd, "pid from coords", in, out, 1, 1, 0, 1)) {
    check_empty_mesh(pmesh);
    mlab_vect v   = in.pop().to_scalar_vector(pmesh->dim(), -1);
    mlab_vect w   = out.pop().create_vector(1, v.getn());
    for (size_type j=0; j < v.getn(); j++) {
      getfem::size_type id = pmesh->search_point(v.col_to_bn(j));
      if (id == getfem::size_type(-1)) w[j] = double(-1);
      else w[j] = double(id + 1);
    }
  } else if (check_cmd(cmd, "cvid from pid", in, out, 1, 1, 0, 1)) {
    check_empty_mesh(pmesh);
    dal::bit_vector pts = in.pop().to_bit_vector(&pmesh->points().index());
    dal::bit_vector cvchecked;

    std::vector<size_type> cvlst;

    /* loop over the points */
    for (dal::bit_vector::const_iterator it_pt = pts.begin(); it_pt != pts.end(); ++it_pt) {
      if (!*it_pt) continue;
      size_type ip = it_pt.index();

      /* iterators over the convexes attached to point ip */
      bgeot::mesh_convex_ind_ct::const_iterator 
	cvit = pmesh->convex_to_point(ip).begin(),
	cvit_end = pmesh->convex_to_point(ip).end();    

      for ( ; cvit != cvit_end; cvit++) {
	size_type ic = *cvit;

	//	cerr << "cv = " << ic+1 << endl;

	if (!cvchecked.is_in(ic)) {
	  bool ok = true;
	  
	  bgeot::ref_mesh_point_ind_ct cvpt = pmesh->ind_points_of_convex(ic);
	  /* check that each point of the convex is in the list */
	  for (bgeot::ref_mesh_point_ind_ct::const_iterator it = cvpt.begin();
	       it != cvpt.end(); ++it) {
	    //	    cerr << "  test pt " << (*it)+1 << ": is_in=" << pts.is_in(*it) << endl;
	    if (!pts.is_in(*it)) {
	      ok = false; break;
	    }
	  }
	  if (ok) cvlst.push_back(ic);
	  cvchecked.add(ic);
	}
      }
    }
    mlab_vect w = out.pop().create_vector(1,cvlst.size());
    for (size_type j=0; j < cvlst.size(); j++) w[j] = cvlst[j]+1;
  } else if (check_cmd(cmd, "faces from pid", in, out, 1, 1, 0, 1)) {
    check_empty_mesh(pmesh);
    faces_from_pid(*pmesh, in, out);
    /*
    mlab_vect v   = in.pop().to_int_vector(-1);

    if (!v.in_range(1,pmesh->nb_points())) 
      DAL_THROW(matlabint_bad_arg, "invalid point ids");

    std::vector<size_type> pts(v.size());
    for (size_type i=0; i < v.size(); i++) pts[i] = (int)v[i]-1;

    // we select the first point 
    bgeot::size_type ip = pts[0]; 
    // iterators over the convexes attached to point ip 
    bgeot::mesh_convex_ind_ct::const_iterator 
      cvit = pmesh->convex_to_point(ip).begin(),
      cvit_end = pmesh->convex_to_point(ip).end();    
    
    std::vector<bgeot::size_type> cvlst;
    std::vector<bgeot::size_type> facelst;
    for ( ; cvit != cvit_end; cvit++) {
      if (pmesh->is_convex_has_points(*cvit, pts.size(), pts.begin())) {
	// now, find the faces 
	for (bgeot::size_type f = 0; f < pmesh->structure_of_convex(*cvit)->nb_faces(); f++) {
	  if (pmesh->is_convex_face_has_points(*cvit, f, pts.size(), pts.begin())) {
	    cvlst.push_back(*cvit);
	    facelst.push_back(f);
	  }
	}
      }
    }
    mlab_vect w = out.pop().create_vector(2,cvlst.size());
    for (size_type j=0; j < cvlst.size(); j++) { w(0,j) = cvlst[j]+1; w(1,j) = facelst[j]+1; }
    */
  } else if (check_cmd(cmd, "outer faces", in, out, 0, 1, 0, 1)) {
    check_empty_mesh(pmesh);
    outer_faces(*pmesh, in, out);
  } else if (check_cmd(cmd, "save", in, out, 1, 1, 0, 0)) {
    std::string fname = in.pop().to_string();
    //    mexPrintf("saving to %s\n", fname.c_str());
    pmesh->write_to_file(fname);
  } else bad_cmd(cmd);
}

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  catch_errors(nlhs, plhs, nrhs, prhs, gf_mesh_get);
}
