#include <matlabint.h>
#include <matlabint_workspace.h>
#include <matlabint_mesh.h>

/*
  $Id: gf_mesh.C,v 1.6 2002/09/03 14:32:23 pommier Exp $

  ChangeLog:
  *Log:$
 */
using namespace matlabint;


static void
cartesian_mesh(getfem::getfem_mesh *pmesh, matlabint::mexargs_in &in)
{
  matlabint::size_type dim = in.remaining();
  
  if (dim == 0) DAL_THROW(matlabint_bad_arg, "not enough input arguments");

  std::vector<mlab_vect> ppos(dim);
  std::vector<size_type> npts(dim);
  dal::uint32_type grid_npoints=1, grid_nconvex=1;
  for (size_type i = 0; i < dim; i++) {
    ppos[i] = in.pop().to_scalar_vector();
    npts[i] = ppos[i].size();
    grid_npoints *= npts[i];
    grid_nconvex *= (npts[i]-1);
  }
  
  /* add the points in 'fortran style' order */
  getfem::base_node pt(dim);
  for (dal::uint32_type i=0; i < grid_npoints; i++) {
    dal::uint32_type k = i;    
    for (matlabint::size_type j = 0; j < dim; j++) {
      pt[j] = ppos[j][k % (npts[j])];
      k /= (npts[j]);
    }

    size_type id_pt = pmesh->add_point(pt);
    if (id_pt != i) {
      DAL_THROW(matlabint_error, 
		"something has changed in getfem, you need to reconsider "
		"gf_mesh('cartesian')\nfor point " << i << 
		", the index is " << id_pt << endl);
    }
    /*    if (i == grid_npoints-1) {
      cerr << "nb de pts ajoutes: " << grid_npoints << " id dernier =" << id_pt << endl;
      }*/
  }
  

  std::vector<int> ipt(dim);
  std::vector<getfem::base_node> pts(1 << dim+1);
  
  /* add the convexes */
  for (dal::uint32_type i=0; i < grid_nconvex; i++) {
    dal::uint32_type k = i;

    //    mexPrintf("convex %d/%d (k=%d)\n", i, grid_nconvex,k);
    
    /* find point location */
    for (matlabint::size_type j = 0; j < dim; j++) {
      ipt[j] = k % (npts[j]-1);
      //      mexPrintf("ipt[%d]=%d, k=%d\n", j, ipt[i], k);
      k /= (npts[j]-1);
    }

    /* build the vertices list */
    for (matlabint::size_type j = 0; j < (unsigned(1)<<dim); j++) {
      //      mexPrintf("[");
      pts[j].resize(dim);
      for (dal::uint32_type d=0; d < dim; d++) {
	if ((j >> d) & 1) {
	  pts[j][d] = ppos[d][ipt[d]+1];
	} else {
	  pts[j][d] = ppos[d][ipt[d]];
	}
	//	mexPrintf("%s%f", d==0 ? "" : "," ,pts[j][d]);
      }
      //mexPrintf("], ");
    }
    //mexPrintf("\n");
    pmesh->add_parallelepiped_by_points(dim, pts.begin());
  }
}

static void
triangles_grid_mesh(getfem::getfem_mesh *pmesh, matlabint::mexargs_in &in)
{
  if (in.remaining() != 2) DAL_THROW(matlabint_bad_arg, "not enough input arguments");
  
  mlab_vect X = in.pop().to_scalar_vector();
  mlab_vect Y = in.pop().to_scalar_vector();
  if (X.size() < 1 || Y.size() < 1) DAL_THROW(matlabint_bad_arg, "bad dimensions");

  size_type ni = Y.size(), nj = X.size();
  for (size_type i=0; i < ni; i++) {
    for (size_type j=0; j < nj; j++) {
      getfem::base_node pt(2);
      pt[0] = X[j]; pt[1] = Y[i];
      cerr << "pt = " << pt << endl;
      pmesh->add_point(pt);
    }
  }
  for (size_type i=0; i < ni-1; i++) {
    for (size_type j=0; j < nj-1; j++) {
      cerr << "i=" << i <<" j=" << j << endl;
      pmesh->add_triangle(i*nj + j, (i+1)*nj + j  , (i+1)*nj + j+1);
      pmesh->add_triangle(i*nj + j,     i*nj + j+1, (i+1)*nj + j+1);
    }
  }
}

static void
curved_mesh(getfem::getfem_mesh *dest_mesh, matlabint::mexargs_in &in)
{
  const getfem::getfem_mesh *src_mesh = in.pop().to_const_mesh();
  mlab_vect F = in.pop().to_scalar_vector(src_mesh->points().index().last()+1);

  int dim = src_mesh->dim();
  bgeot::base_node pt(dim+1);
  dest_mesh->clear();
  bgeot::size_type i, nbpt = src_mesh->points().index().last()+1;
  for (i = 0; i < nbpt; ++i)
  {
    std::copy(src_mesh->points()[i].begin(), src_mesh->points()[i].end(),
	      pt.begin());
    pt[dim] = F[i];
    dest_mesh->add_point(pt);
  }

  dal::bit_vector nn = src_mesh->convex_index();
  for (i << nn; i != bgeot::size_type(-1); i << nn)
  {
    dest_mesh->add_convex(src_mesh->trans_of_convex(i),
		       src_mesh->ind_points_of_convex(i).begin());
  }
}

static void
prismatic_mesh(getfem::getfem_mesh *dest_mesh, matlabint::mexargs_in &in)
{
  const getfem::getfem_mesh *src_mesh = in.pop().to_const_mesh();
  unsigned nblay = in.pop().to_integer(0,2500000);
  int dim = src_mesh->dim();
  bgeot::base_node pt(dim+1);

  dest_mesh->clear();
  bgeot::size_type i, nbpt = src_mesh->points().index().last()+1;
  
  for (i = 0; i < nbpt; ++i)
  {
    std::copy(src_mesh->points()[i].begin(), src_mesh->points()[i].end(),
	      pt.begin());
    pt[dim] = 0.0;
    for (size_type j = 0; j <= nblay; ++j, pt[dim] += 1.0 / nblay)
      dest_mesh->add_point(pt);
  }
  
  dal::bit_vector nn = src_mesh->convex_index();
  std::vector<bgeot::size_type> tab(5);
  for (i << nn; i != bgeot::size_type(-1); i << nn) // optimisable.
  {
    size_type nbp = src_mesh->nb_points_of_convex(i);
    if (2*nbp > tab.size()) tab.resize(2*nbp);
    for (size_type j = 0; j < nblay; ++j)
    {
      for (size_type k = 0; k < nbp; ++k)
	tab[k] = (nblay+1)*src_mesh->ind_points_of_convex(i)[k] + j;
      for (size_type k = 0; k < nbp; ++k)
	tab[k+nbp] = (nblay+1)*src_mesh->ind_points_of_convex(i)[k] + j + 1;
      bgeot::pgeometric_trans pgt = 
	bgeot::product_geotrans(src_mesh->trans_of_convex(i), bgeot::simplex_geotrans(1,1));
      /*
	bgeot::geometric_trans_descriptor("GT_PRODUCT(" +
					  bgeot::name_of_geometric_trans(src_mesh->trans_of_convex(i)) +
					  ",GT_PK(1,1))");
      */
      dest_mesh->add_convex(pgt, tab.begin());
      
      /*
      bgeot::convex_product_trans(
		         src_mesh->trans_of_convex(i),
			 bgeot::simplex_trans(1, 1)), tab.begin());
      */
    }
  }  
}

static void
pt2D_mesh(getfem::getfem_mesh *mesh, matlabint::mexargs_in &in)
{
  mlab_vect P = in.pop().to_scalar_vector(2, -1);
  mlab_vect T = in.pop().to_scalar_vector(4, -1);
  id_type zone = 0;
  if (in.remaining()) zone = in.pop().to_integer(1,65000);

  getfem::base_node pt(2);
  for (size_type i = 0; i < P.getn(); ++i) {
    pt[0] = P[2*i]; pt[1] = P[2*i+1];
    size_type id = mesh->add_point(pt);

    /* une hypothse bien commode pour la "compatibilit pdetool" */
    if (id != i) {
      DAL_THROW(matlabint_error, "Error, the numbering of mesh points will be different");
    }
  }

  for (size_type i = 0; i < T.getn(); ++i) {
    id_type u = id_type(T(0,i)), v = id_type(T(1,i));
    id_type w = id_type(T(2,i)), x = id_type(T(3,i));
    if (u > P.size() || v > P.size() || w > P.size())
      DAL_THROW(matlabint_bad_arg, "Bad tringulation.");
    if (zone == 0 || zone == x) mesh->add_triangle(u-1, v-1, w-1);
  }
}

/*MLABCOM

  FUNCTION i = gf_mesh([operation [, args]])

  General constructor for mesh object. Returns a getfem handle 
  to the newly created mesh object.
  
  * m = gf_mesh() 
  Creates a new empty mesh.

  * M = gf_mesh('cartesian', vec X[, vec Y[, vec Z,..]])
  Builds quickly a cartesian mesh

  * M = gf_mesh('triangles grid', vec X[, vec Y[, vec Z,..]])
  Builds quickly a regular planar mesh composed of triangle (the
  advantage over the 'cartesian' mesh is that the geometrical
  transformation is linear).

  * M = gf_mesh('curved', mesh I, vec F)
  Build a curved 3D (resp 2D) mesh m from a planar (resp linear) mesh i:
  The mesh m has one additional dimension. The additional coordinate is 
  given by the vector F. This can be used to obtain meshes for shells.
  (I may be a mesh_fem, in that case its linked mesh will be used)

  * M = gf_mesh('prismatic', mesh I, int K)
  Builds a prismatic mesh M from a mesh I. In the additional dimension
  there is K layers of elements built from 0 to 1.

  * M = gf_mesh('pt2D', p, t[, n])
  Converts a pdetool mesh exported in variables p and t in a GETFEM mesh i.
  n is optional and is a zone number. If n is specified only the wone number
  n is converted.

  * M = gf_mesh('load', string FILENAME)
  Loads a mesh from an ascii file (which may have been created by
  gf_mesh_get(M,'save',FILENAME))

  $Id: gf_mesh.C,v 1.6 2002/09/03 14:32:23 pommier Exp $
MLABCOM*/
void gf_mesh(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  matlabint_mesh *mi_mesh = new matlabint_mesh();
  out.pop().from_object_id(workspace().push_object(mi_mesh));
  getfem::getfem_mesh *pmesh = &mi_mesh->mesh();

  if (in.narg() == 0) {
    if (out.narg() > 1)
      DAL_THROW(matlabint_bad_arg, "Wrong number of output arguments");
  } else {
    std::string cmd    = in.pop().to_string();
    if (check_cmd(cmd, "cartesian", in, out, 1, 32, 1, 1)) {
      cartesian_mesh(pmesh, in);
    } else if (check_cmd(cmd, "triangles grid", in, out, 2, 2, 1, 1)) {
      triangles_grid_mesh(pmesh, in);
    } else if (check_cmd(cmd, "curved", in, out, 2, 2, 1, 1)) {
      curved_mesh(pmesh, in);
    } else if (check_cmd(cmd, "prismatic", in, out, 2, 2, 1, 1)) {
      prismatic_mesh(pmesh, in);
    } else if (check_cmd(cmd, "pt2D", in, out, 2, 3, 1, 1)) {
      pt2D_mesh(pmesh, in);
    } else if (check_cmd(cmd, "load", in, out, 1, 1, 1, 1)) {
      std::string fname = in.pop().to_string();
      pmesh->read_from_file(fname);
    } else bad_cmd(cmd);
  }
}

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  catch_errors(nlhs, plhs, nrhs, prhs, gf_mesh, "gf_mesh");
}
