/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom)                                    */
/* File    : sup_point_to_mesh.C : suppress a point to a mesh variable.    */
/*     									   */
/*                                                                         */
/* Date : September 27, 2001.                                              */
/* Author : Yves Renard, Yves.Renard@gmm.insa-tlse.fr                      */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "mex.h"
#include <matlabint_mesh.h>

/**@name sup_point_to_mesh(i, a1, a2, ...) 
 * Suppress from an existing mesh i, the points of index a1, a2, ...
 */
//@{
//@}

#define id_type matlabint::id_type


void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  mexint_test_narg(3, 1000, 0, 0);
 
  id_type *p = (id_type *)(mxGetData(prhs[0]));
  if (mxGetClassID(prhs[0]) != mxUINT32_CLASS 
      || (*p & 0xFF000000) != MATLABINT_MESH_MASK)
    mexErrMsgTxt("First input argument should be a descriptor returned by new_mesh.");
  int i = *p - MATLABINT_MESH_MASK;
  if (!matlabint::exists_mesh(i))
    mexErrMsgTxt("This mesh no longer exists.");
  getfem::getfem_mesh *pmesh = &(matlabint::mesh(i));


  for (int n = 1; n < nrhs; ++n)
  {
    void *q = mxGetData(prhs[1]);
    id_type a;
  
    switch (mxGetClassID(prhs[1]))
    {
      case mxCHAR_CLASS   : a = id_type(*((char *)q));             break;
      case mxDOUBLE_CLASS : a = id_type(*((double *)q));           break;
      case mxSINGLE_CLASS : a = id_type(*((float *)q));            break;
      case mxINT8_CLASS   : a = id_type(*((dal::int8_type *)q));   break;
      case mxUINT8_CLASS  : a = id_type(*((dal::uint8_type *)q));  break;
      case mxINT16_CLASS  : a = id_type(*((dal::int16_type *)q));  break;
      case mxUINT16_CLASS : a = id_type(*((dal::uint16_type *)q)); break;
      case mxINT32_CLASS  : a = id_type(*((dal::int32_type *)q));  break;
      case mxUINT32_CLASS : a = id_type(*((dal::uint32_type *)q)); break;
      default : mexErrMsgTxt("Second argument should be numeric.");
    }
    if (pmesh->point_is_valid(a))
      mexErrMsgTxt("Impossible to suppress a point if a convex is still attached to it.");
    pmesh->sup_point(a-1);
  }
  
}
