/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom)                                    */
/* File    : set_poly.C : set a polynomial variable.                       */
/*     									   */
/*                                                                         */
/* Date : October 10, 2001.                                                */
/* Author : Yves Renard, Yves.Renard@gmm.insa-tlse.fr                      */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "mex.h"
#include <matlabint_poly.h>

/**@name set_poly(i, a, b) or set_poly(i, a)
 * Set a polynomial variable. i should be a descriptor given
 * by new_poly. a is the total number of variable, b is the variable setted 
 * to 1. For instance set_poly(i, 2, 0) means that the polynomial has two
 * variables and is setted to x. set_poly(i, 2) means that the polynomial has
 * two variables and is setted two 1.0 .
 */
//@{
//@}


void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  if (nrhs != 3 && nrhs != 2)
    mexErrMsgTxt("Wrong number of input arguments.");
  if (nlhs != 0)
    mexErrMsgTxt("No output argument needed.");
  matlabint::id_type *p = (matlabint::id_type *)(mxGetData(prhs[0]));
  if (mxGetClassID(prhs[0]) != mxUINT32_CLASS 
      || (*p & 0xFF000000) != MATLABINT_POLY_MASK)
    mexErrMsgTxt("Input argument should be a descriptor returned by new_poly.");
  int i = *p - MATLABINT_POLY_MASK;
  if (!matlabint::exists_poly(i))
    mexErrMsgTxt("This polynomial no longer exists.");

  
  void *q1 = mxGetData(prhs[1]), *q2;
  int a, b;
  if (nrhs == 3) q2 = mxGetData(prhs[2]);
  
  switch (mxGetClassID(prhs[1]))
  {
    case mxCHAR_CLASS   : a = int(*((char *)q1));             break;
    case mxDOUBLE_CLASS : a = int(*((double *)q1));           break;
    case mxSINGLE_CLASS : a = int(*((float *)q1));            break;
    case mxINT8_CLASS   : a = int(*((dal::int8_type *)q1));   break;
    case mxUINT8_CLASS  : a = int(*((dal::uint8_type *)q1));  break;
    case mxINT16_CLASS  : a = int(*((dal::int16_type *)q1));  break;
    case mxUINT16_CLASS : a = int(*((dal::uint16_type *)q1)); break;
    case mxINT32_CLASS  : a = int(*((dal::int32_type *)q1));  break;
    case mxUINT32_CLASS : a = int(*((dal::uint32_type *)q1)); break;
    default : mexErrMsgTxt("Second argument should be numeric.");
  }

  if (nrhs == 3) switch (mxGetClassID(prhs[2]))
  {
    case mxCHAR_CLASS   : b = int(*((char *)q2));             break;
    case mxDOUBLE_CLASS : b = int(*((double *)q2));           break;
    case mxSINGLE_CLASS : b = int(*((float *)q2));            break;
    case mxINT8_CLASS   : b = int(*((dal::int8_type *)q2));   break;
    case mxUINT8_CLASS  : b = int(*((dal::uint8_type *)q2));  break;
    case mxINT16_CLASS  : b = int(*((dal::int16_type *)q2));  break;
    case mxUINT16_CLASS : b = int(*((dal::uint16_type *)q2)); break;
    case mxINT32_CLASS  : b = int(*((dal::int32_type *)q2));  break;
    case mxUINT32_CLASS : b = int(*((dal::uint32_type *)q2)); break;
    default : mexErrMsgTxt("Second argument should be numeric.");
  }

  if (nrhs == 2)
  {
    matlabint::poly(i) = bgeot::base_poly(a, 1);
    matlabint::poly(i).one();
  }
  else
  {
    if (a > 255 || b >= a || a <= 0 || b < 0)
      mexErrMsgTxt("One of the arguments is out of range.");
    matlabint::poly(i) = bgeot::base_poly(a, 1, b);
  }
}
