/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom)                                    */
/* File    : set_finite_element.C : set the fem in a mesh_fem.             */
/*     									   */
/*                                                                         */
/* Date : October 19, 2001.                                                */
/* Author : Yves Renard, Yves.Renard@gmm.insa-tlse.fr                      */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "mex.h"
#include <matlabint_mesh_fem.h>
#include <matlabint_intset.h>
#include <matlabint_pfem.h>
#include <matlabint_pfi.h>

/**@name set_finite_elem(mf, is, fem, ppi)
 * set the finite element method to fem with ppi as integration method
 * for all the convexes of index is in
 * the mesh linked to mf.
 */
//@{
//@}

typedef bgeot::size_type size_type;
typedef matlabint::id_type id_type;

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  if (nrhs != 4)
    mexErrMsgTxt("Wrong number of input arguments, should be 4.");
  if (nlhs != 0)
    mexErrMsgTxt("No output argument needed.");
  id_type *p = (id_type *)(mxGetData(prhs[0]));
  if (mxGetClassID(prhs[0]) != mxUINT32_CLASS 
      || (*p & 0xFF000000) != MATLABINT_MESHFEM_MASK)
    mexErrMsgTxt("First input argument should be a descriptor returned by new_mesh_fem.");
  int i = *p - MATLABINT_MESHFEM_MASK;
  if (!matlabint::exists_mesh_fem(i))
    mexErrMsgTxt("This mesh fem no longer exists.");

  getfem::mesh_fem *pmf = &(matlabint::mesh_fem(i));

  p = (id_type *)(mxGetData(prhs[1]));
  if (mxGetClassID(prhs[1]) != mxUINT32_CLASS 
      || (*p & 0xFF000000) != MATLABINT_INTSET_MASK)
    mexErrMsgTxt("Second input argument should be a descriptor returned by new_intset.");
  i = *p - MATLABINT_INTSET_MASK;
  if (!matlabint::exists_intset(i))
    mexErrMsgTxt("This intset no longer exists.");

  dal::bit_vector *bv = &(matlabint::intset(i));

  // si ce n'est pas un int_set, prendre comme entie en vrifiant ...

  p = (id_type *)(mxGetData(prhs[2]));
  if (mxGetClassID(prhs[1]) != mxUINT32_CLASS 
      || (*p & 0xFF000000) != MATLABINT_PFEM_MASK)
    mexErrMsgTxt("Third input argument should be a fem descriptor.");
  i = *p - MATLABINT_PFEM_MASK;
  if (!matlabint::exists_pfem(i))
    mexErrMsgTxt("This fem does not exist.");
  getfem::pfem pfe = matlabint::addr_pfem(i);

  p = (id_type *)(mxGetData(prhs[3]));
  if (mxGetClassID(prhs[3]) != mxUINT32_CLASS 
      || (*p & 0xFF000000) != MATLABINT_PFI_MASK)
    mexErrMsgTxt("Third input argument should be a integration method descriptor.");
  i = *p - MATLABINT_PFI_MASK;
  if (!matlabint::exists_pfi(i))
    mexErrMsgTxt("This integration method does not exist.");
  getfem::pintegration_method ppi = matlabint::addr_pfi(i);

  dal::bit_vector nn = *bv;
  size_type k;
  for (k << nn; k != size_type(-1); k << nn)
  {
    if (!((pmf->linked_mesh().convex_index())[k]))
      mexErrMsgTxt("At least one of the convexes does not exist.");
    if (pfe->basic_structure()
	  != pmf->linked_mesh().structure_of_convex(k))
      mexErrMsgTxt("Incompatibility between finite element method and one convex");
  }
  

  pmf->set_finite_element(*bv, pfe, ppi);
  
  
}
