/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom) version 1.0                        */
/* File    : assem_linear_elasticity.C : .                                 */
/*     				          		                   */
/*                                                                         */
/* Date : March 2002.                                                      */
/* Author : J. Pommier pommier@gmm.insa-tlse.fr                            */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "matlabint_misc.h"

#define MEXERROR mexErrMsgTxt(" ==> error in set_boundary_number.C <==");


/**@name B = set_boundary_number(boundary_num, mf, edge_list)
 * 
 * 
 * output: none
 */
//@{
//@}
void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  getfem::mesh_fem *mf;
  const mxArray *mxEdge;
  double *dr;
  int nb_edge, nb_pts;
  int boundary_num;

  if (nrhs != 3) 
    mexErrMsgTxt("Wrong number of input argument. [boundary_num, mesh_fem, edges]");
  else if (nlhs != 0)
    mexErrMsgTxt("Wrong number of output argument.");

  boundary_num = check_for_scalar_integer_arg(prhs[0], 1, "boundary_num");
  mf = check_for_mesh_fem_arg(prhs[1]);
  mxEdge = prhs[2];
  nb_edge = mxGetN(mxEdge);
  nb_pts  = mxGetM(mxEdge);
  
  if (!mxIsDouble(mxEdge)) {
    mexPrintf("arg 3 should be a standard matrix containing point numbers, its data is expected to be stored as real numbers\n");
    MEXERROR;
  }
  dr = mxGetPr(mxEdge);

  const getfem::getfem_mesh &mesh = mf->linked_mesh();

  /* loop over the edges of mxEdge */
  for (int iedge=0; iedge < nb_edge; iedge++) {
    std::vector<int> pts(nb_pts);

    /* we store the node numbers in the pts array */
    //    mexPrintf("pts[%d]=[", iedge+1);
    for (int i=0; i < nb_pts; i++) {
      pts[i] = (int)(dr[iedge*nb_pts+i]);
      if ((double)pts[i] != dr[iedge*nb_pts+i]) {
	mexPrintf("arg 3 should contain integer values, but we found '%f' at column %d, row %d\n",
		  dr[iedge*nb_pts+i], iedge+1, i+1);
	MEXERROR;
      }
      pts[i]--; /* matlab starts at 1, but we start at 0 */
      if (pts[i] < 0 || pts[i] >= mesh.nb_points()) {
	mexPrintf("invalid node number '%f' at column %d, row %d: is not in the range [1:%d]\n",
		  dr[iedge*nb_pts+i], iedge+1, i+1, mesh.nb_points());
	MEXERROR;
      }
      //      mexPrintf("%d ", pts[i]+1);
    }
    //    mexPrintf("]\n");


    /* we select the first point */
    bgeot::size_type ip = pts[0]; 

    /* iterators over the convexes attached to point ip */
    bgeot::mesh_convex_ind_ct::const_iterator 
      cvit = mesh.convex_to_point(ip).begin(),
      cvit_end = mesh.convex_to_point(ip).end();    
    
    /* loop over the convexes */
    int convex_num=-1, face_num=-1;
    for ( ; cvit != cvit_end; cvit++) {
      int nfaces = mesh.structure_of_convex(*cvit)->nb_faces();

      //      mexPrintf("iedge=%d, convex_num = %d ", iedge + 1, *cvit + 1);

      /* if we found the right convex */
      if (mesh.is_convex_has_points(*cvit, pts.size(), pts.begin())) {
	/* check that this is the first (and only convex) which contains the points */
	if (convex_num == -1) {	  
	  convex_num = *cvit;
	  //	  mexPrintf("le convexe %d a les points de l'arete\n", convex_num+1);

	  /* now, find the correct face */
	  for (bgeot::size_type f = 0; f < nfaces; f++) {
	    /*	    bgeot::ind_ref_mesh_point_ind_ct pt = mesh.ind_points_of_face_of_convex(*cvit, f);
	     	    mexPrintf("pt = %d, %d\n", *(pt.begin()) + 1, *(pt.begin()+1) + 1);
	    */
	    
	    if (mesh.is_convex_face_has_points<std::vector<int>::const_iterator>(*cvit, f, 
									    pts.size(), 
									    pts.begin())) {
	      //	      mexPrintf("la face %d contient les %d points\n", f+1, pts.size());
	      if (face_num == -1) {
		face_num = f;
	      } else {
		mexPrintf("the points of 'edge' %d are contained by two differents faces of convex %d!\n"
			  "it looks like you didn't use enough points to describe a unique faced\n",
			  iedge+1, convex_num+1);
		MEXERROR;
	      }
	    }
	  }
	} else {
	  mexPrintf("the points of edge %d are contained by both convex %d\n"
		    "and convex %d: this is not a border edge\n", iedge+1, convex_num+1, *cvit+1);
	  MEXERROR;
	}
      }      
    }

    /* just check if the face/convex was found */
    if (convex_num == -1) {
      mexPrintf("no convex contains the points of edge %d, sorry\n", iedge+1);
      MEXERROR;
    }
    if (face_num == -1) {
      mexPrintf("you probably found a bug! (iedge=%d)\n",iedge);
      MEXERROR;
    }
    mexPrintf("adding convex %d, face %d to boundary %d..", convex_num+1, face_num+1, boundary_num);
    mf->add_boundary_elt(boundary_num, convex_num, face_num);
    if (!mf->is_convex_on_boundary(convex_num, boundary_num)) {
      mexPrintf("there was a problem..\n");
    } else {
      mexPrintf("ok, number of faces on this boundary: %d\n",
		mf->convex_on_boundary(boundary_num).card()
		);
    }
  }
  /// Add to the boundary b the face f of the element 
 //void 
}
  
