/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom)                                    */
/* File    : prismatic_mesh.C : Build a prismatic mesh.                    */
/*     									   */
/*                                                                         */
/* Date : October 12, 2001.                                                */
/* Author : Yves Renard, Yves.Renard@gmm.insa-tlse.fr                      */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "mex.h"
#include <matlabint_mesh.h>

/**@name prismatic_mesh(i, j, k)
 * Build a prismatic mesh j with a mesh i. In the additional dimension
 * there is k lawers of elements built from 0 to 1.
 */
//@{
//@}

#define id_type matlabint::id_type

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  mexint_test_narg(3, 3, 0, 0);
  
  id_type *p = (id_type *)(mxGetData(prhs[0]));
  if (mxGetClassID(prhs[0]) != mxUINT32_CLASS 
      || (*p & 0xFF000000) != MATLABINT_MESH_MASK)
    mexErrMsgTxt("First input argument should be a descriptor returned by new_mesh.");
  int i1 = *p - MATLABINT_MESH_MASK;
  if (!matlabint::exists_mesh(i1))
    mexErrMsgTxt("First mesh no longer exists.");
  getfem::getfem_mesh *pmesh1 = &(matlabint::mesh(i1));

  p = (id_type *)(mxGetData(prhs[1]));
  if (mxGetClassID(prhs[1]) != mxUINT32_CLASS 
      || (*p & 0xFF000000) != MATLABINT_MESH_MASK)
    mexErrMsgTxt("Second input argument should be a descriptor returned by new_mesh.");
  int i2 = *p - MATLABINT_MESH_MASK;
  if (!matlabint::exists_mesh(i2))
    mexErrMsgTxt("Second mesh no longer exists.");
  getfem::getfem_mesh *pmesh2 = &(matlabint::mesh(i2));

  void *q3 = mxGetData(prhs[2]); id_type a = 0;
    
  switch (mxGetClassID(prhs[2]))
  {
    case mxCHAR_CLASS   : a = id_type(*((char *)q3));             break;
    case mxDOUBLE_CLASS : a = id_type(*((double *)q3));           break;
    case mxSINGLE_CLASS : a = id_type(*((float *)q3));            break;
    case mxINT8_CLASS   : a = id_type(*((dal::int8_type *)q3));   break;
    case mxUINT8_CLASS  : a = id_type(*((dal::uint8_type *)q3));  break;
    case mxINT16_CLASS  : a = id_type(*((dal::int16_type *)q3));  break;
    case mxUINT16_CLASS : a = id_type(*((dal::uint16_type *)q3)); break;
    case mxINT32_CLASS  : a = id_type(*((dal::int32_type *)q3));  break;
    case mxUINT32_CLASS : a = id_type(*((dal::uint32_type *)q3)); break;
    default : mexErrMsgTxt("Fourth argument should be numeric.");
  }
    
  if (a == 0 || a > 250000)
    mexErrMsgTxt("Third argument is out of range.");

  int dim = pmesh1->dim();
  bgeot::base_node pt(dim+1);

  pmesh2->clear();
  bgeot::size_type i, nbpt = pmesh1->points().index().last()+1;
  
  for (i = 0; i < nbpt; ++i)
  {
    std::copy(pmesh1->points()[i].begin(), pmesh1->points()[i].end(),
	      pt.begin());
    pt[dim] = 0.0;
    for (int j = 0; j <= a; ++j, pt[dim] += 1.0 / a)
      pmesh2->add_point(pt);
  }
  
  dal::bit_vector nn = pmesh1->convex_index();
  std::vector<bgeot::size_type> tab(5);
  for (i << nn; i != bgeot::size_type(-1); i << nn) // optimisable.
  {
    int nbp = pmesh1->nb_points_of_convex(i);
    if (2*nbp > tab.size()) tab.resize(2*nbp);
    for (int j = 0; j < a; ++j)
    {
      for (int k = 0; k < nbp; ++k)
	tab[k] = (a+1)*pmesh1->ind_points_of_convex(i)[k] + j;
      for (int k = 0; k < nbp; ++k)
	tab[k+nbp] = (a+1)*pmesh1->ind_points_of_convex(i)[k] + j + 1;
      pmesh2->add_convex(bgeot::convex_product_trans(
		         pmesh1->trans_of_convex(i),
			 bgeot::simplex_trans(1, 1)), tab.begin());
    }
  }
  
}
