/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom)                                    */
/* File    : mesh_from_pdetool.C : convert a pdetool mesh to a GETFEM mesh.*/
/*     									   */
/*                                                                         */
/* Date : October 12, 2001.                                                */
/* Author : Yves Renard, Yves.Renard@gmm.insa-tlse.fr                      */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "mex.h"
#include <matlabint_mesh.h>

/**@name mesh_from_pdetool(i, p, t) or mesh_from_pdetool(i, p, t, n) 
 * convert a pdetool mesh exported in variables p and t in a GETFEM mesh i.
 * n is optional and is a zone number. If n is specified only the wone number
 * n is converted. If the mesh i contains something, it is first erased.
 */
//@{
//@}

#define id_type matlabint::id_type

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  mexint_test_narg(3, 4, 0, 0);
  
  id_type *p = (id_type *)(mxGetData(prhs[0]));
  if (mxGetClassID(prhs[0]) != mxUINT32_CLASS 
      || (*p & 0xFF000000) != MATLABINT_MESH_MASK)
    mexErrMsgTxt("First input argument should be a descriptor returned by new_mesh.");
  int i = *p - MATLABINT_MESH_MASK;
  if (!matlabint::exists_mesh(i))
    mexErrMsgTxt("This mesh no longer exists.");
  getfem::getfem_mesh *pmesh = &(matlabint::mesh(i));
  

  double *q1 = (double *)(mxGetData(prhs[1]));
  int n1 = mxGetN(prhs[1]), m1 = mxGetM(prhs[1]);
  if (mxGetClassID(prhs[1]) != mxDOUBLE_CLASS || m1 != 2)
    mexErrMsgTxt("Second argument is not in a good format.");

  double *q2 = (double *)(mxGetData(prhs[2]));
  int n2 = mxGetN(prhs[2]), m2 = mxGetM(prhs[2]);
  if (mxGetClassID(prhs[2]) != mxDOUBLE_CLASS || m2 != 4)
    mexErrMsgTxt("Second argument is not in a good format.");

  void *q3; id_type a = 0;
  if (nrhs == 4)
  { 
    q3 = mxGetData(prhs[3]);
    
    switch (mxGetClassID(prhs[3]))
    {
      case mxCHAR_CLASS   : a = id_type(*((char *)q3));             break;
      case mxDOUBLE_CLASS : a = id_type(*((double *)q3));           break;
      case mxSINGLE_CLASS : a = id_type(*((float *)q3));            break;
      case mxINT8_CLASS   : a = id_type(*((dal::int8_type *)q3));   break;
      case mxUINT8_CLASS  : a = id_type(*((dal::uint8_type *)q3));  break;
      case mxINT16_CLASS  : a = id_type(*((dal::int16_type *)q3));  break;
      case mxUINT16_CLASS : a = id_type(*((dal::uint16_type *)q3)); break;
      case mxINT32_CLASS  : a = id_type(*((dal::int32_type *)q3));  break;
      case mxUINT32_CLASS : a = id_type(*((dal::uint32_type *)q3)); break;
      default : mexErrMsgTxt("Fourth argument should be numeric.");
    }
    
    if (a == 0 || a > 65000)
      mexErrMsgTxt("Fourth argument is out of range.");
  }

  pmesh->clear();
  getfem::base_node pt(2);
  for (int i = 0; i < n1; ++i)
  {
    pt[0] = *q1++; pt[1] = *q1++;
    pmesh->add_point(pt);
  }

  for (int i = 0; i < n2; ++i)
  {
    id_type u = id_type(*q2++), v = id_type(*q2++);
    id_type w = id_type(*q2++), x = id_type(*q2++);
    if (u > n1 || v > n1 || w > n1)
      mexErrMsgTxt("Second argument is not in a good format.");
    if (a == 0 || a == x) pmesh->add_triangle(u-1, v-1, w-1);
  }
}
