/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom)                                    */
/* File    : mat_elem.C : gives an elementary matrix.                      */
/*     									   */
/*                                                                         */
/* Date : October 19, 2001.                                                */
/* Author : Yves Renard, Yves.Renard@gmm.insa-tlse.fr                      */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "mex.h"
#include <matlabint_mesh_fem.h>
#include <matlabint_matelemtype.h>
#include <matlabint_intset.h>

/**@name M = mat_elem(mf, met, i)
 * Gives the elementary matrix coresponding to the convex i of mf and the
 * elmentary matrix type met.
 */
//@{
//@}

typedef matlabint::id_type id_type;

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  if (nrhs != 3)
    mexErrMsgTxt("Wrong number of input arguments, should be 3.");
  if (nlhs > 1)
    mexErrMsgTxt("Wrong number of output arguments.");
  matlabint::id_type *p = (matlabint::id_type *)(mxGetData(prhs[0]));
  if (mxGetClassID(prhs[0]) != mxUINT32_CLASS 
      || (*p & 0xFF000000) != MATLABINT_MESHFEM_MASK)
    mexErrMsgTxt("First input argument should be a descriptor returned by new_mesh_fem.");
  int i = *p - MATLABINT_MESHFEM_MASK;
  if (!matlabint::exists_mesh_fem(i))
    mexErrMsgTxt("This mesh fem no longer exists.");

  getfem::mesh_fem *pmf = &(matlabint::mesh_fem(i));

  p = (matlabint::id_type *)(mxGetData(prhs[1]));
  if (mxGetClassID(prhs[1]) != mxUINT32_CLASS 
      || (*p & 0xFF000000) != MATLABINT_MATELEMTYPE_MASK)
    mexErrMsgTxt("Second input argument should be a elementary matrix descriptor.");
  i = *p - MATLABINT_MATELEMTYPE_MASK;
  if (!matlabint::exists_matelemtype(i))
    mexErrMsgTxt("This mat elem type no longer exists.");

  getfem::pmat_elem_type pmet = matlabint::addr_matelemtype(i);

  id_type a; void *q1 = mxGetData(prhs[2]);
  switch (mxGetClassID(prhs[2]))
  {
    case mxCHAR_CLASS   : a = id_type(*((char *)q1));             break;
    case mxDOUBLE_CLASS : a = id_type(*((double *)q1));           break;
    case mxSINGLE_CLASS : a = id_type(*((float *)q1));            break;
    case mxINT8_CLASS   : a = id_type(*((dal::int8_type *)q1));   break;
    case mxUINT8_CLASS  : a = id_type(*((dal::uint8_type *)q1));  break;
    case mxINT16_CLASS  : a = id_type(*((dal::int16_type *)q1));  break;
    case mxUINT16_CLASS : a = id_type(*((dal::uint16_type *)q1)); break;
    case mxINT32_CLASS  : a = id_type(*((dal::int32_type *)q1));  break;
    case mxUINT32_CLASS : a = id_type(*((dal::uint32_type *)q1)); break;
    default : mexErrMsgTxt("Third argument should be numeric.");
  }
  
  if (a == 0 || !(pmf->convex_index().is_in(a-1)))
    mexErrMsgTxt("This convex is no interfaced.");
  a--;
  // des verifications  faire sur les mef ...
  getfem::base_tensor t;
  getfem::pmat_elem_computation pmec = getfem::mat_elem(pmet, pmf->int_method_of_element(a) , pmf->linked_mesh().trans_of_convex(a));
  pmec->gen_compute(t, pmf->linked_mesh().points_of_convex(a));
  
  std::vector<int> tab(t.order());
  std::copy(t.sizes().begin(), t.sizes().end(), tab.begin());
  plhs[0] = mxCreateNumericArray(t.order(), &(tab.begin()[0]),mxDOUBLE_CLASS, mxREAL);
  double *q = (double *)(mxGetData(plhs[0]));
  std::copy(t.begin(), t.end(), q);       
  
}
