/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom)                                    */
/* File    : gf_interpolate_on_other_mesh.C                                */
/*     								           */
/*                                                                         */
/* Date : February 2002001.                                                */
/* Author : J. Pommier pommier@gmm.insa-tlse.fr                            */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "matlabint_misc.h"
#include <getfem_export.h>
 
using namespace getfem;

void
warn_for_non_lagrange_elements(mesh_fem &mf)
{
  dal::bit_vector nn = mf.convex_index();

  size_type cv, cnt=0, total=0;
  for (cv << nn; cv != ST_NIL; cv << nn) {
    if (!mf.fem_of_element(cv)->is_lagrange()) cnt++; 
    total++;
  }
  if (cnt) {
    mexPrintf("Warning: %d elements on %d are NOT lagrange elements\n", cnt, total);
  }
}

/**@name k = gf_interpolate_on_other_mesh(meshfem mf_src, meshfem mf_dest, VEC Usrc)
 * interpolates a field defined on mesh 'mf_src' on another (lagrangian) mesh 'mf_dest'
 */
//@{
//@}
void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  getfem::mesh_fem *mf_src=NULL, *mf_dest=NULL;

  VECbidon vec_src, vec_dest;
  size_type dim;
  //  const double * vec_lambda, * vec_mu;
    

  if (nrhs != 3) 
    mexErrMsgTxt("Wrong number of input argument.");
  else if (nlhs > 1)
    mexErrMsgTxt("Wrong number of output argument.");

  mf_src = check_for_mesh_fem_arg(prhs[0]);
  mf_dest = check_for_mesh_fem_arg(prhs[1]);
  vec_src.ve = check_for_node_values(prhs[2], mf_src, 0, &dim);

  warn_for_non_lagrange_elements(*mf_dest);

  plhs[0] = mxCreateDoubleMatrix(dim, mf_dest->nb_dof(), mxREAL); assert(plhs[0] != 0);
  vec_dest.ve = mxGetPr(plhs[0]);
  
  interpolation_solution<VECbidon>(*mf_src, *mf_dest,
				   vec_src, vec_dest, dim);
}

