/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom) version 1.0                        */
/* File    : gf_get_refined_P1_tri_mesh                                    */
/*     				          		     */
/*                                                                         */
/* Date : March 2002001.                                                   */
/* Author : J. Pommier pommier@gmm.insa-tlse.fr                            */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "matlabint_misc.h"

using namespace getfem;


void eval_sub_nodes(int N, bgeot::pconvex_ref cv, std::vector<base_node>& spt)
{
  assert(N>0);
  spt.resize(((N+1)*(N+2))/2);
  
  if (cv->nb_points() != 3) {
    mexPrintf("Error, element %d is not a triangle (convex with %d points):\n", cv, cv->nb_points());
    mexErrMsgTxt("sorry\n");
  } else {   
    /*
      .      pt[0]
      .        /\
      .       /  \
      . pt[2]/____\pt[1]
      
      refinment:
      
      .         0     <- layer 0
      .        111    <- layer 1 etc..
      .       22222
      
    */
    
    spt[0] = cv->points()[0];
    int pcnt = 1;

    /* find the three nodes of the each sub-triangle */
    for (int layer = 1; layer <= N; layer++) {
      base_node A,B;
      scalar_type c;
      c = ((scalar_type)layer)/N;

      A = cv->points()[0] + (cv->points()[2] - cv->points()[0]) * c;
      B = cv->points()[0] + (cv->points()[1] - cv->points()[0]) * c;

      for (int inode = 0; inode <= layer; inode++, pcnt++) {
	spt[pcnt] = A + (B-A) * ((inode)/(scalar_type)layer);
	/*	mexPrintf("layer %d, inode %d: [%f,%f]\n",
		layer, inode, spt[pcnt][0], spt[pcnt][1]);*/
      }
    }
    assert(pcnt == spt.size());
  }
}

/**@name B = gf_get_refined_P1_tri_mesh(mesh_fem mf,int dim_u, vec u, int refine_lvl)
 * 
 * get the list of interpolation points of the mesh_fem
 * output: B is a matrix [Nxn_points] containing the point locations
 */
//@{
//@}
void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  mesh_fem *mf;
  int warn_cnt  = 0;
  int Nrefine = 0;
  double *U;

  if (nrhs != 3) 
    mexErrMsgTxt("Wrong number of input argument. [arg1 : mesh_fem, arg2: U, arg3 : Nrefine]");
  else if (nlhs > 1)
    mexErrMsgTxt("Wrong number of output argument.");

  mf = check_for_mesh_fem_arg(prhs[0]);
  U = check_for_node_values(prhs[1], mf);
  Nrefine = check_for_scalar_integer_arg(prhs[2], 1, "Number of refinements");


  dal::bit_vector nn = mf->convex_index();
  int nb_convex = nn.card();

  plhs[0] = mxCreateDoubleMatrix(6 + 3,nb_convex*(Nrefine)*(Nrefine), mxREAL); assert(plhs[0]);
  double *out = mxGetPr(plhs[0]);

  size_type cv;
  std::vector<base_node> pt(Nrefine * Nrefine);
  int cv_cnt = 0;
  int refined_tri_cnt = 0;

  for (cv << nn; cv != ST_NIL; cv << nn, cv_cnt++) {
    /* on pourrait aussi tester que la fem est la mme sur tous les lments */
    /*
    if (!mf->fem_of_element(cv)->is_lagrange() && warn_cnt==0) {
      mexPrintf("warning: some elements are *NOT* lagrange elements");
      warn_cnt++;
    }
    */

    pfem cv_fem(mf->fem_of_element(cv)); assert(cv_fem);
    bgeot::pconvex_ref cv_ref(cv_fem->ref_convex());
    bgeot::pgeometric_trans pgt = mf->linked_mesh().trans_of_convex(cv);
    

    //    mexPrintf("convex %d, nb_pts=%d\n", cv, cv_ref->nb_points());

    eval_sub_nodes(Nrefine, cv_ref, pt);


    std::vector<getfem::scalar_type> pt_val(pt.size());
    interpolate_on_convex_ref(mf, cv, pt, U, pt_val);

    /* apply the geometric transformation to the points, in order to
       find their real location on the mesh */
    for (std::vector<base_node>::iterator it = pt.begin();
	 it < pt.end(); it++) {
      base_node P(it->size()); P.fill(0.0);

      for (size_type j = 0; j < pgt->nb_points(); ++j) {
	P.addmul(pgt->poly_vector()[j].eval(it->begin()),
		 mf->linked_mesh().points_of_convex(cv)[j]);
      }
      //      mexPrintf("%d %d [%f %f]->[%f %f]\n", cv, it->size(), (*it)[0], (*it)[1], P[0], P[1]);

      *it = P;
    }
    
    
      /*
	.      pt[0]
	.        /\
	.       /  \
	. pt[2]/____\pt[1]
	
	refinment:

	.         0     <- layer 0
	.        111    <- layer 1 etc..
	.       22222
	
      */

    /* find the three nodes of the each sub-triangle */
    for (int layer = 0; layer < Nrefine; layer++) {
      for (int itri = 0; itri < layer*2+1; itri++, refined_tri_cnt++) {
	size_type a,b,c;

	if ((itri & 1) == 0) {
	  /*
	    .           A
	    .          /\
	    .       C /__\ B
	  */
	  a = (layer*(layer+1))/2 + itri/2;
	  b = a + layer+1; c = b+1;
	} else {
	  /*
	    .       B ____ C
	    .         \  / 
	    .          \/
	    .           A
	  */
	  b = (layer*(layer+1))/2 + itri/2;
	  c = b+1;
	  a = c+layer+1;
	}

	assert(a < pt.size());
	assert(b < pt.size());
	assert(c < pt.size());
	/*
	mexPrintf("sous-triangle %d [%d] : [%f,%f] - [%f,%f] - [%f,%f] - [%f,%f,%f]\n",
		  refined_tri_cnt, cv_cnt, pt[a][0], pt[a][1], pt[b][0], pt[b][1], pt[c][0], pt[c][1],
		  pt_val[a], pt_val[b], pt_val[c]);
	*/
	out[refined_tri_cnt*9+0] = pt[a][0];
	out[refined_tri_cnt*9+1] = pt[a][1];
	out[refined_tri_cnt*9+2] = pt[b][0];
	out[refined_tri_cnt*9+3] = pt[b][1];
	out[refined_tri_cnt*9+4] = pt[c][0];
	out[refined_tri_cnt*9+5] = pt[c][1];
	out[refined_tri_cnt*9+6] = pt_val[a];
	out[refined_tri_cnt*9+7] = pt_val[b]; 
	out[refined_tri_cnt*9+8] = pt_val[c];
      }
    }
    //    mexPrintf("ok\n");
  }
}
