/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom) version 1.0                        */
/* File    : gf_get_faces_boundary_numbers                                 */
/*     				          		     */
/*                                                                         */
/* Date : March 2002001.                                                   */
/* Author : J. Pommier pommier@gmm.insa-tlse.fr                            */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "matlabint_misc.h"

using namespace getfem;

/**@name B = gf_get_faces_boundary_numbers(mesh_fem mf)
 * 
 * return a list of faces, with their boundary numbers
 * B is a (3 or more)x(nb_total_faces) matrix
 * row 1 is the convex number
 * row 2 is the face number
 * row 3 is the boundary number of that face
 */
//@{
//@}
void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  mesh_fem *mf;
  int warn_cnt  = 0;
  int Nrefine = 0;
  double *Ux, *Uy;

  if (nrhs != 1) 
    mexErrMsgTxt("Wrong number of input argument. [arg1 : mesh_fem]");
  else if (nlhs > 1)
    mexErrMsgTxt("Wrong number of output argument.");

  mf = check_for_mesh_fem_arg(prhs[0]);

  dal::bit_vector nn = mf->convex_index();
  /* count total number of faces */
  int total_nb_faces=0;
  //int faces_max_nb_vertices=0;
  size_type cv;
  for (cv << nn; cv != ST_NIL; cv << nn) {
    int nf = mf->linked_mesh().structure_of_convex(cv)->nb_faces();
    total_nb_faces += nf;
    /*    for (int i = 0; i < nf; i++) {
      faces_max_nb_vertices = max<int>(faces_max_nb_vertices, 
				       mf->linked_mesh().structure_of_convex(cv)->nb_points_of_face(i));
				       }
    */
  }
  
  assert(total_nb_faces>0);
  

  int nbrow = 3;
  plhs[0] = mxCreateNumericMatrix(nbrow, 
				  total_nb_faces, mxINT32_CLASS, mxREAL); assert(plhs[0]!=0);
  
  dal::int32_type *out = static_cast<dal::int32_type*>(mxGetData(plhs[0])); assert(out!=0);

  nn = mf->convex_index();
  int fcnt = 0;
  const dal::bit_vector &blist = mf->get_valid_boundaries();
  for (cv << nn; cv != ST_NIL; cv << nn) {
    int nf = mf->linked_mesh().structure_of_convex(cv)->nb_faces();

    for (int i = 0; i < nf; i++) {
      out[(fcnt+i)*nbrow+0] = cv+1; /* +1 for matlab indices */
      out[(fcnt+i)*nbrow+1] = i+1;
      out[(fcnt+i)*nbrow+2] = -1;
    }

    /* loop of boudary numbers */
    for (dal::bit_vector::const_iterator bbit=blist.begin(); bbit != blist.end(); bbit++) {
      if (!*bbit) continue;
      size_type bb = bbit.index();
      if (!mf->is_convex_on_boundary(cv,bb)) continue;
      const dal::bit_vector &faces_on_bound = mf->faces_of_convex_on_boundary(cv,bb);

      /* loop over faces belonging to the boundary */
      for (dal::bit_vector::const_iterator bbfit=faces_on_bound.begin();
	   bbfit != faces_on_bound.end(); bbfit++) {
	if (!*bbfit) continue;
	size_type bbf = bbfit.index();

	out[(fcnt+bbf)*nbrow+2] = bb;
      }
    }
    fcnt+=nf;
  }
}
