/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom) version 1.0                        */
/* File    : gf_get_deformed_mesh_points                                   */
/*     				          		     */
/*                                                                         */
/* Date : March 2002001.                                                   */
/* Author : J. Pommier pommier@gmm.insa-tlse.fr                            */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "matlabint_misc.h"

using namespace getfem;




/**@name B = gf_get_deformed_mesh_points(mesh_fem mf, int refine_lvl, [vec ux, vec uy])
 * 
 * get the list of interpolation points of the mesh_fem
 * output: B is a matrix [Nxn_points] containing the point locations
 *
 * use only with 2D meshes
 */
//@{
//@}
void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  mesh_fem *mf;
  int warn_cnt  = 0;
  int Nrefine = 0;
  double *Ux, *Uy;

  if (nrhs != 2 && nrhs != 4) 
    mexErrMsgTxt("Wrong number of input argument : mesh_fem mf, int refine_lvl, [vec ux, vec uy]");
  else if (nlhs > 1)
    mexErrMsgTxt("Wrong number of output argument.");

  mf = check_for_mesh_fem_arg(prhs[0]);
  Nrefine = check_for_scalar_integer_arg(prhs[1], 1, "Number of refinements"); assert(Nrefine>0);

  if (nrhs == 4) {
    Ux = check_for_node_values(prhs[2], mf);
    Uy = check_for_node_values(prhs[3], mf);
  } else {
    Ux = NULL; Uy = NULL;
  }

  dal::bit_vector nn = mf->convex_index();

  size_type cv;
  int total_nb_faces=0;
  for (cv << nn; cv != ST_NIL; cv << nn) {
    int nf = mf->linked_mesh().structure_of_convex(cv)->nb_faces();
    total_nb_faces += nf;
  }

  int nbrow = mf->linked_mesh().dim()*(Nrefine+1);
  plhs[0] = mxCreateDoubleMatrix(nbrow, 
				 total_nb_faces, mxREAL); assert(plhs[0] != 0);
  double *out = mxGetPr(plhs[0]);
  

  nn = mf->convex_index();
  
  std::vector<base_node> pt(Nrefine+1);
  std::vector<base_node> ref_pt(Nrefine+1);
  std::vector<scalar_type> pt_def_x(Nrefine+1);
  std::vector<scalar_type> pt_def_y(Nrefine+1);

  int fcnt = 0;
  for (cv << nn; cv != ST_NIL; cv << nn) {
    for (size_type f = 0; f < mf->linked_mesh().structure_of_convex(cv)->nb_faces(); f++, fcnt++) {
      if (mf->linked_mesh().structure_of_convex(cv)->nb_points_of_face(f) != 2) {
	mexPrintf("warning, face %d of convex %d contains %d points (2 expected)\n", 
		  f, cv, mf->linked_mesh().structure_of_convex(cv)->nb_points_of_face(f));
      }
      size_type i1  = mf->ind_points_of_face_of_convex(cv, f)[0];
      size_type i2  = mf->ind_points_of_face_of_convex(cv, f)[1];
      size_type ri1 = mf->structure_of_convex(cv)->ind_points_of_face(f)[0];
      size_type ri2 = mf->structure_of_convex(cv)->ind_points_of_face(f)[1];

      base_node P1  = mf->point_of_dof(i1); // node located on the (deformed) convex
      base_node P2  = mf->point_of_dof(i2); // node located on the (deformed) convex
      base_node rP1 = mf->fem_of_element(cv)->node_of_dof(ri1); // node location on the reference convex
      base_node rP2 = mf->fem_of_element(cv)->node_of_dof(ri2); // node location on the reference convex
      
      for (int i=0; i <= Nrefine; i++) {
	pt[i] = P1 + (P2-P1)*(i/(scalar_type)(Nrefine));
	ref_pt[i] = rP1 + (rP2-rP1)*(i/(scalar_type)(Nrefine));
      }
      if (Ux && Uy) {
	interpolate_on_convex_ref(mf, cv, ref_pt, Ux, pt_def_x);
	interpolate_on_convex_ref(mf, cv, ref_pt, Uy, pt_def_y);
	for (int i=0; i < pt.size(); i++) {
	  pt[i][0] += pt_def_x[i];
	  pt[i][1] += pt_def_y[i];
	}
      }
      for (int i=0; i < pt.size(); i++) {
	std::copy(pt[i].begin(), pt[i].end(), out);
      }
    }
  }
  assert(fcnt == total_nb_faces);
  assert(out == mxGetPr(plhs[0]) + total_nb_faces*nbrow);
}


