/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom)                                    */
/* File    : gf_compute_gradient.C                                         */
/*     								           */
/*                                                                         */
/* Date : June 2002.                                                       */
/* Author : J. Pommier pommier@gmm.insa-tlse.fr                            */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "mex.h"
#include "matlabint_misc.h"
#include <getfem_derivatives.h>
 
using namespace getfem;

static void
error_for_non_lagrange_elements(mesh_fem &mf)
{
  dal::bit_vector nn = mf.convex_index();

  size_type cv, cnt=0, cnt_equiv, total=0;
  for (cv << nn; cv != ST_NIL; cv << nn) {
    if (!mf.fem_of_element(cv)->is_lagrange()) cnt++; 
    total++;
  }
  if (cnt) {
    mexPrintf("Warning: %d elements on %d are NOT lagrange elements\n", cnt, total);
    mexErrMsgTxt("sorry");
  }
}


/**@name DU = gf_compute_gradient(meshfem mf_u, meshfem mf_grad_u, VEC U);
 * compute the gradient of the field $U$ defined on meshfem $mf_u$. The gradient is 
 * interpolated on the meshfem $mf_grad_u$, and returned in DU.
 */
//@{
//@}
void toto(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  getfem::mesh_fem *mf_U=NULL, *mf_DU=NULL;

  VECbidon vec_U, vec_DU;
  size_type dim;

  if (nrhs != 3) 
    mexErrMsgTxt("Wrong number of input argument.");
  else if (nlhs > 1)
    mexErrMsgTxt("Wrong number of output argument.");
  
  mf_U  = check_for_mesh_fem_arg(prhs[0]);
  mf_DU = check_for_mesh_fem_arg(prhs[1]);
  vec_U.ve = check_for_node_values(prhs[2], mf_U, 0, &dim);
  
  error_for_non_lagrange_elements(*mf_DU);
  
  plhs[0] = mxCreateDoubleMatrix(dim*mf_U->linked_mesh().dim(), mf_DU->nb_dof(), mxREAL); assert(plhs[0] != 0);
  vec_DU.ve = mxGetPr(plhs[0]);
  compute_gradient<VECbidon>(*mf_U, *mf_DU, vec_U, vec_DU, dim);
}

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  matlabint::catch_errors(nlhs, plhs, nrhs, prhs, toto);
}

