/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom)                                    */
/* File    : gf_cartesian_mesh.C                                           */
/*     								           */
/*                                                                         */
/* Date : February 2002001.                                                */
/* Author : J. Pommier pommier@gmm.insa-tlse.fr                            */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "mex.h"
#include <matlabint_mesh.h>
#include <matlabint_misc.h>

using namespace getfem;

/**@name gf_cartesian_mesh(m,x[,y[,z,..]])
 * builds quickly a cartesian mesh in a GETFEM mesh 'm'
 */
//@{
//@}

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  if (nrhs < 2) {
    mexErrMsgTxt("Too few arguments");
  } else if (nlhs != 0) {
    mexErrMsgTxt("Too much output arguments");
  }
  
  getfem_mesh *pmesh = check_for_mesh_arg(prhs[0]); assert(pmesh != 0);

  size_type dim = nrhs-1;

  if (dim > 32) {
    mexErrMsgTxt("I think you're doing something stupid");
  }

  /* check for vector arguments */
  for (int i = 0; i < dim; i++) {
    if ((mxGetN(prhs[i+1]) != 1 && mxGetM(prhs[i+1]) != 1) ||
	(mxGetN(prhs[i+1]) == 0 && mxGetM(prhs[i+1]) == 0)) {
      mexErrMsgTxt("supplied arguments should be vectors");
    }
  }

  /* read number of points along each dimension */
  std::vector<size_type> npts(dim);
  std::vector<double*> ppos(dim);

  dal::uint32_type grid_npoints=1, grid_nconvex=1;
  for (int i = 0; i < dim; i++) { 
    npts[i] = std::max<int>(mxGetN(prhs[i+1]), mxGetM(prhs[i+1]));
    ppos[i] = mxGetPr(prhs[i+1]);
    grid_npoints *= npts[i];
    grid_nconvex *= (npts[i]-1);
  }
  /*
  mexPrintf("creating a ");
  for (int i=0; i < dim; i++) mexPrintf("%d%s", npts[i], i != dim-1 ? "x" : " mesh..\n");
  */
  pmesh->clear();

  /* add the points in 'fortran style' order */
  base_node pt(dim);
  for (dal::uint32_type i=0; i < grid_npoints; i++) {
    dal::uint32_type k = i;    
    for (size_type j = 0; j < dim; j++) {
      pt[j] = ppos[j][k % (npts[j])];
      k /= (npts[j]);
    }
    int id_pt = pmesh->add_point(pt);
    if (id_pt != i) {
      mexPrintf("something has changed in getfem, you need to reconsider gf_cartesian_mesh\n");
      mexPrintf("for point %i, the index is %d\n", i, id_pt);
      mexErrMsgTxt("sorry");
    }
  }
  
  std::vector<int> ipt(dim);
  std::vector<base_node> pts(1 << dim+1);

  
  /* add the convexes */
  for (dal::uint32_type i=0; i < grid_nconvex; i++) {
    dal::uint32_type k = i;

    //    mexPrintf("convex %d/%d (k=%d)\n", i, grid_nconvex,k);
    
    /* find point location */
    for (size_type j = 0; j < dim; j++) {
      ipt[j] = k % (npts[j]-1);
      //      mexPrintf("ipt[%d]=%d, k=%d\n", j, ipt[i], k);
      k /= (npts[j]-1);
    }

    /* build the vertices list */
    for (size_type j = 0; j < (1<<dim); j++) {
      //      mexPrintf("[");
      pts[j].resize(dim);
      for (dal::uint32_type d=0; d < dim; d++) {
	if ((j >> d) & 1) {
	  pts[j][d] = ppos[d][ipt[d]+1];
	} else {
	  pts[j][d] = ppos[d][ipt[d]];
	}
	//	mexPrintf("%s%f", d==0 ? "" : "," ,pts[j][d]);
      }
      //mexPrintf("], ");
    }
    //mexPrintf("\n");
    pmesh->add_parallelepiped_by_points(dim, pts.begin());
  }
}
