/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom) version 1.0                        */
/* File    : assem_general_boundary_condition.C : .                        */
/*     				          		     */
/*                                                                         */
/* Date : March 2002001.                                                   */
/* Author : J. Pommier pommier@gmm.insa-tlse.fr                            */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "matlabint_misc.h"

using namespace getfem;


/**@name [Q,G,H,R] = assem_general_boundary_condition(boundary_num, mesh_fem mf_u, mesh_fem mf_d, mat q, vec g, vec h, vec r)
 * 
 * assemblate general boundary conditions (as specified in the PDE Toolbox manual, sec 3-15, 4-9, 4-10
 * the matrices q,g,h,r should have the same number n of columns, which is the number of degree of freedom
 * of mf_d, multiplied by the dimension of the unkown u, stored in the following order:
 * Q = TRANSPOSE[q11_x, q21_x, q12_x, q22_x; <--- dof #1, x component
 *               q11_y, q21_y, q12_y, q22_y; <--- dof #1, y component
 *               q11_x, q21_x, q12_x, q22_x; <--- dof #2, x component
 *               q11_y, q21_y, q12_y, q22_y; <--- dof #2, y component
 *                ...]
 *
 * output: boundary condition matrices
 */
//@{
//@}

#define MEXERROR mexErrMsgTxt(" ==> error in assem_general_boundary_conditions.C <==");


/* check that the dimensions are coherent */
static double*
check_bc_dimensions(const mxArray *arg, int nb_dof, int is_vec_data, int &dim, 
		    int arg_num, const char *arg_name) {
  int n,m,d;
  assert(nb_dof>0);
  n = mxGetN(arg); m = mxGetM(arg);
  
  if ( n != nb_dof ) {
    mexPrintf("wrong number of columns (%d) for argument %d (%s)\n", n, arg_num, arg_name);
    mexPrintf("the number of columns is expected to be the numbre of degree of freedom (%d) of the data mesh_fem\n", nb_dof);
    MEXERROR;
  }
  if (is_vec_data) {
    if (m != dim) {
      mexPrintf("wrong number of rows (%d) for argument %d (%s)\n", m, arg_num, arg_name);
      mexPrintf("this argument should contain vector data in each column, so it should have %d rows\n", dim);
      MEXERROR;
    }
  } else { 
    if (m != dim*dim) {
      mexPrintf("wrong number of rows (%d) for argument %d (%s)\n", m, arg_num, arg_name);
      mexPrintf("this argument should contain [%dx%d] matrix data in each column, so it should have %d rows\n", dim, dim, dim*dim);
      mexPrintf("this data should be stored in this order (for a 2x2 matrix): \n"
		"M(1,1), M(2,1), M(1,2), M(2,2)\n");
      MEXERROR;
    }
  }
  if (!mxIsDouble(arg)) {
    mexPrintf("argument %d (%s) is expected to contain double precision data\n", arg_num, arg_name);
    MEXERROR;
  }
  return mxGetPr(arg);
}

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  getfem::mesh_fem *mf_u, *mf_d;
  double *pq, *pg, *ph, *pr;
  int mesh_dim, u_dim;
  const mxArray *mx_q, *mx_g, *mx_h, *mx_r; /* input data */
  mxArray *mx_Q, *mx_G, *mx_H, *mx_R; /* output (assembled) data */
  int boundary_num;

  if (nrhs != 7) 
    mexErrMsgTxt("Wrong number of input argument (7 needed :boundary_num, mesh_fem mf_u, mesh_fem mf_data, q, g, h, r)");
  else if (nlhs != 4)
    mexErrMsgTxt("Wrong number of output argument (4 needed : Q,G,H,R)");

  boundary_num = check_for_scalar_integer_arg(prhs[0], 1, "boundary_num");
  mf_u = check_for_mesh_fem_arg(prhs[1]);
  mf_d = check_for_mesh_fem_arg(prhs[2]);
  mx_q = prhs[3]; mx_g = prhs[4]; mx_h = prhs[5]; mx_r = prhs[6];

  u_dim = mxGetM(mx_g);
  pq = check_bc_dimensions(mx_q, mf_d->nb_dof(), 0, u_dim, 4, "q"); assert(pq != NULL);
  pg = check_bc_dimensions(mx_g, mf_d->nb_dof(), 1, u_dim, 5, "g"); assert(pg != NULL);
  ph = check_bc_dimensions(mx_h, mf_d->nb_dof(), 0, u_dim, 6, "h"); assert(ph != NULL);
  pr = check_bc_dimensions(mx_r, mf_d->nb_dof(), 1, u_dim, 7, "r"); assert(pr != NULL);

  mesh_dim = mf_u->linked_mesh().dim();
  if (mesh_dim != mf_d->linked_mesh().dim()) {
    mexErrMsgTxt("?? what are you doing?");
  }


  mx_G = mxCreateDoubleMatrix(u_dim*mf_u->nb_dof(), 1, mxREAL); 
  mx_R = mxCreateDoubleMatrix(u_dim*mf_u->nb_dof(), 1, mxREAL);
  
  VECbidon vec_G(mxGetPr(mx_G));
  VECbidon vec_R(mxGetPr(mx_R));
  VECbidon vec_g(mxGetPr(mx_g));
  VECbidon vec_r(mxGetPr(mx_r));

  getfem::assembling_Neumann_condition<VECbidon,VECbidon>(vec_G,
							  *mf_u, boundary_num,
							  *mf_d, 
							  vec_g,
							  u_dim);
  getfem::assembling_Neumann_condition<VECbidon,VECbidon>(vec_R,
							  *mf_u, boundary_num,
							  *mf_d, 
							  vec_r,
							  u_dim);

  bgeot::smatrix<getfem::scalar_type> mat_Q(mf_u->nb_dof() * u_dim, 
				   mf_u->nb_dof() * u_dim);
  bgeot::smatrix<getfem::scalar_type> mat_H(mf_u->nb_dof() * u_dim, 
				   mf_u->nb_dof() * u_dim);

  VECbidon vec_q(mxGetPr(mx_q));
  VECbidon vec_h(mxGetPr(mx_h));

  getfem::assembling_boundary_qu_term<bgeot::smatrix<getfem::scalar_type>, VECbidon> 
    (mat_Q, *mf_u, boundary_num, *mf_d, vec_q, u_dim);
  getfem::assembling_boundary_qu_term<bgeot::smatrix<getfem::scalar_type>, VECbidon> 
    (mat_H, *mf_u, boundary_num, *mf_d, vec_h, u_dim);
  
  mx_Q = convert_to_mxSparse(mat_Q);
  mx_H = convert_to_mxSparse(mat_H);

  plhs[0] = mx_Q;
  plhs[1] = mx_G;
  plhs[2] = mx_H;
  plhs[3] = mx_R;
}
