/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom)                                    */
/* File    : add_to_intset.C : add elements to an intset.                  */
/*     									   */
/*                                                                         */
/* Date : September 26, 2001.                                              */
/* Author : Yves Renard, Yves.Renard@gmm.insa-tlse.fr                      */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "mex.h"
#include <matlabint_intset.h>

/**@name add_to_intset(i, a) or add_to_intset(i, a, b) 
 * Add to an existing intset i, the integer a or integer intervall [a,b].
 */
//@{
//@}

#define id_type matlabint::id_type

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  if (nrhs <= 1 || nrhs > 3)
    mexErrMsgTxt("Wrong number of input arguments.");
  else if (nlhs != 0)
    mexErrMsgTxt("No output argument needed.");
  id_type *p = (id_type *)(mxGetData(prhs[0]));
  if (mxGetClassID(prhs[0]) != mxUINT32_CLASS 
      || (*p & 0xFF000000) != MATLABINT_INTSET_MASK)
    mexErrMsgTxt("First input argument should be a descriptor returned by new_intset.");
  int i = *p - MATLABINT_INTSET_MASK;
  if (!matlabint::exists_intset(i))
    mexErrMsgTxt("This intset no longer exists.");

  void *q1 = mxGetData(prhs[1]), *q2;
  id_type a, b = 1;
  if (nrhs == 3) q2 = mxGetData(prhs[2]);
  
  switch (mxGetClassID(prhs[1]))
  {
    case mxCHAR_CLASS   : a = id_type(*((char *)q1));             break;
    case mxDOUBLE_CLASS : a = id_type(*((double *)q1));           break;
    case mxSINGLE_CLASS : a = id_type(*((float *)q1));            break;
    case mxINT8_CLASS   : a = id_type(*((dal::int8_type *)q1));   break;
    case mxUINT8_CLASS  : a = id_type(*((dal::uint8_type *)q1));  break;
    case mxINT16_CLASS  : a = id_type(*((dal::int16_type *)q1));  break;
    case mxUINT16_CLASS : a = id_type(*((dal::uint16_type *)q1)); break;
    case mxINT32_CLASS  : a = id_type(*((dal::int32_type *)q1));  break;
    case mxUINT32_CLASS : a = id_type(*((dal::uint32_type *)q1)); break;
    default : mexErrMsgTxt("Second argument should be numeric.");
  }

  if (nrhs == 3) switch (mxGetClassID(prhs[2]))
  {
    case mxCHAR_CLASS   : b = id_type(*((char *)q2));             break;
    case mxDOUBLE_CLASS : b = id_type(*((double *)q2));           break;
    case mxSINGLE_CLASS : b = id_type(*((float *)q2));            break;
    case mxINT8_CLASS   : b = id_type(*((dal::int8_type *)q2));   break;
    case mxUINT8_CLASS  : b = id_type(*((dal::uint8_type *)q2));  break;
    case mxINT16_CLASS  : b = id_type(*((dal::int16_type *)q2));  break;
    case mxUINT16_CLASS : b = id_type(*((dal::uint16_type *)q2)); break;
    case mxINT32_CLASS  : b = id_type(*((dal::int32_type *)q2));  break;
    case mxUINT32_CLASS : b = id_type(*((dal::uint32_type *)q2)); break;
    default : mexErrMsgTxt("Second argument should be numeric.");
  }
  if (a & 0xF0000000 || b & 0xF0000000 || a == 0 || b == 0)
    mexErrMsgTxt("One of the arguments is out of range.");
  if (nrhs == 3) matlabint::intset(i).add(a-1,b-1);
  else matlabint::intset(i).add(a-1);
 
}
