/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom)                                    */
/* File    : add_point_to_mesh.C : add a point to a mesh variable.         */
/*     									   */
/*                                                                         */
/* Date : September 27, 2001.                                              */
/* Author : Yves Renard, Yves.Renard@gmm.insa-tlse.fr                      */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "mex.h"
#include <matlabint_mesh.h>

/**@name V = add_point_to_mesh(i, P1, P2, ...) 
 * Add to an existing mesh i, the points P1, P2, ... and return in V
 * the vector of indices.
 */
//@{
//@}


void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  if (nrhs <= 1)
    mexErrMsgTxt("Wrong number of input arguments.");
  else if (nlhs > 1)
    mexErrMsgTxt("Wrong number of output argument.");
  matlabint::id_type *p = (matlabint::id_type *)(mxGetData(prhs[0]));
  if (mxGetClassID(prhs[0]) != mxUINT32_CLASS 
      || (*p & 0xFF000000) != MATLABINT_MESH_MASK)
    mexErrMsgTxt("First input argument should be a descriptor returned by new_mesh.");
  int i = *p - MATLABINT_MESH_MASK;
  if (!matlabint::exists_mesh(i))
    mexErrMsgTxt("This mesh no longer exists.");
  getfem::getfem_mesh *pmesh = &(matlabint::mesh(i));

  int aux = nrhs - 1;
  plhs[0] = mxCreateNumericArray(1, &aux, mxUINT32_CLASS, mxREAL);
  matlabint::id_type *qs = (matlabint::id_type *)(mxGetData(plhs[0]));

  for (int n = 1; n < nrhs; ++n)
  {
    float *q = (float *)(mxGetData(prhs[n]));
    int nc = mxGetN(prhs[n]) * mxGetM(prhs[n]);
    
    if ((pmesh->dim() != bgeot::dim_type(-1) && pmesh->dim() != nc) || nc <= 0)
      mexErrMsgTxt("Incorrect point(s) dimension(s).");

    

    bgeot::base_node PT(nc);
    for (int j = 0; j < nc; ++j, ++q)
    {
      switch (mxGetClassID(prhs[n]))
      {
        case mxDOUBLE_CLASS : PT[j] = (*((double *)q)); ++q;   break;
        case mxSINGLE_CLASS : PT[j] = (double)(*((float *)q)); break;
        default : mexErrMsgTxt("Incorrect arguments.");
      }
    }
    *qs++ = pmesh->add_point(PT) + 1;
  }
  
}
