// -*- c++ -*- (enables emacs c++ mode)
//========================================================================
//
// Copyright (C) 2006-2006 Yves Renard, Julien Pommier.
//
// This file is a part of GETFEM++
//
// Getfem++ is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; version 2.1 of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
// You should have received a copy of the GNU Lesser General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
//========================================================================

#include <getfemint_misc.h>
#include <getfemint_workspace.h>
#include <getfemint_mesh_im.h>
#include <getfemint_mesh.h>

using namespace getfemint;

void gf_mesh_im_set_integ(getfem::mesh_im *mim, getfemint::mexargs_in& in);

/*MLABCOM
  FUNCTION MIM = gf_mesh_im(...)

  General constructor for @tmim object (integration methods on a mesh).

  * gf_mesh_im(mesh M [{integ IM|int IM_DEGREE}])

  Return a getfem handle to the newly created mesh_im object. For
  convenience, optional arguments (IM or IM_DEGREE) can be provided,
  in that case a call to gf_mesh_im_set(mim, 'integ', ..) is issued with these
  arguments.

  @INIT MESHIM:INIT('load')
  @INIT MESHIM:INIT('from string')
  @INIT MESHIM:INIT('clone')

  $Id: gf_mesh_im.cc,v 1.4 2006/03/28 10:06:35 pommier Exp $
MLABCOM*/

void gf_mesh_im(getfemint::mexargs_in& in, getfemint::mexargs_out& out)
{
  if (in.narg() < 1) THROW_BADARG("Wrong number of input arguments");
  getfemint_mesh *mm = NULL;
  getfemint_mesh_im *mim = NULL;
  if (in.front().is_string()) {
    std::string cmd = in.pop().to_string();
    if (check_cmd(cmd, "load", in, out, 1, 2, 0, 1)) {
      /*@INIT MESHIM:INIT('load', fname[, @tmesh M]) 
	Load a @tmim from a file. 
	
	If the mesh M is not supplied (this kind of file does not
	store the mesh), then it is read from the file and its
	descriptor is returned as the second output argument. @*/
      std::string fname = in.pop().to_string();
      if (in.remaining()) mm = in.pop().to_getfemint_mesh();
      else {
	mm = new getfemint_mesh();
	mm->mesh().read_from_file(fname);
	workspace().push_object(mm);
      }
      mim = new getfemint_mesh_im(mm);
      mim->mesh_im().read_from_file(fname);
    } else if (check_cmd(cmd, "from string", in, out, 1, 2, 0, 1)) {
      /*@INIT MESHIM:INIT('from string', str[, mesh M])
	Create a @tmim object from its string description.
	
	See also MESHIM:GET('char')
	@*/      
      std::stringstream ss(in.pop().to_string());
      if (in.remaining()) mm = in.pop().to_getfemint_mesh();
      else {
	mm = new getfemint_mesh();
	mm->mesh().read_from_file(ss);
	workspace().push_object(mm);
      }
      mim = new getfemint_mesh_im(mm);
      mim->mesh_im().read_from_file(ss);
    } else if (check_cmd(cmd, "clone", in, out, 1, 1, 0, 1)) {
      /*@INIT MESHIM:INIT('clone', @tmim MIM2)
	Create a copy of a @tmim.
	@*/
      getfemint_mesh_im *mim2 = in.pop().to_getfemint_mesh_im();
      mm = object_to_mesh(workspace().object(mim2->linked_mesh_id()));
      mim = new getfemint_mesh_im(mm);
      std::stringstream ss; /* not very elegant ! */
      mim2->mesh_im().write_to_file(ss);
      mim->mesh_im().read_from_file(ss);
    } else bad_cmd(cmd);
    out.pop().from_object_id(workspace().push_object(mim), MESHIM_CLASS_ID);      
  } else {
    if (!out.narg_in_range(1, 1)) THROW_BADARG("Wrong number of output arguments");
    mm = in.pop().to_getfemint_mesh();
    mim = new getfemint_mesh_im(mm);
    out.pop().from_object_id(workspace().push_object(mim), MESHIM_CLASS_ID);
    if (in.remaining()) {
      gf_mesh_im_set_integ(&mim->mesh_im(), in);
    }
    if (in.remaining()) THROW_BADARG("Wrong number of input arguments");
  }
  workspace().set_dependance(mim, mm);
}

