// -*- c++ -*- (enables emacs c++ mode)
//========================================================================
//
// Copyright (C) 2006-2006 Yves Renard, Julien Pommier.
//
// This file is a part of GETFEM++
//
// Getfem++ is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; version 2.1 of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
// You should have received a copy of the GNU Lesser General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
//========================================================================

#include <getfemint_misc.h>
#include <getfemint_workspace.h>
#include <getfemint_mesh_fem.h>
#include <getfemint_mesh.h>

using namespace getfemint;

/*MLABCOM
  FUNCTION MF = gf_mesh_fem(...)

  General constructor for mesh_fem objects (Finite Element basis
  functions on a mesh).

  * gf_mesh_fem(mesh M [, int Qdim=1])

  Return a getfem handle to the newly created mesh_fem object.

  @INIT MESHFEM:INIT('load')
  @INIT MESHFEM:INIT('from string')
  @INIT MESHFEM:INIT('clone')

  $Id: gf_mesh_fem.cc,v 1.4 2006/03/28 10:06:35 pommier Exp $
MLABCOM*/

void gf_mesh_fem(getfemint::mexargs_in& in, getfemint::mexargs_out& out)
{
  if (in.narg() < 1) THROW_BADARG("Wrong number of input arguments");
  getfemint_mesh *mm = NULL;
  getfemint_mesh_fem *mmf = NULL;
  unsigned q_dim = 1;
  if (in.front().is_string()) {
    std::string cmd = in.pop().to_string();
    if (check_cmd(cmd, "load", in, out, 1, 2, 0, 1)) {
      /*@INIT MESHFEM:INIT('load', fname[, @tmesh M]) 
	Load a @tmf from a file. 
	
	If the mesh M is not supplied (this kind of file does not
	store the mesh), then it is read from the file and its
	descriptor is returned as the second output argument. @*/
      std::string fname = in.pop().to_string();
      if (in.remaining()) mm = in.pop().to_getfemint_mesh();
      else {
	mm = new getfemint_mesh();
	mm->mesh().read_from_file(fname);
	workspace().push_object(mm);
      }
      mmf = new getfemint_mesh_fem(mm,q_dim);
      mmf->mesh_fem().read_from_file(fname);
    } else if (check_cmd(cmd, "from string", in, out, 1, 2, 0, 1)) {
      /*@INIT MESHFEM:INIT('from string', str[, mesh M])
	Create a mesh_fem object from its string description.
	
	See also MESHFEM:GET('char')
	@*/      
      std::stringstream ss(in.pop().to_string());
      if (in.remaining()) mm = in.pop().to_getfemint_mesh();
      else {
	mm = new getfemint_mesh();
	mm->mesh().read_from_file(ss);
	workspace().push_object(mm);
      }
      mmf = new getfemint_mesh_fem(mm,q_dim);
      mmf->mesh_fem().read_from_file(ss);
    } else if (check_cmd(cmd, "clone", in, out, 1, 1, 0, 1)) {
      /*@INIT MESHFEM:INIT('clone', @tmf MF2)
	Create a copy of a mesh_fem.
	@*/
      getfemint_mesh_fem *mmf2 = in.pop().to_getfemint_mesh_fem();
      mm = object_to_mesh(workspace().object(mmf2->linked_mesh_id()));
      mmf = new getfemint_mesh_fem(mm,q_dim);
      std::stringstream ss; 
      mmf2->mesh_fem().write_to_file(ss);
      mmf->mesh_fem().read_from_file(ss);
    } else bad_cmd(cmd);
    out.pop().from_object_id(workspace().push_object(mmf), MESHFEM_CLASS_ID);      
  } else {
    if (in.narg()  >  3) THROW_BADARG("Wrong number of input arguments");
    if (!out.narg_in_range(1, 1)) THROW_BADARG("Wrong number of output arguments");
    mm = in.pop().to_getfemint_mesh();
    if (in.remaining()) q_dim = in.pop().to_integer(0,256);
    if (in.remaining()) {
      unsigned q_dim2 = in.pop().to_integer(0,256);
      mmf = new getfemint_mesh_fem(mm,q_dim * q_dim2); 
      mmf->mesh_fem().set_qdim_mn(q_dim, q_dim2);
    } else
      mmf = new getfemint_mesh_fem(mm,q_dim);
    out.pop().from_object_id(workspace().push_object(mmf), MESHFEM_CLASS_ID);
  }
  workspace().set_dependance(mmf, mm);
}

