#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-
#
#
# Getfem-Python interface
#
# Date : March, 2004.
# Author : Julien Pommier, pommier@gmm.insa-tlse.fr
#                                                       
# This file is a part of GETFEM++                                         
#                                                                         
# Getfem++  is  free software;  you  can  redistribute  it  and/or modify it
# under  the  terms  of the  GNU  Lesser General Public License as published
# by  the  Free Software Foundation;  either version 2.1 of the License,  or
# (at your option) any later version.
# This program  is  distributed  in  the  hope  that it will be useful,  but
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or  FITNESS  FOR  A PARTICULAR PURPOSE.  See the GNU Lesser General Public
# License for more details.
# You  should  have received a copy of the GNU Lesser General Public License
# along  with  this program;  if not, write to the Free Software Foundation,
# Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301, USA.
#                                                                         
"""Getfem-interface classes.

Provides access to the pseudo-objects exported by the getfem-python interface.
"""


__version__ = "$Revision: 2719 $"
# $Source: /var/lib/cvs/getfem_matlab/src/python/getfem.base.py,v $

import sys
#import Numeric
import numarray

sys.path.insert(0,'build/lib.linux-i686-2.3/');
from _getfem import *
obj_count = {}
getfem('workspace','clear all')

def generic_constructor(self,clname,*args):
    """Internal function -- acts as a constructor for all getfem objects."""
#    print 'generic_constructor.'+clname+'('+str(args)+')'
    if (len(args)==1 and type(args[0]) is GetfemObject):
        self.id = args[0]
    else:
        self.id = getfem_from_constructor(clname,*args)
    obj_count[self.id] = obj_count.get(self.id,0)+1

def generic_destructor(self,destructible=True):
    """Internal function -- acts as a destructor for all getfem objects."""
    if (not hasattr(self,'id')):
        return
#    print "Mesh.__del__       ",self.id,'count=',obj_count[self.id]
    if (obj_count.has_key(self.id)):
        obj_count[self.id] = obj_count[self.id]-1
        if (destructible and obj_count[self.id] == 0):
#            print "effective deletion"
            getfem('delete',self.id)

# stub classes for getfem-interface objects

class Mesh:
    """Class for getfem mesh objects."""
    def __init__(self, *args):
        """General constructor for Mesh objects.

 *  M=Mesh('empty', int dim)
  Create a new empty mesh.
 *  M=Mesh('cartesian', vec X[, vec Y[, vec Z,..]])
  Build quickly a regular mesh of quadrangles, cubes, etc.
 * M=Mesh('regular_simplices', vec X[, vec Y[, vec Z,.., ]]['degree', int
K]['noised'])
  Mesh a n-dimensionnal parallelepipeded with simplices (triangles,
tetrahedrons etc) .  The optional degree may be used to build meshes with non
linear geometric transformations.
 *  M=Mesh('triangles_grid', vec X, vec Y)
  Build quickly a regular mesh of triangles.   This is a very limited and
somehow deprecated function (See also Mesh('ptND'), Mesh('regular_simplices')
and Mesh('cartesian')).
 *  M=Mesh('curved', M0, vec F)
  Build a curved (n+1)-dimensions mesh from a n-dimensions mesh M0.   The
points of the new mesh have one additional coordinate, given by the vector F.
This can be used to obtain meshes for shells. M0 may be a meshfem object, in
that case its linked mesh will be used.
 *  M=Mesh('prismatic', M0, int NLAY)
  Extrude a prismatic mesh M from a mesh M0.   In the additional dimension
there are NLAY layers of elements built from 0 to 1.
 *  M=Mesh('pt2D', mat P, ivec T[, int N])
  Build a mesh from a 2D triangulation.  Each column of P contains a point
coordinate, and each column of T contains the point indices of a triangle. N
is optional and is a zone number. If N is specified then only the zone number
'N' is converted (in that case, T is expected to have 4 rows, the fourth
containing these zone numbers).
 *  M=Mesh('ptND', mat P, imat T)
  Build a mesh from a N-dimensional "triangulation".  Similar function to
'pt2D', for building simplexes meshes from a triangulation given in T, and a
list of points given in P. The dimension of the mesh will be the number of
rows of P, and the dimension of the simplexes will be the number of rows of T.
 *  M=Mesh('load', string FILENAME)
  Load a mesh from a GETFEM++ ascii mesh file. See also Mesh.save(FILENAME).
 *  M=Mesh('from_string', string S)
  Load a mesh from a string description. For example, a string returned by
Mesh.char().
 *  M=Mesh('import', string FORMAT, string FILENAME)
  Import a mesh, FORMAT may be:

 - 'gmsh'   for a mesh created with gmsh ( http://www.geuz.org/gmsh )
 - 'gid'    for a mesh created with GiD  ( http://gid.cimne.upc.es )
 - 'am_fmt' for a mesh created with emc2 (
http://pauillac.inria.fr/cdrom/www/emc2/fra.htm )
 *  M=Mesh('clone', mesh M2)
  Create a copy of a mesh.
        """
        generic_constructor(self,'mesh',*args)
    def __del__(self):
        generic_destructor(self,destructible=True)
    def get(self, *args):
        return getfem('mesh_get',self.id, *args)
    def set(self, *args):
        return getfem('mesh_set',self.id, *args)
    def __str__(self):
        return self.char()
    def __repr__(self):
        return '<getfem.Mesh %dD, %d points, %d convexes, %d bytes>' % \
               (self.dim(),self.nbpts(),self.nbcvs(),self.memsize())

    #@RDATTR MESH:GET('dim')
    def dim(self):
        """Get the dimension of the mesh (2 for a 2D mesh, etc)."""
        return self.get("dim")


    #@GET    MESH:GET('pts')
    def pts(self, PIDLST=None):
        """Return the list of point coordinates of the mesh.\\\\

Synopsis: [mat PT]=Mesh.pts([, ivec PIDLST])
Each column of the returned matrix contains the coordinates of one point.  If
the optional argument PIDLST was given, only the points whose #id is listed in
this vector are returned. Otherwise, the returned matrix will have
Mesh.max_pid() columns, which might be greater than Mesh.nbpts() (if some
points of the mesh have been destroyed and no call to
Mesh.optimize_structure() have been issued).  The columns corresponding to
deleted points will be filled with NaN. You can use Mesh.pid() to filter such
invalid points."""
        return self.get("pts", PIDLST)


    #@RDATTR MESH:GET('nbpts')
    def nbpts(self):
        """Get the number of points of the mesh."""
        return self.get("nbpts")


    #@RDATTR MESH:GET('nbcvs')
    def nbcvs(self):
        """Get the number of convexes of the mesh."""
        return self.get("nbcvs")


    #@GET    MESH:GET('pid')
    def pid(self):
        """Return the list of points #id of the mesh.\\\\

Synopsis: [ivec PID]=Mesh.pid()
Note that their numbering is not supposed to be contiguous from 0 to
Mesh.nbpts()-1,  especially if you destroyed some convexes. You can use
Mesh.optimize_structure() to enforce a contiguous numbering."""
        return self.get("pid")


    #@GET    MESH:GET('cvid')
    def cvid(self):
        """Return the list of all convex #id.\\\\

Synopsis: [CVID] = Mesh.cvid()
Note that their numbering is not supposed to be contiguous from 0 to
Mesh.nbcvs()-1,  especially if some points have been removed from the mesh.
You can use Mesh.optimize_structure() to enforce a contiguous numbering."""
        return self.get("cvid")


    #@GET    MESH:GET('max pid')
    def max_pid(self):
        """Return the maximum #id of all points in the mesh (see 'max cvid')."""
        return self.get("max pid")


    #@GET    MESH:GET('max cvid')
    def max_cvid(self):
        """Return the maximum #id of all convexes in the mesh (see 'max pid')."""
        return self.get("max cvid")


    #@GET    MESH:GET('pid from cvid')
    def pid_from_cvid(self, CVLST=None):
        """Return the points attached to each convex of the mesh.\\\\

Synopsis:  [PID,IDX]=Mesh.pid_from_cvid([,CVLST])
If CVLST is omitted, all the convexes will be considered (equivalent to CVLST
= @Mesh.max_cvid()). IDX is a vector, length(IDX) = length(CVLIST)+1. PID is a
vector containing the concatenated list of points of each convex in cvlst.
Each entry of IDX is the position of the corresponding convex point list in
PID. Hence, for example, the list of points of the second convex is
PID[IDX(2):IDX(3)].\\  If CVLST contains convex #id which do not exist in the
mesh, their point list will be empty."""
        return self.get("pid from cvid", CVLST)


    #@GET    MESH:GET('pid from coords')
    def pid_from_coords(self, PT):
        """Search point #id whose coordinates are listed in PT.\\\\

Synopsis: [ivec PIDLST]=Mesh.pid_from_coords( mat PT)
PT is an array containing a list of point coordinates. On return, PIDLST is a
vector containing points #id for each point found, and -1  -1 for those which
where not found in the mesh."""
        return self.get("pid from coords", PT)


    #@GET    MESH:GET('orphaned pid')
    def orphaned_pid(self):
        """Returns the list of mesh nodes which are not linked to a convex."""
        return self.get("orphaned pid")


    #@GET    MESH:GET('cvid from pid')
    def cvid_from_pid(self, PIDLST):
        """Returns the convex #ids that share the point #ids given in PIDLST."""
        return self.get("cvid from pid", PIDLST)


    #@GET    MESH:GET('faces from pid')
    def faces_from_pid(self, PIDLST):
        """Return the convex faces whose vertex #ids are in PIDLST.\\\\

Synopsis: [imat CVFLST]=Mesh.faces_from_pid( ivec PIDLST)
For a convex face to be returned, EACH of its points have to be listed in
PIDLST. On output, the first row of CVFLST contains the convex number, and the
second row contains the face number (local number in the convex)."""
        return self.get("faces from pid", PIDLST)


    #@GET    MESH:GET('faces from cvid')
    def faces_from_cvid(self, CVLST):
        """Return a list of convexes faces from a list of convex #id.\\\\

Synopsis: [imat CVFLST]=Mesh.faces_from_cvid( ivec CVLST,[ 'merge'])
CVFLST is a two-rows matrix, the first row lists convex #ids, and the second
lists face numbers. The optional argument 'merge' merges faces shared by two
convexes of CVLST."""
        return self.get("faces from cvid", CVLST)


    #@GET    MESH:GET('outer faces')
    def outer_faces(self, CVLST=None):
        """Return the faces which are not shared by two convexes.

Synopsis: [CVFLST]=Mesh.outer_faces([, CVLST])
If CVLST is not given, it basically returns the mesh boundary. If CVLST is
given, it returns the boundary of the convex set whose #ids are listed in
CVLST."""
        return self.get("outer faces", CVLST)


    #@GET    MESH:GET('edges')
    def edges(self, *args):
        """[OBSOLETE FUNCTION! will be removed in a future release]\\\\

Synopsis: [E,C]=Mesh.edges([, CVLST][,'merge'])
Return the list of edges of mesh M for the convexes listed in the row vector
CVLST. E is a 2 x nb_edges matrix containing point indices. If CVLST is
omitted, then the edges of all convexes are returned. If CVLST has two rows
then the first row is supposed to contain convex numbers, and the second face
numbers, of which the edges will be returned.  If 'merge' is indicated, all
common edges of convexes are merged in a single edge.  If the optional output
argument C is specified, it will contain the convex number associated with
each edge."""
        return self.get("edges", *args)


    #@GET    MESH:GET('curved edges')
    def curved_edges(self, N, CVLST=None):
        """[OBSOLETE FUNCTION! will be removed in a future release]\\\\

Synopsis: [E,C]=Mesh.curved_edges( int N [, CVLST])
More sophisticated version of Mesh.edges() designed for curved elements. This
one will return N (N>=2) points of the (curved) edges. With N==2, this is
equivalent to Mesh.edges(). Since the points are no more always part of the
mesh, their coordinates are returned instead of points number, in the array E
which is a [ mesh_dim x 2 x nb_edges ] array.  If the optional output argument
C is specified, it will contain the convex number associated with each edge."""
        return self.get("curved edges", N, CVLST)


    #@GET    MESH:GET('triangulated surface')
    def triangulated_surface(self, Nrefine, CVLIST=None):
        """[OBSOLETE FUNCTION! will be removed in a future release]

Synopsis: [mat T]=Mesh.triangulated_surface( int Nrefine [,CVLIST])
Similar function to Mesh.curved_edges() : split (if necessary, i.e. if the
geometric transformation if non-linear) each face into sub-triangles and
return their coordinates in T (see also compute_eval on P1 tri mesh(mf, U, ))"""
        return self.get("triangulated surface", Nrefine, CVLIST)


    #@GET    MESH:GET('normal of face')
    def normal_of_face(self, CV, F, FPTNUM=None):
        """Evaluates the normal of convex CV, face F at the FPTNUMth point of the face.
If FPTNUM is not specified, then the normal is evaluated at each geometrical
node of the face."""
        return self.get("normal of face", CV, F, FPTNUM)


    #@GET    MESH:GET('normal of faces')
    def normal_of_faces(self, CVFLST):
        """Evaluates (at face centers) the normals of convexes.

Synopsis: [mat N] = Mesh.normal_of_faces( imat CVFLST)
CVFLST is supposed to contain convex numbers in its first row and convex face
number in its second row."""
        return self.get("normal of faces", CVFLST)


    #@GET    MESH:GET('quality')
    def quality(self, CVLST=None):
        """Return an estimation of the quality of each convex (0 <= Q <= 1)."""
        return self.get("quality", CVLST)


    #@GET    MESH:GET('cvstruct')
    def cvstruct(self, CVLST=None):
        """Return an array of the convex structures.

Synopsis: [CVS, ivec CV2STRUC]=Mesh.cvstruct([ivec CVLST])
If CVLST is not given, all convexes are considered. Each convex structure is
listed once in CVS, and CV2STRUC maps the convexes indice in CVLST to the
indice of its structure in CVS."""
        return self.get("cvstruct", CVLST)


    #@GET    MESH:GET('geotrans')
    def geotrans(self, CVLST=None):
        """Returns an array of the geometric transformations.

Synopsis: [GT,GT2STRUCT]=Mesh.geotrans([CVLST])
See also Mesh.cvstruct()."""
        return self.get("geotrans", CVLST)


    #@GET    MESH:GET('regions')
    def regions(self):
        """Return the list of valid regions stored in the mesh."""
        return self.get("regions")


    #@GET    MESH:GET('region')
    def region(self, rnum):
        """Return the list of convexes/faces on the region 'rnum'.

Synopsis: CVFLST = Mesh.region( int rnum)
On output, the first row of I contains the convex numbers, and the second row
contains the face numbers (and -1 when the whole convex is in the region)."""
        return self.get("region", rnum)


    #@GET    MESH:GET('save')
    def save(self, FILENAME):
        """Save the mesh object to an ascii file.

Synopsis: Mesh.save( stringing FILENAME)
This mesh can be restored with Mesh('load', FILENAME)."""
        return self.get("save", FILENAME)


    #@GET    MESH:GET('char')
    def char(self):
        """Output a string description of the mesh."""
        return self.get("char")


    #@GET    MESH:GET('export to vtk')
    def export_to_vtk(self, filename, *args):
        """Exports a mesh to a VTK file .

Synopsis: Mesh.export_to_vtk( string filename, ... [,'ascii'][,'quality'])
If 'quality' is specified, an estimation of the quality of each convex will be
written to the file.  See also MeshFem.export_to_vtk() ,
Slice.export_to_vtk()."""
        return self.get("export to vtk", filename, *args)


    #@GET    MESH:GET('export to dx')
    def export_to_dx(self, filename, *args):
        """Exports a mesh to an OpenDX file.

Synopsis: Mesh.export_to_dx( string filename, ...
[,'ascii'][,'append'][,'as',string name,[,'serie',string
serie_name]][,'edges'])
See also MeshFem.export_to_dx() , Slice.export_to_dx()."""
        return self.get("export to dx", filename, *args)


    #@GET    MESH:GET('memsize')
    def memsize(self):
        """Return the amount of memory (in bytes) used by the mesh."""
        return self.get("memsize")



    #@SET MESH:SET('pts')
    def set_pts(self, P):
        """Replace the coordinates of the mesh points with those given in P."""
        return self.set("pts", P)


    #@SET MESH:SET('add point')
    def add_point(self, PT):
        """Insert new points in the mesh and return their #id.

Synopsis: [IDX]=Mesh.add_point( mat PT)
PT should be an [n x m] matrix , where n is the mesh dimension, and m is the
number of points that will be added to the mesh. On output, IDX contains the
indices of these new points.

 Remark: if some points are already part of the mesh (with a small tolerance
of approximately 1e-8), they won't be inserted again, and IDX will contain the
previously assigned indices of the points."""
        return self.set("add point", PT)


    #@SET MESH:SET('del point')
    def del_point(self, PIDLST):
        """Removes one or more points from the mesh.

Synopsis: Mesh.del_point( ivec PIDLST)
PIDLST should contain the  point #id, such as the one returned by the 'add
point' command."""
        return self.set("del point", PIDLST)


    #@SET MESH:SET('add convex')
    def add_convex(self, CVTR, CVPTS):
        """Add a new convex into the mesh.

Synopsis: IDX = Mesh.add_convex( @geotrans CVTR, mat CVPTS)
The convex structure (triangle, prism,...) is given by CVTR (obtained with
GeoTrans('...')), and its points are given by the columns of CVPTS. On return,
IDX contains the convex number. CVPTS might be a 3-dimensional array in order
to insert more than one convex (or a two dimensional array correctly shaped
according to Fortran ordering)."""
        return self.set("add convex", CVTR, CVPTS)


    #@SET MESH:SET('del convex')
    def del_convex(self, IDX):
        """Remove one or more convexes from the mesh.

Synopsis: Mesh.del_convex( IDX)
IDX should contain the convexes #ids, such as the ones returned bu
Mesh.add_convex()."""
        return self.set("del convex", IDX)


    #@SET MESH:SET('del convex of dim')
    def del_convex_of_dim(self, DIM):
        """Remove all convexes of dimension listed in DIM.

Synopsis: Mesh.del_convex_of_dim( ivec DIM)
For example Mesh.del_convex_of_dim( [1,2]) remove all line segments, triangles
and quadrangles."""
        return self.set("del convex of dim", DIM)


    #@SET MESH:SET('translate')
    def translate(self, V):
        """Translates each point of the mesh from V."""
        return self.set("translate", V)


    #@SET MESH:SET('transform')
    def transform(self, T):
        """Applies the matrix T to each point of the mesh.

Synopsis: Mesh.transform( mat T)
Note that T is not required to be a NxN matrix (with N=Mesh.dim()). Hence it
is possible to transform a 2D mesh into a 3D one (and reciprocally)."""
        return self.set("transform", T)


    #@SET MESH:SET('merge')
    def merge(self, M2):
        """Merge with the mesh M2.

Synopsis: Mesh.merge( @mesh M2)
Overlapping points won't be duplicated. If M2 is a mesh_fem object, its linked
mesh will be used."""
        return self.set("merge", M2)


    #@SET MESH:SET('optimize structure')
    def optimize_structure(self):
        """Reset point and convex numbering.

Synopsis: Mesh.optimize_structure()
After optimisation, the points (resp. convexes) will be consecutively numbered
from 0 to Mesh.max_pid()-1 (resp. Mesh.max_cvid()-1)."""
        return self.set("optimize structure")


    #@SET MESH:SET('refine')
    def refine(self, CVLST=None):
        """Use a Bank strategy for mesh refinement.

Synopsis: Mesh.refine([, ivec CVLST])
If CVLST is not given, the whole mesh is refined. Note that the regions, and
the finite element methods and integration methods of the mesh_fem and mesh_im
objects linked to this mesh will be automagically refined."""
        return self.set("refine", CVLST)


    #@SET MESH:SET('region')
    def set_region(self, rnum, CVFLST):
        """Assigns the region number rnum to the convex faces stored in each column of
the matrix CVFLST.   The first row of CVFLST contains a convex number, and the
second row contains a face number in the convex (or -1 for the whole convex --
regions are usually used to store a list of convex faces, but you may also use
them to store a list of convexes)."""
        return self.set("region", rnum, CVFLST)


    #@SET MESH:SET('region_intersect')
    def region_intersect(self, R1, R2):
        """Replace the region number R1 with its intersection with region number R2."""
        return self.set("region_intersect", R1, R2)


    #@SET MESH:SET('region_merge')
    def region_merge(self, R1, R2):
        """Merge region number R2 into region number R1."""
        return self.set("region_merge", R1, R2)


    #@SET MESH:SET('region_substract')
    def region_substract(self, R1, R2):
        """Replace the region number R1 with its difference with region number R2."""
        return self.set("region_substract", R1, R2)


    #@SET MESH:SET('delete region')
    def delete_region(self, RLST):
        """Remove the regions whose #ids are listed in RLST"""
        return self.set("delete region", RLST)


class MeshFem:
    def __init__(self, *args):
        """General constructor for MeshFem objects.

* MeshFem(mesh M [, int Qdim=1])
Build a new MeshFem object. The Qdim parameter is optional.
 * MeshFem('load', fname[, mesh M])
  Load a meshfem from a file.   If the mesh M is not supplied (this kind of
file does not store the mesh), then it is read from the file and its
descriptor is returned as the second output argument.
 * MeshFem('from_string', str[, mesh M])
  Create a mesh_fem object from its string description.  See also
MeshFem.char()
 * MeshFem('clone', meshfem MF2)
  Create a copy of a mesh_fem.
        """
        generic_constructor(self,'mesh_fem',*args)
    def __del__(self):
        generic_destructor(self,destructible=True)
    def get(self, *args):
        return getfem('mesh_fem_get',self.id, *args)
    def set(self, *args):
        return getfem('mesh_fem_set',self.id, *args)
    def __str__(self):
        return self.char()
    def __repr__(self):
        return '<getfem.MeshFem Q=%d, %d dof, %d (+%d) bytes>' % \
               (self.qdim(),self.nbdof(),self.memsize(), \
                self.linked_mesh().memsize())

    #@RDATTR MESHFEM:GET('nbdof')
    def nbdof(self):
        """Return the number of Degrees of Freedom (DoF) of the meshfem MF."""
        return self.get("nbdof")


    #@GET    MESHFEM:GET('dof from cv')
    def dof_from_cv(self, CVLST):
        """Return the DoF of the convexes listed in CVLST.

Synopsis: I = MeshFem.dof_from_cv( CVLST)
WARNING: the Degree of Freedom might be returned in ANY order, do not use this
function in your assembly routines. Use 'dof from cvid' instead, if you want
to be able to map a convex number with its associated degrees of freedom.  One
can also get the list of dof on a set on convex faces, by indicating on the
second row of CVLST the faces numbers (with respect to the convex number on
the first row)."""
        return self.get("dof from cv", CVLST)


    #@GET    MESHFEM:GET('dof from cvid')
    def dof_from_cvid(self, CVLST=None):
        """Return the degrees of freedom attached to each convex of the mesh.\\\\

Synopsis:  [DOFS,IDX]=MeshFem.dof_from_cvid([,CVLST])
If CVLST is omitted, all the convexes will be considered (equivalent to CVLST
= 1 ... @Mesh.max_cvid()).  IDX is a vector, length(IDX) = length(CVLIST)+1.
DOFS is a vector containing the concatenated list of dof of each convex in
CVLST. Each entry of IDX is the position of the corresponding convex point
list in DOFS. Hence, for example, the list of points of the second convex is
DOFS[IDX(2):IDX(3)].\\  If CVLST contains convex #id which do not exist in the
mesh, their point list will be empty."""
        return self.get("dof from cvid", CVLST)


    #@GET    MESHFEM:GET('non conformal dof')
    def non_conformal_dof(self, CVLST=None):
        """Return partially linked degrees of freedom.

Synopsis: MeshFem.non_conformal_dof([,CVLST])
Return the dof located on the border of a convex and which belong to only one
convex, except the ones which are located on the border of the mesh.  For
example, if the convex 'a' and 'b' share a common face, 'a' has a P1 FEM, and
'b' has a P2 FEM, then the dof on the middle of the face will be returned by
this function (this can be useful when searching the interfaces between
classical FEM and hierarchical FEM)."""
        return self.get("non conformal dof", CVLST)


    #@RDATTR MESHFEM:GET('qdim')
    def qdim(self):
        """Return the dimension Q of the field interpolated by the mesh_fem.

Synopsis: MeshFem.qdim()
By default, Q=1 (scalar field). This has an impact on the DOF numbering."""
        return self.get("qdim")


    #@GET    MESHFEM:GET('fem')
    def fem(self, CVLST=None):
        """Return a list of FEM used by the meshfem.

Synopsis: [FEMLST, CV2F] = MeshFem.fem([, CVLST])
FEMLST is an array of all @tfem objects found in the convexes given in CVLST.
If CV2F was supplied as an output argument, it contains, for each convex
listed in CVLST, the index of its correspounding FEM in FEMLST.   Convexes
which are not part of the mesh, or convexes which do not have any FEM have
their correspounding entry in CV2F set to -1."""
        return self.get("fem", CVLST)


    #@GET    MESHFEM:GET('is_lagrangian')
    def is_lagrangian(self, CVLST=None):
        """Test if the meshfem is Lagrangian.

Synopsis: I = MeshFem.is_lagrangian([,CVLST])
Lagrangian means that each base function Phi[i] is such that
    Phi[i](P[j]) = delta(i,j),
 where P[j] is the DoF location of the jth base function, and delta(i,j) = 1
if i==j, else 0.
  If CVLST is omitted, it returns 1 if all convexes in the mesh are
Lagrangian. If CVLST is used, it returns the convex indices (with respect to
CVLST) which are Lagrangian."""
        return self.get("is_lagrangian", CVLST)


    #@GET    MESHFEM:GET('is_equivalent')
    def is_equivalent(self, CVLST=None):
        """Test if the meshfem is equivalent.

Synopsis: I = MeshFem.is_equivalent([,CVLST])
See MeshFem.is_lagrangian()"""
        return self.get("is_equivalent", CVLST)


    #@GET    MESHFEM:GET('is_polynomial')
    def is_polynomial(self, CVLST=None):
        """Test if all base functions are polynomials.

Synopsis: I = MeshFem.is_polynomial([,CVLST])
See MeshFem.is_lagrangian()"""
        return self.get("is_polynomial", CVLST)


    #@GET    MESHFEM:GET('dof on region')
    def dof_on_region(self, RLIST):
        """Return the list of dof lying on one of the mesh regions listed in RLIST.

Synopsis: DOFLST = MeshFem.dof_on_region( RLIST)
More precisely, this function returns the DoF whose support is non-null on one
of regions whose #ids are listed in RLIST (note that for boundary regions,
some dof nodes may not lie exactly on the boundary, for example the dof of
PK(n,0) lies on the center of the convex, but the base function in not null on
the convex border)."""
        return self.get("dof on region", RLIST)


    #@GET    MESHFEM:GET('dof nodes')
    def dof_nodes(self, DOFLST=None):
        """Get location of Degrees of Freedom.

Synopsis: [DOF_XY] = MeshFem.dof_nodes([, DOFLST])
Return the list of interpolation points for the specified dof #IDs in DOFLST
(if DOFLST is omitted, all DoF are considered)."""
        return self.get("dof nodes", DOFLST)


    #@GET    MESHFEM:GET('dof partition')
    def dof_partition(self):
        """Get the 'dof_partition' array.

Synopsis: [DOFP] = MeshFem.dof_partition()
Return the array which associates an integer (the partition number) to each
convex of the mesh_fem. By default, it is an all-zero array. The degrees of
freedom of each convex of the mesh_fem are connected only to the dof of
neighbouring convexes which have the same partition number, hence it is
possible to create partially discontinuous mesh_fem very easily."""
        return self.get("dof partition")


    #@GET    MESHFEM:GET('interpolate_convex_data')
    def interpolate_convex_data(self, Ucv):
        """Interpolate data given on each convex of the mesh to the meshfem dof.  The
meshfem has to be lagrangian, and should be discontinuous (typically a
FEM_PK(N,0) or FEM_QK(N,0) should be used).  The last dimension of the input
vector Ucv should have Mesh.max_cvid() elements.  Example of use:
MeshFem.interpolate_convex_data( Mesh.quality())"""
        return self.get("interpolate_convex_data", Ucv)


    #@GET    MESHFEM:GET('save')
    def save(self, filename, opt=None):
        """Save a meshfem in a text file (and optionaly its linked mesh object if opt is
the string 'with_mesh')."""
        return self.get("save", filename, opt)


    #@GET    MESHFEM:GET('char')
    def char(self, opt=None):
        """Output a string description of the mesh_fem.

Synopsis: MeshFem.char([, opt])
By default, it does not include the description of the linked mesh object,
except if opt is 'with_mesh'"""
        return self.get("char", opt)


    #@GET    MESHFEM:GET('linked mesh')
    def linked_mesh(self):
        """Return a reference to the mesh object linked to MF."""
        return self.get("linked mesh")


    #@GET    MESHFEM:GET('export to vtk')
    def export_to_vtk(self, FILENAME, *args):
        """Export a mesh_fem and some fields to a vtk file.

Synopsis: MeshFem.export_to_vtk( string FILENAME, ... ['ascii'], U, 'name'...)
The FEM and geometric transformations will be mapped to order 1 or 2
isoparametric PK (or QK) FEMs (as VTK does not handle higher order elements).
If you need to represent high-order FEMs or high-order geometric
transformations, you should consider @Slice.export_to_vtk()."""
        return self.get("export to vtk", FILENAME, *args)


    #@GET    MESHFEM:GET('export to dx')
    def export_to_dx(self, FILENAME, *args):
        """Export a mesh_fem and some fields to an OpenDX file.

Synopsis: MeshFem.export_to_dx( string FILENAME, ... ['as', string
mesh_name][,'edges']['serie',string serie_name][,'ascii'][,'append'], U,
'name'...)
This function will fail if the mesh_fem mixes different convex types (i.e.
quads and triangles), or if OpenDX does not handle a specific element type
(i.e. prism connections are not known by OpenDX).  The FEM will be mapped to
order 1 PK (or QK) FEMs. If you need to represent high-order FEMs or high-
order geometric transformations, you should consider @Slice.export_to_dx()."""
        return self.get("export to dx", FILENAME, *args)


    #@GET    MESHFEM:GET('memsize')
    def memsize(self):
        """Return the amount of memory (in bytes) used by the mesh_fem object.

Synopsis: MeshFem.memsize()
The result does not take into account the linked mesh object."""
        return self.get("memsize")


    #@SET MESHFEM:SET('fem')
    def set_fem(self, FEM, CVIDX=None):
        """Set the Finite Element Method.

Synopsis: MeshFem.set_fem( @tfem FEM [, ivec CVIDX])
Assign a FEM to all convexes whose #ids are listed in CVIDX. If CVIDX is not
given, the integration is assigned to all convexes.  See the help of Fem to
obtain a list of available FEM methods."""
        return self.set("fem", FEM, CVIDX)


    #@SET MESHFEM:SET('classical fem')
    def set_classical_fem(self, K, CVIDX=None):
        """Assign a classical (Lagrange polynomial) fem of order K to the meshfem.

Synopsis: MeshFem.set_classical_fem( int K, [,ivec CVIDX])
Uses FEM_PK for simplexes, FEM_QK for parallelepipeds etc."""
        return self.set("classical fem", K, CVIDX)


    #@SET MESHFEM:SET('classical discontinuous fem')
    def set_classical_discontinuous_fem(self, *args):
        """Assigns a classical (Lagrange polynomial) discontinuous fem or order K.

Synopsis: MeshFem.set_classical_discontinuous_fem( int K, [int IM_DEGREE
[,ivec CVIDX]])
Similar to MeshFem.set_classical_fem() except that FEM_PK_DISCONTINUOUS is
used."""
        return self.set("classical discontinuous fem", *args)


    #@SET MESHFEM:SET('qdim')
    def set_qdim(self, Q):
        """Change the Q dimension of the field that is interpolated by the meshfem.

Synopsis: MeshFem.set_qdim( int Q)
Q=1 means that the meshfem describes a scalar field, Q=N means that the
meshfem describes a vector field of dimension N."""
        return self.set("qdim", Q)


    #@SET MESHFEM:SET('dof partition')
    def dof_partition(self, DOFP):
        """Change the dof_partition array.

Synopsis: MeshFem.dof_partition( ivec DOFP)
DOFP is a vector holding a integer value for each convex of the mesh_fem. See
MeshFem.dof_partition() for a description of "dof partition"."""
        return self.set("dof partition", DOFP)

    def eval(self, expression):
        """interpolate an expression on the (lagrangian) MeshFem.

        Examples:
          mf.eval('x[0]*x[1]') interpolates the function 'x*y'
          mf.eval('[x[0],x[1]]') interpolates the vector field '[x,y]'
        """
        P=self.dof_nodes()
        nbd = P.shape[1];

        if not self.is_lagrangian:
            raise RuntimeError('cannot eval on a non-Lagragian MeshFem')
        if self.qdim() != 1:
            raise RuntimeError('only works (for now) with qdim == 1')
        x=P[:,0]; r=numarray.array(eval(expression))
        Z=numarray.zeros(r.shape + (nbd,),'d')
        for i in range(0,nbd):
            x=P[:,i]
            Z[...,i]=eval(expression)
        return Z


class MeshIm:
    def __init__(self, *args):
        """General constructor for MeshIm objects.

* MeshIm(mesh M, [{Integ Im|int IM_DEGREE}])
Build a new MeshIm object. For convenience, optional arguments (IM or IM_DEGREE) can be provided, in that case a call to MESHIM:SET('integ') is issued with these arguments.

 * MeshIm('load', fname[, mesh M])
  Load a meshim from a file.   If the mesh M is not supplied (this kind of
file does not store the mesh), then it is read from the file and its
descriptor is returned as the second output argument.
 * MeshIm('from_string', str[, mesh M])
  Create a meshim object from its string description.  See also MeshIm.char()
 * MeshIm('clone', meshim MIM2)
  Create a copy of a meshim.
        """
        generic_constructor(self,'mesh_im',*args)
    def __del__(self):
        generic_destructor(self,destructible=True)
    def get(self, *args):
        return getfem('mesh_im_get',self.id, *args)
    def set(self, *args):
        return getfem('mesh_im_set',self.id, *args)
    def __str__(self):
        return self.char()
    def __repr__(self):
        return '<getfem.MeshIm %d (+%d) bytes>' % \
               (self.memsize(), \
                self.linked_mesh().memsize())

    #@GET    MESHIM:GET('integ')
    def integ(self, CVLIST=None):
        """Return a list of integration methods used by the meshim.

Synopsis: [INTEG, CV2I] = MeshIm.integ([, CVLIST])
INTEG is an array of all @tinteg objects found in the convexes given in CVLST.
If CV2F was supplied as an output argument, it contains, for each convex
listed in CVLST, the index of its correspounding integration method in INTEG.
Convexes which are not part of the mesh, or convexes which do not have any
integration method have their correspounding entry in CV2I set to -1."""
        return self.get("integ", CVLIST)


    #@GET    MESHIM:GET('eltm')
    def eltm(self, MET, CV, F=None):
        """Return the elementary matrix (or tensor) integrated on the convex CV.

Synopsis: M = MeshIm.eltm( @eltm MET, int CV [int F])
!!WARNING!! Be sure that the fem used for the construction of MET is
compatible with the fem assigned to element CV ! This is not checked by the
function ! If the argument F is given, then the elementary tensor is
integrated on the face F of CV instead of the whole convex."""
        return self.get("eltm", MET, CV, F)


    #@GET    MESHIM:GET('save')
    def save(self, filename):
        """Saves a meshim in a text file (and optionaly its linked mesh object)."""
        return self.get("save", filename)


    #@GET    MESHIM:GET('char')
    def char(self):
        """Output a string description of the mesh_im.

Synopsis: MeshIm.char([,'with mesh'])
By default, it does not include the description of the linked mesh object."""
        return self.get("char")


    #@GET    MESHIM:GET('linked mesh')
    def linked_mesh(self):
        """Returns a reference to the mesh object linked to MIM."""
        return self.get("linked mesh")


    #@GET    MESHIM:GET('memsize')
    def memsize(self):
        """Return the amount of memory (in bytes) used by the mesh_im object.

Synopsis: MeshIm.memsize()
The result does not take into account the linked mesh object."""
        return self.get("memsize")


    #@SET MESHIM:SET('integ')
    def set_integ(self, *args):
        """Set the integration method.

Synopsis: MeshIm.set_integ( {integ IM|int IM_DEGREE} [, ivec CVIDX])
Assign an integration method to all convexes whose #ids are listed in CVIDX.
If CVIDX is not given, the integration is assigned to all convexes. It is
possible to assign a specific integration method with an integration method
handle IM obtained via Integ('IM_SOMETHING'), or to let getfem choose a
suitable integration method with IM_DEGREE (choosen such that polynomials of
degree <= IM_DEGREE are exactly integrated. If IM_DEGREE=-1, then the dummy
integration method IM_NONE will be used.)"""
        return self.set("integ", *args)



class MdBrick:
    def __init__(self, *args):
        """General constructor for MdBrick objects.

 * B=MdBrick('constraint', mdbrick parent, string CTYPE [, int numfem])
  Build a generic constraint brick.   It may be useful in some situations,
such as the Stokes problem where the pressure in defined modulo a constant. In
such a situation, this brick can be used to add an additional constraint on
the pressure value.       CTYPE has to be chosen among 'augmented',
'penalized', and 'eliminated'.  The constraint can be specified with
MdBrick.set_constraints(). Note that Dirichlet bricks (except the 'generalized
Dirichlet' one) are also specializations of the 'constraint' brick.
 * B=MdBrick('dirichlet', mdbrick parent, int BNUM, meshfem MFMULT, string
CTYPE [, int numfem])
  Build a Dirichlet condition brick which impose the value of a field along a
mesh boundary.  The BNUM parameter selects on which mesh region the Dirichlet
condition is imposed. CTYPE has to be chosen among 'augmented', 'penalized',
and 'eliminated'. The MFMULT may generally be taken as the meshfem of the
unknown, but for 'augmented' Dirichlet conditions, you may have to respect the
Inf-Sup condition and choose an adequate meshfem.
 * B=MdBrick('dirichlet_on_normal_component', mdbrick parent, int BNUM,
meshfem MFMULT, string CTYPE [, int numfem])
  Build a Dirichlet condition brick which imposes the value of the normal
component of a vector field.
 * B=MdBrick('dirichlet_on_normal_derivative', mdbrick parent, int BNUM,
meshfem MFMULT, string CTYPE [, int numfem])
  Build a Dirichlet condition brick which imposes the value of the normal
derivative of the unknown.
 * B=MdBrick('generalized_dirichlet', mdbrick parent, int BNUM [, int numfem])
  This is the "old" Dirichlet brick of getfem.  This brick can be used to
impose general Dirichlet conditions 'h(x)u(x) = r(x)' , however it may have
some issues with elaborated FEM (such as Argyris, etc). It should be avoided
when possible.
 * B=MdBrick('source_term', mdbrick parent, [, int BNUM=-1[, int numfem]])
  Add a boundary or volumic source term ( \int B.v ).  If BNUM is omitted (or
set to -1) , the brick adds a volumic source term on the whole mesh. For BNUM
>= 0, the source term is imposed on the mesh region BNUM. Use
MdBrick.set_param('source term',mf,B) to set the source term field. The source
term is expected as a vector field of size Q (with Q = qdim).
 * B=MdBrick('normal_source_term', mdbrick parent, int BNUM [, int numfem])
  Add a boundary source term ( \int (Bn).v ).  The source term is imposed on
the mesh region BNUM (which of course is not allowed to be a volumic region,
only boundary regions are allowed). Use MdBrick.set_param('source term',mf,B)
to set the source term field. The source term B is expected as tensor field of
size QxN (with Q = qdim, N = mesh dim). For example, if you consider an
elasticity problem, this brick may be used to impose a force on the boundary
with B as the stress tensor.
 * B=MdBrick('normal_derivative_source_term', mdbrick parent, int BNUM [, int
numfem])
  Add a boundary source term ( \int (\partial_n B).v ).  The source term is
imposed on the mesh region BNUM. Use MdBrick.set_param('source term',mf,B) to
set the source term field, which is expected as a vector field  of size Q
(with Q = qdim).
 * B=MdBrick('neumann KirchhoffLove source term', mdbrick parent, int BNUM [,
int numfem])
   Add a boundary source term for neumann Kirchhoff-Love plate problems
(should be used with the Kirchhoff-Love flavour of the bilaplacian brick).
 * B=MdBrick('qu_term', mdbrick parent, [, int BNUM [, int numfem]])
  Update the tangent matrix with a \int (Qu).v term.  The Q(x) parameter is a
matrix field of size qdim x qdim. An example of use is for the "iku" part of
Robin boundary conditions \partial_n u + iku = ...
 * B=MdBrick('mass_matrix', meshim mim, meshfem mf_u [,'real'|'complex'])
  Build a mass-matrix brick.
 * B=MdBrick('generic_elliptic', meshim MIM, meshfem mfu
[,'scalar'|'matrix'|'tensor'][,'real'|'complex'])
  Setup a generic elliptic problem ( (A*grad(U)).grad(V) ).  The brick
parameter 'A' may be a scalar field, a matrix field, or a tensor field
(default is scalar).
 * B=MdBrick('helmholtz', meshim MIM, meshfem mfu [,'real'|'complex'])
  Setup a Helmholtz problem.  The brick has one parameter, 'wave_number'.
 * B=MdBrick('isotropic_linearized_elasticity', meshim MIM, meshfem mfu)
  Setup a linear elasticity problem.  The brick has two scalar parameter,
'lambda' and 'mu' (the Lame coefficients).
 * B=MdBrick('linear_incompressibility_term', mdbrick parent, meshfem mf_p [,
int numfem])
  Add an incompressibily constraint (div u = 0).
 * B=MdBrick('nonlinear_elasticity', meshim MIM, meshfem mfu, string lawname)
  Setup a nonlinear elasticity (large deformations) problem.  The material law
can be chosen among - 'SaintVenant Kirchhoff' (linearized material law) -
'Mooney Rivlin' (to be used with the nonlinear incompressibily term) -
'Ciarlet Geymonat'
 * B=MdBrick('nonlinear_elasticity_incompressibility_term', mdbrick parent,
meshfem mf_p [, int numfem])
  Add an incompressibily constraint to a large strain elasticity problem.
 * B=MdBrick('small_deformations_plasticity', meshim MIM, meshfem mfu, scalar
THRESHOLD)
  Setup a plasticity problem (with small deformations).  The THRESHOLD
parameter is the maximum value of the Von Mises stress before 'plastification'
of the material.
 * B=MdBrick('dynamic', mdbrick parent, scalar rho [, int numfem])
  Dynamic brick. This brick is not ready.
 * B=MdBrick('navier_stokes', meshim MIM, meshfem mfu, meshfem mfp)
  Setup a Navier-Stokes problem (this brick is not ready, do not use it).
 * B=MdBrick('bilaplacian', meshim MIM, meshfem mfu, ['Kirchhoff-Love'])
  Setup a bilaplacian problem.  If the Kirchhoff-Love option is specified, the
Kirchhoff-Love plate model is used.
 * B=MdBrick('isotropic_linearized_plate', meshim MIM, meshim MIMSUB, meshfem
MF_UT, meshfem MF_U3, meshfem MF_THETA, scalar EPSILON)
     Setup a linear plate model brick (for moderately thick plates, using the
Reissner-Mindlin model). EPSILON is the plate thinkness, the @mf MF_UT and
MF_U3 are used respectively for the membrane displacement and the transverse
displacement of the plate. The @mf MF_THETA is the rotation of the normal
("section rotations").  The second integration method MIMSUB can be chosen
equal to MIM, or different if you want to perform sub-integration on the
transverse shear term (mitc4 projection).  This brick has two parameters
"lambda" and "mu" (the Lam coefficients)
 * B=MdBrick('mixed_isotropic_linearized_plate', meshim MIM, meshfem MF_UT,
meshfem MF_U3, meshfem MF_THETA, scalar EPSILON)
   Setup a mixed linear plate model brick (for thin plates, using Kirchhoff-
Love model).  For a non-mixed version, use the bilaplacian brick.
 * B=MdBrick('plate_source_term', mdbrick parent, [, int BNUM=-1[, int
numfem]])
   Add a boundary or a volumic source term to a plate problem. This brick has
two parameters: "B" is the displacement (ut and u3) source term, "M" is the
moment source term (i.e. the source term on the rotation of the normal).
 * B=MdBrick('plate_simple_support', mdbrick parent, int BNUM, string CTYPE [,
int numfem])
   Add a "simple support" boundary condition to a plate problem (homogeneous
Dirichlet condition on the displacement, free rotation). CTYPE specifies how
the constraint is enforced ('penalized', 'augmented' or 'eliminated').
 * B=MdBrick('plate_clamped_support', mdbrick parent, int BNUM, string CTYPE
[, int numfem])
   Add a "clamped support" boundary condition to a plate problem (homogeneous
Dirichlet condition on the displacement and on the rotation). CTYPE specifies
how the constraint is enforced ('penalized', 'augmented' or 'eliminated').
 * B=MdBrick('plate_closing', mdbrick parent [, int numfem])
   Add a free edges condition for the mixed plate model brick.  This brick is
required when the mixed linearized plate brick is used. It must be inserted
after all other boundary conditions (the reason is that the brick has to
inspect all other boundary conditions to determine the number of disconnected
boundary parts which are free edges).
        """
        generic_constructor(self,'mdbrick',*args)
    def __del__(self):
        generic_destructor(self,destructible=True)
    def get(self, *args):
        return getfem('mdbrick_get',self.id, *args)
    def set(self, *args):
        return getfem('mdbrick_set',self.id, *args)
    def __str__(self):
        return self.char()
    def __repr__(self):
        return '<getfem.MdBrick %d bytes>' % \
               (self.memsize(),)

    #@RDATTR MDBRICK:GET('nbdof')
    def nbdof(self):
        """Get the total number of dof of the current problem.

Synopsis: MdBrick.nbdof()
This is the sum of the brick specific dof plus the dof of the parent bricks."""
        return self.get("nbdof")


    #@RDATTR MDBRICK:GET('dim')
    def dim(self):
        """Get the dimension of the main mesh (2 for a 2D mesh, etc)."""
        return self.get("dim")


    #@RDATTR MDBRICK:GET('is_linear')
    def is_linear(self):
        """Return true if the problem is linear."""
        return self.get("is_linear")


    #@RDATTR MDBRICK:GET('is_symmetric')
    def is_symmetric(self):
        """Return true if the problem is symmetric."""
        return self.get("is_symmetric")


    #@RDATTR MDBRICK:GET('is_coercive')
    def is_coercive(self):
        """Return true if the problem is coercive."""
        return self.get("is_coercive")


    #@RDATTR MDBRICK:GET('is_complex')
    def is_complex(self):
        """Return true if the problem uses complex numbers."""
        return self.get("is_complex")


    #@GET MDBRICK:GET('mixed_variables') 
    def mixed_variables(self):
        """Identify the indices of mixed variables (typically the pressure, etc.) in the
tangent matrix."""
        return self.get("mixed_variables")


    #@RDATTR MDBRICK:GET('subclass')
    def subclass(self):
        """Get the typename of the brick."""
        return self.get("subclass")


    #@GET MDBRICK:GET('param_list')
    def param_list(self):
        """Get the list of parameters names.

Synopsis: MdBrick.param_list()
Each brick embeds a number of parameters (the Lam coefficients for the
linearized elasticity brick, the wave number for the Helmholtz brick,...),
described as a (scalar, or vector, tensor etc) field on a mesh_fem. You can
read/change the parameter values with MdBrick.param() and MdBrick.set_param()."""
        return self.get("param_list")


    #@GET MDBRICK:GET('param')
    def param(self, parameter_name):
        """Get the parameter value.

Synopsis: MdBrick.param( string parameter_name)
When the parameter has been assigned a specific mesh_fem, it is returned  as a
large array (the last dimension being the mesh_fem dof). When no mesh_fem has
been assigned, the parameter is considered to be constant over the mesh."""
        return self.get("param", parameter_name)


    #@GET MDBRICK:GET('solve')
    def solve(self, mds, *args):
        """Run the standard getfem solver.

Synopsis: MdBrick.solve( @mdstate mds [,...])
Note that you should be able to use your own solver if you want (it is
possible to obtain the tangent matrix and its right hand side with the
MdState.tangent_matrix() etc.).   Various options can be specified:

  - 'noisy' or 'very noisy' : the solver will display some
 information showing the progress (residual values etc.).
 - 'max_iter', NIT         : set the maximum iterations numbers.
 - 'max_res', RES          : set the target residual value.
 - 'lsolver', SOLVERNAME   : select explicitely the solver used for< the
linear systems (the default value is 'auto', which lets getfem choose itself).
Possible values are 'superlu', 'mumps' (if supported), 'cg/ildlt', 'gmres/ilu'
and 'gmres/ilut'."""
        return self.get("solve", mds, *args)


    #@GET MDBRICK:GET('von mises')
    def von_mises(self, mds, MFVM):
        """Compute the Von Mises stress on the mesh_fem MFVM.

Synopsis: VM=MdBrick.von_mises( @mdstate mds, meshfem MFVM)
Only available on bricks where it has a meaning: linearized elasticity,
plasticity, nonlinear elasticity.. Note that in 2D it is not the "real" Von
Mises (which should take into account the 'plane stress' or 'plane strain'
aspect), but a pure 2D Von Mises."""
        return self.get("von mises", mds, MFVM)


    #@GET MDBRICK:GET('memsize')
    def memsize(self):
        """Return the amount of memory (in bytes) used by the model brick."""
        return self.get("memsize")


    #@GET MDBRICK:GET('tresca')
    def tresca(self, mds, MFVM):
        """Compute the Tresca stress criterion on the mesh_fem MFVM.

Synopsis: VM=MdBrick.tresca( @mdstate mds, meshfem MFVM)
Only available on bricks where it has a meaning: linearized elasticity,
plasticity, nonlinear elasticity.."""
        return self.get("tresca", mds, MFVM)


    #@SET MDBRICK:SET('param')
    def set_param(self, name, *args):
        """Change the value of a brick parameter.

Synopsis: MdBrick.set_param( string name, {meshfem MF,V | V})
V should contain the new parameter value. If a meshfem is given , V should
hold the field values over that meshfem (i.e. its last dimension should be
MeshFem.nbdof())."""
        return self.set("param", name, *args)


    #@SET MDBRICK:SET('constraints');
    def set_constraints(self, H, R):
        """Set the constraints imposed by a constraint brick.

Synopsis: MdBrick.set_constraints( mat H, vec R)
This is only applicable to the bricks which inherit from the constraint brick,
such as the Dirichlet ones. Imposes HU=R."""
        return self.set("constraints", H, R)


    #@SET MDBRICK:SET('constraints_rhs');
    def set_constraints_rhs(self, H, R):
        """Set the right hand side of the constraints imposed by a constraint brick.

Synopsis: MdBrick.set_constraints_rhs( mat H, vec R)
This is only applicable to the bricks which inherit from the constraint brick,
such as the Dirichlet ones."""
        return self.set("constraints_rhs", H, R)


    #@SET MDBRICK:SET('penalization_epsilon');
    def penalization_epsilon(self, eps):
        """Change the penalization coefficient of a constraint brick.

Synopsis: MdBrick.penalization_epsilon( eps)
This is only applicable to the bricks which inherit from the constraint brick,
such as the Dirichlet ones. And of course it is not effective when the
constraint is enforced via direct elimination or via Lagrange multipliers. The
default value of eps is 1e-9."""
        return self.set("penalization_epsilon", eps)


class MdState:
    def __init__(self, *args):
        """General constructor for MdState objects.

        These objects hold the global model data of a chain of
        MdBricks, such as the right hand side, the tangent matrix and
        the constraints.

        * MDS=MdState(mdbrick B)
        Build a modelstate for the brick B (selects the real or
        complex state from the complexity of B).

 * MDS=MdState('real')
  Build a model state for real unknowns.

 * MDS=MdState('complex')
  Build a model state for complex unknowns.
        """
        generic_constructor(self,'mdstate',*args)
    def __del__(self):
        generic_destructor(self,destructible=True)
    def get(self, *args):
        return getfem('mdstate_get',self.id, *args)
    def set(self, *args):
        return getfem('mdstate_set',self.id, *args)
    def __str__(self):
        return self.char()
    def __repr__(self):
        return '<getfem.MdState %d bytes>' % \
               (self.memsize(),)

    #@RDATTR MDSTATE:GET('is_complex')
    def is_complex(self):
        """Return 0 is the model state is real, 1 if it is complex."""
        return self.get("is_complex")


    #@GET MDSTATE:GET('tangent_matrix')
    def tangent_matrix(self):
        """Return the tangent matrix stored in the model state."""
        return self.get("tangent_matrix")


    #@GET MDSTATE:GET('constraints_matrix')
    def constraints_matrix(self):
        """Return the constraints matrix stored in the model state."""
        return self.get("constraints_matrix")


    #@GET MDSTATE:GET('reduced_tangent_matrix')
    def reduced_tangent_matrix(self):
        """Return the reduced tangent matrix (i.e. the tangent matrix after elimination
of the constraints)."""
        return self.get("reduced_tangent_matrix")


    #@GET MDSTATE:GET('constraints_nullspace')
    def constraints_nullspace(self):
        """Return the nullspace of the constraints matrix."""
        return self.get("constraints_nullspace")


    #@GET MDSTATE:GET('state')
    def state(self):
        """Return the vector of unknowns, which contains the solution after
MdBrick.solve()."""
        return self.get("state")


    #@GET MDSTATE:GET('residual')
    def residual(self):
        """Return the residual."""
        return self.get("residual")


    #@GET MDSTATE:GET('reduced_residual')
    def reduced_residual(self):
        """Return the residual on the reduced system."""
        return self.get("reduced_residual")


    #@GET MDSTATE:GET('unreduce')
    def unreduce(self, U):
        """Reinsert the constraint eliminated from the system."""
        return self.get("unreduce", U)


    #@GET MDSTATE:GET('memsize')
    def memsize(self):
        """Return the amount of memory (in bytes) used by the model state."""
        return self.get("memsize")


    #@SET MDSTATE:SET('compute_reduced_system')
    def compute_reduced_system(self):
        """Compute the reduced system from the tangent matrix and constraints."""
        return self.set("compute_reduced_system")


    #@SET MDSTATE:SET('compute_reduced_residual')
    def compute_reduced_residual(self):
        """Compute the reduced residual from the residual and constraints."""
        return self.set("compute_reduced_residual")


    #@SET MDSTATE:SET('compute_residual')
    def compute_residual(self, B):
        """Compute the residual for the brick B."""
        return self.set("compute_residual", B)


    #@SET MDSTATE:SET('compute_tangent_matrix')
    def compute_tangent_matrix(self, B):
        """Update the tangent matrix from the brick B."""
        return self.set("compute_tangent_matrix", B)


    #@SET MDSTATE:SET('clear')
    def clear(self):
        """Clear the model state."""
        return self.set("clear")

    
class GeoTrans:
    def __init__(self, *args):
        """
General function for building descriptors to geometric transformations.  Name
can be:  * 'GT_PK(N,K)'   : Transformation on simplexes, dim N, degree K *
'GT_QK(N,K)'   : Transformation on parallelepipeds, dim N, degree K *
'GT_PRISM(N,K)'          : Transformation on prisms, dim N, degree K *
'GT_PRODUCT(a,b)'        : tensorial product of two transformations *
'GT_LINEAR_PRODUCT(a,b)' : Linear tensorial product of two transformations
        """
        generic_constructor(self,'geotrans',*args)
    def __del__(self):
        generic_destructor(self,destructible=False)
    def get(self, *args):
        return getfem('geotrans_get',self.id, *args)
    def __str__(self):
        return self.get('char')
    def __repr__(self):
        return '<getfem.Geotrans '+str(self)+'>'

    #@RDATTR GEOTRANS:GET('dim')
    def dim(self):
        """Get the dimension of the geometric transformation.

Synopsis: CvStruct.dim()
This is the dimension of the source space, i.e. the dimension of the reference
convex."""
        return self.get("dim")


    #@RDATTR GEOTRANS:GET('is_linear')
    def is_linear(self):
        """Return 0 if the geometric transformation is not linear."""
        return self.get("is_linear")


    #@RDATTR GEOTRANS:GET('nbpts')
    def nbpts(self):
        """return the number of points of the geometric transformation."""
        return self.get("nbpts")


    #@GET GEOTRANS:GET('pts')
    def pts(self):
        """Return the reference convex points.

Synopsis: CvStruct.pts()
The points are stored in the columns of the output matrix."""
        return self.get("pts")


    #@GET GEOTRANS:GET('normals')
    def normals(self):
        """Get the normals for each face of the reference convex"""
        return self.get("normals")


    #@GET GEOTRANS:GET('transform')
    def transform(self, G, pts):
        """Apply the geometric transformation to a set of points.

Synopsis: CvStruct.transform(G,pts)
G is the set of vertices of the real convex, pts is the set of points (in the
reference convex) that are to be transformed. The corresponding set of points
in the real convex is returned."""
        return self.get("transform", G, pts)


    #@GET GEOTRANS:GET('char')
    def char(self):
        """Output a (unique) string representation of the @tgeotrans.

Synopsis: CvStruct.char()
This can be used to perform comparisons between two different @tgeotrans
objects."""
        return self.get("char")


class Fem:
    """FEM (Finite Element Method) objects."""
    def __init__(self, fem_name):
        """Build a FEM object from a string description.

* FEM_PK(N,K)
   classical Lagrange element PK on a simplex of dimension N
 * FEM_PK_DISCONTINUOUS(N,K[,alpha])}
    discontinuous Lagrange element PK on a simplex of dim N
 * FEM_QK(N,K)}
   classical Lagrange element QK on quadrangles, hexahedrons etc
 * FEM_QK_DISCONTINUOUS(N,K[,alpha])}
   discontinuous Lagrange element QK on quadrangles, hexahedrons etc
 * FEM_Q2_INCOMPLETE
   incomplete 2D Q2 element with 8 dof (serendipity Quad 8 element)  *
FEM_PK_PRISM(N,K)
   classical Lagrange element PK on a prism
 * FEM_PK_PRISM_DISCONTINUOUS(N,K[,alpha])
   classical discontinuous Lagrange element PK on a prism.
  * FEM_PK_WITH_CUBIC_BUBBLE(N,K)
   classical Lagrange element PK on a simplex with an additional volumic
bubble function.
 * FEM_P1_NONCONFORMING
   non-conforming P1 method on a triangle.
 * FEM_P1_BUBBLE_FACE(N)
   P1 method on a simplex with an additional bubble function on face 0.
 * FEM_P1_BUBBLE_FACE_LAG
   P1 method on a simplex with an additional lagrange dof on face 0.
  * FEM_PK_HIERARCHICAL(N,K)
   PK element with a hierarchical basis
 * FEM_QK_HIERARCHICAL(N,K)
   QK element with a hierarchical basis
 * FEM_PK_PRISM_HIERARCHICAL(N,K)
   PK element on a prism with a hierarchical basis
 * FEM_STRUCTURED_COMPOSITE(FEM, K)
   Composite fem on a grid with K divisions
 * FEM_PK_HIERARCHICAL_COMPOSITE(N,K,S)
   PK composite element on a grid with S subdivisions and with a hierarchical
basis
 * FEM_PK_FULL_HIERARCHICAL_COMPOSITE(N,K,S)
   PK composite element with S subdivisions and a hierarchical basis on both
degree and subdivision
 * FEM_PRODUCT(FEM1,FEM2)
   tensorial product of two polynomial elements
 * FEM_HERMITE(N)
   Hermite element P3 on a simplex of dimension N=1,2,3
 * FEM_ARGYRIS
   Argyris element P5 on the triangle.
 * FEM_HCT_TRIANGLE
    Hsieh-Clough-Tocher element on the triangle (composite P3   element which
is C^1), should be used with IM_HCT_COMPOSITE()   integration method.
 * FEM_QUADC1_COMPOSITE
   Quadrilateral element, composite P3 element and C^1 (16 dof).
 * FEM_REDUCED_QUADC1_COMPOSITE
   Quadrilateral element, composite P3 element and C^1 (12 dof).
 * FEM(:_(BRT0(N)
   Raviart-Thomas element of order 0 on a simplex of dimension N.
 * FEM(:_(BNEDELEC(N)
   Nedelec edge element of order 0 on a simplex of dimension N.

 * f=Fem('interpolated_fem', @mf MF1, @mim MIM2, [ivec blocked_dof])
  Build a special fem which is interpolated from another mesh_fem.  Using this
special finite element, it is possible to interpolate a given mesh_fem MF1 on
another mesh, given the integration method that will be used on this mesh.
Note that this finite element may be quite slow, and eats much memory.
        """
        generic_constructor(self,'fem',fem_name)
    def __del__(self):
        generic_destructor(self,destructible=False)
    def get(self, *args):
        return getfem('fem_get',self.id, *args)
    def __str__(self):
        return self.get('char')
    def __repr__(self):
        return '<getfem.Fem '+str(self)+'>'

    #@RDATTR FEM:GET('nbdof')
    def nbdof(self, CV=None):
        """Return the number of DOF for the FEM.

Synopsis: Fem.nbdof([, int CV])
Some specific FEM (for example 'interpolated_fem') may require a convex number
CV to give their result (interpolated fems for example). In most of the case,
you can omit this convex number."""
        return self.get("nbdof", CV)


    #@RDATTR FEM:GET('dim')
    def dim(self):
        """Return the dimension (dimension of the reference convex) of the FEM."""
        return self.get("dim")


    #@RDATTR FEM:GET('target_dim')
    def target_dim(self):
        """Return the dimension of the target space.

Synopsis: Fem.target_dim()
The target space dimension is usually 1, except for vector FEMs (none of them
has been implemented in getfem++ for now)"""
        return self.get("target_dim")


    #@GET FEM:GET('pts')
    def pts(self, CV=None):
        """Get the location of the degrees of freedom on the reference element.

Synopsis: Fem.pts([, int CV])
Some specific FEM may require a convex number CV to give their result
(interpolated fems for example). In most of the case, you can omit this convex
number."""
        return self.get("pts", CV)


    #@RDATTR FEM:GET('is_equivalent')
    def is_equivalent(self):
        """Return 0 if the FEM is not equivalent.

Synopsis: Fem.is_equivalent()
Equivalent FEM are evaluated on the reference convex. This is the case of most
classical FEMs."""
        return self.get("is_equivalent")


    #@RDATTR FEM:GET('is_lagrange')
    def is_lagrange(self):
        """Return 0 if the FEM is not of Lagrange type."""
        return self.get("is_lagrange")


    #@RDATTR FEM:GET('is_polynomial')
    def is_polynomial(self):
        """Return 0 if the basis functions are not polynomials."""
        return self.get("is_polynomial")


    #@RDATTR FEM:GET('estimated_degree')
    def estimated_degree(self):
        """Return an estimation of the polynomial degree of the FEM."""
        return self.get("estimated_degree")


    #@GET FEM:GET('base_value')
    def base_value(self, X):
        """Evaluate all base functions at the given point."""
        return self.get("base_value", X)


    #@GET FEM:GET('grad_base_value')
    def grad_base_value(self, X):
        """Evaluate the gradient of all base functions at the given point."""
        return self.get("grad_base_value", X)


    #@GET FEM:GET('hess_base_value')
    def hess_base_value(self, X):
        """Evaluate the Hessian of all base functions at the given point."""
        return self.get("hess_base_value", X)


    #@GET FEM:GET('poly_str')
    def poly_str(self):
        """Return (if possible) the polynomial expressions of the base functions in the
reference element.

Synopsis: Fem.poly_str()
This function will of course fail for non-polynomial FEMs."""
        return self.get("poly_str")


    #@GET FEM:GET('char')
    def char(self):
        """Ouput a (unique) string representation of the FEM.

Synopsis: Fem.char()
This can be used to perform comparisons between two different FEM objects."""
        return self.get("char")

    
class Integ:
    """Integration Method Objects."""
    def __init__(self, *args):
        """
Integ(stringing method) Return a FEM Integration Method from a string
description.

  Possible descriptions are (non exhaustive list):
 * IM_EXACT_SIMPLEX(n)
   Exact integration on simplices (works only with linear geometric
transformations and PK FEMs). Note that 'exact integration'   should be
avoided in general, since they only apply to linear   geometric
transformations, are quite slow, and subject to   numerical stability problems
for high degree FEMs.
 * IM_PRODUCT(a, b)   Product of two integration methods.

  * IM_EXACT_PARALLELEPIPED(n)
   Exact integration on parallelepipeds.

  * IM_EXACT_PRISM(n)
   Exact integration on prisms.

  * IM_GAUSS1D(K)
   Gauss method on the segment, order K (K=1,3,...99).

  * IM_NC(N,K)
   Newton-Cotes approximative integration on simplexes, order K.

  * IM_NC_PARALLELEPIPED(N,K)
   Product of Newton-Cotes integration on parallelepipeds.

  * IM_NC_PRISM(N,K)
   Product of Newton-Cotes integration on prisms.

  * IM_GAUSS_PARALLELEPIPED(N,K)
   Product of Gauss1D integration on parallelepipeds.

  * IM_TRIANGLE(K)
   Gauss methods on triangles (K=1,3,5,6,7,8,9,10,13,17,19).

  * IM_QUAD(K)
   Gauss methods on quadrilaterons (K=2, 3, 5, .. 17). Note that
IM_GAUSS_PARALLELEPIPED should be prefered for QK FEMs.

  * IM_TETRAHEDRON(K)
   Gauss methods on tetrahedrons (K=1, 2, 3, 5, 6 or 8).

  * IM_SIMPLEX4D(3)
   Gauss method on a 4-dimensional simplex.

  * IM_STRUCTURED_COMPOSITE(IM1, K)
    Composite method on a grid with K divisions.

  * IM_HCT_COMPOSITE(IM1)
   Composite integration suited to the HCT composite finite element.

  example:
   Integ('IM_PRODUCT(IM_GAUSS1D(5),IM_GAUSS1D(5))')
   is the same as:
   Integ('IM_GAUSS_PARALLELEPIPED(2,5)')
        """
        generic_constructor(self,'integ',*args)
    def __del__(self):
        generic_destructor(self,destructible=False)
    def get(self, *args):
        return getfem('integ_get',self.id, *args)
    def __str__(self):
        return self.get('char')
    def __repr__(self):
        return '<getfem.Integ '+str(self)+'>'

    #@RDATTR INTEG:GET('is_exact')
    def is_exact(self):
        """Return 0 if the integration is an approximate one."""
        return self.get("is_exact")


    #@RDATTR INTEG:GET('dim')
    def dim(self):
        """Return the dimension of the ref. convex of the method."""
        return self.get("dim")


    #@RDATTR INTEG:GET('nbpts')
    def nbpts(self):
        """Return the total number of integration points.

Synopsis: Integ.nbpts()
Count the points for the volume integration, and points for surface
integration on each face of the reference convex. Raises an error for exact
integration methods."""
        return self.get("nbpts")


    #@GET INTEG:GET('pts')
    def pts(self):
        """Return the list of integration points (only for approximate methods)."""
        return self.get("pts")


    #@GET INTEG:GET('coeffs')
    def coeffs(self):
        """Returns the coefficients associated to each integration point."""
        return self.get("coeffs")


    #@GET INTEG:GET('face_pts')
    def face_pts(self, F):
        """Return the list of integration points for a face."""
        return self.get("face_pts", F)


    #@GET INTEG:GET('face_coeffs')
    def face_coeffs(self, F):
        """Returns the coefficients associated to each integration of a face."""
        return self.get("face_coeffs", F)


    #@GET INTEG:GET('char')
    def char(self):
        """Ouput a (unique) string representation of the integration method.

Synopsis: Integ.char()
This can be used to  comparisons between two different @tinteg objects."""
        return self.get("char")


class Eltm:
    """Descriptor for an elementary matrix type."""
    def __init__(self, *args):
        """
Generates a descriptor for an elementary matrix type.

  * Eltm('base', @tfem FEM)
 Integration of shape functions on elements, using the fem FEM.

  * Eltm('grad', @tfem FEM)
 Integration of gradient of shape functions on elements, using the fem FEM.

  * Eltm('hessian', @tfem FEM)
 Integration of hessian of shape functions on elements, using the fem FEM.

  * Eltm('normal')
 The unit normal to the current convex face

  * Eltm('grad_geotrans')
 The gradient of the geometric transformation.  * Eltm('grad_geotrans_inv')
 The inverse of the gradient of the geometric transformation.  *
Eltm('product', @eltm A, @eltm B)
 Integration of the tensorial product of elementary matrices A and B.

  In order to obtain a numerical value of theses matrices, see MeshIm.eltm().
        """
        generic_constructor(self,'eltm',*args)
    def __del__(self):
        generic_destructor(self,destructible=False)

    
class CvStruct:
    """Descriptor for a convex structure."""
    def __init__(self, *args):
        generic_constructor(self,'cvstruct',*args)
    def __del__(self):
        generic_destructor(self,destructible=False)
    def get(self, *args):
        return getfem('cvstruct_get',self.id, *args)
    def __repr__(self):
        return '<getfem.CvStruct %dD, %d pts>' % (self.dim(),self.nbpts())

    #@RDATTR CVSTRUCT:GET('nbpts')
    def nbpts(self):
        """Get the number of points of the convex structure."""
        return self.get("nbpts")


    #@RDATTR CVSTRUCT:GET('dim')
    def dim(self):
        """Get the dimension of the convex structure."""
        return self.get("dim")


    #@RDATTR CVSTRUCT:GET('basic structure')
    def basic_structure(self):
        """Get the simplest structure.

Synopsis:  cs=CvStruct.basic_structure()
For example, the 'basic structure' of the 6-node triangle, is the canonical
3-noded triangle."""
        return self.get("basic structure")


    #@RDATTR CVSTRUCT:GET('face')
    def face(self, F):
        """Return the structure of the face F."""
        return self.get("face", F)


    #@GET CVSTRUCT:GET('facepts')
    def facepts(self, F):
        """Return the list of point indices for the face F."""
        return self.get("facepts", F)


class Poly:
    pass

class Slice:
    """Mesh slices.

    The slices may be considered as a (non-conformal) mesh of
    simplexes which provides fast interpolation on a P1-discontinuous
    MeshFem.
    
    It is used mainly for post-processing purposes.
    """
    def __init__(self, *args):
        """General constructor for Slice objects.

sl = Slice(sliceop, mesh M, int REFINE [, CVFLST])
 sl = Slice(sliceop, mesh_fem MF, vec U, int REFINE [, CVFLST])
 sl = Slice(sliceop, slice SL)
 sl = Slice('streamlines', mesh_fem MF, vec U, mat SEEDS)
 sl = Slice('points', mesh M, mat PTS)
 sl = Slice('load', mesh M)

  Creation of a mesh slice. Mesh slices are very similar to a P1-discontinuous
mesh_fem on which interpolation is very fast. The slice is built from a mesh
object, and a description of the slicing operation, for example,

  sl = Slice(('planar',+1,[0;0],[1;0]), m, 5);

  cuts the original mesh with the half space {y>0}. Each convex of the
original mesh m is simplexified (for example a quadrangle is splitted into 2
triangles), and each simplex is refined 5 times.

  Slicing operations can be:
  - cutting with a plane, a sphere or a cylinder
  - intersection or union of slices
  - isovalues surfaces/volumes
  - "points", "streamlines" (see below)

  If the first argument is a mesh_fem mf instead of a mesh, and if it is
followed by a field U , then the deformation U will be applied to the mesh
before the slicing operation.

  The first argument can also be a slice.

  Slicing operations:
===============
 They are specified with TUPLES, do not forget the extra parentheses!. The
first element is the name of the operation, followed the slicing options.

   * ('none')
  Does not cut the mesh.

  * ('planar', orient, p, n)
   Planar cut. p and n define a half-space, p being a point belong to the
boundary of the half-space, and n being its normal. If orient is equal to -1
(resp. 0, +1), then the slicing operation will cut the mesh with the
"interior" (resp. "boundary", "exterior") of the half-space. Orient may also
be set to +2 which means that the mesh will be sliced, but both the outer and
inner parts will be kept.

  * ('ball', orient, c, r)
  Cut with a ball of center c and radius r.

  * ('cylinder', orient, p1, p2, r)
  Cut with a cylinder whose axis is the line (p1,p2) and whose radius is r.

  * ('isovalues',orient, mesh_fem MF, vec U, scalar V)
  Cut using the isosurface of the field U (defined on the mesh_fem MF). The
result is the set {x such that U(x) <= V} or {x such that U(x) == V} or {x
such that U(x) <= V} depending on the value of ORIENT.

  * ('boundary'[, SLICEOP])
  Return the boundary of the result of SLICEOP, where SLICEOP is any slicing
operation. If SLICEOP is not specified, then the whole mesh is considered
(i.e. it is equivalent to ('boundary',{'none'})).

  * ('explode', coef)
  Build an 'exploded' view of the mesh: each convex is shrinked (0 < coef <=
1). In the case of 3D convexes, only their faces are kept.

  * ('union', SLICEOP1, SLICEOP2)
 * ('intersection', SLICEOP1, SLICEOP2)
 * ('comp', SLICEOP)
 * ('diff', SLICEOP1, SLICEOP2)
  Boolean operations: returns the union,intersection,complementary or
difference of slicing operations.

  * ('mesh', MESH)
  Build a slice which is the intersection of the sliced mesh with another
mesh. The slice is such that all of its simplexes are stricly contained into a
convex of each mesh.

  EXAMPLE:
  sl = Slice((intersection',('planar',+1,[0;0;0],[0;0;1]),...
               ('isovalues',-1,mf2,U2,0)),mf,U,5);

   SPECIAL SLICES:

  There are also some special calls to Slice(...)

  * Slice('streamlines',mf, U, mat SEEDS)

  compute streamlines of the (vector) field U, with seed points given by the
columns of SEEDS.

  * Slice('points', m, mat PTS)

  return the "slice" composed of points given by the columns of PTS (useful
for interpolation on a given set of sparse points, see
compute_interpolate_on(mf, U, sl).  * Slice('load', filename [, m])

  load the slice (and its linked_mesh if it is not given as an argument) from
a text file.
"""
        generic_constructor(self,'slice',*args)
    def __del__(self):
        generic_destructor(self,destructible=True)
    def get(self, *args):
        return getfem('slice_get',self.id, *args)
    def set(self, *args):
        return getfem('slice_set',self.id, *args)
    def mesh(self):
        return self.get('linked_mesh')

    #@RDATTR SLICE:GET('dim')
    def dim(self):
        """Return the dimension of the slice (2 for a 2D mesh, etc..)."""
        return self.get("dim")


    #@GET SLICE:GET('area')
    def area(self):
        """Return the area of the slice."""
        return self.get("area")


    #@GET SLICE:GET('cvs')
    def cvs(self):
        """Return the list of convexes of the original mesh contained in the slice."""
        return self.get("cvs")


    #@RDATTR SLICE:GET('nbpts')
    def nbpts(self):
        """Return the number of points in the slice."""
        return self.get("nbpts")


    #@GET SLICE:GET('pts')
    def pts(self):
        """Return the list of point coordinates."""
        return self.get("pts")


    #@RDATTR SLICE:GET('nbsplxs')
    def nbsplxs(self, DIM=None):
        """Return the number of simplexes in the slice.

Synopsis: Slice.nbsplxs([,int DIM])
Since the slice may contain points (simplexes of dim 0), segments (simplexes
of dimension 1), triangles etc., the result is a vector of size Slice.dim()+1
, except if the optional argument DIM is used."""
        return self.get("nbsplxs", DIM)


    #@GET SLICE:GET('splxs')
    def splxs(self, DIM):
        """Return the list of simplexes of dimension DIM.

Synopsis: [S,CV2SPLX]=Slice.splxs( int DIM)
On output, S has DIM+1 rows, each column contains the point numbers of a
simplex.  The vector CV2SPLX can be used to find the list of simplexes for any
convex stored in the slice. For example S[:,CV2SPLX[4]:CV2SPLX[5]] gives the
list of simplexes for the fourth convex."""
        return self.get("splxs", DIM)


    #@GET SLICE:GET('edges')
    def edges(self):
        """Return the edges of the linked mesh contained in the slice.

Synopsis: [mat P, ivec E1, ivec E2]=Slice.edges()
P contains the list of all edge vertices, E1 contains the indices of each mesh
edge in P, and E2 contains the indices of each "edges" which is on the border
of the slice. This function is useless except for post-processing purposes."""
        return self.get("edges")


    #@GET SLICE:GET('interpolate_convex_data')
    def interpolate_convex_data(self, Ucv):
        """Interpolate data given on each convex of the mesh to the slice nodes.  The
input array Ucv may have any number of dimensions, but its last dimension
should be equal to Mesh.max_cvid().  Example of use:
Slice.interpolate_convex_data( Mesh.quality())"""
        return self.get("interpolate_convex_data", Ucv)


    #@GET SLICE:GET('linked mesh')
    def linked_mesh(self):
        """Return the mesh on which the slice was taken."""
        return self.get("linked mesh")


    #@GET SLICE:GET('export to vtk')
    def export_to_vtk(self, FILENAME, *args):
        """Export a slice to VTK.

Synopsis: Slice.export_to_vtk( string FILENAME ... [, 'ascii'][, 'edges'] ...)
Following the file name, you may use any of the following options:    - if
'ascii' is not used, the file will contain binary data (non portable, but
fast).    - if 'edges' is used, the edges of the original mesh will be written
instead of the slice content.  More than one dataset may be written, just list
them. Each dataset consists of either:    * a field interpolated on the slice,
followed by an optional name.   * a mesh_fem and a field, followed by an
optional name.  The field might be a scalar field, a vector field or a tensor
field.  examples: @Slice.export_to_vtk( 'test.vtk', Uslice, 'first_dataset',
mf, U2, 'second_dataset') @Slice.export_to_vtk( 'test.vtk', 'ascii', mf, U2)
@Slice.export_to_vtk( 'test.vtk', 'edges', 'ascii', Uslice)"""
        return self.get("export to vtk", FILENAME, *args)


    #@GET SLICE:GET('export to pov')
    def export_to_pov(self, FILENAME, *args):
        """Export a the triangles of the slice to POV-RAY."""
        return self.get("export to pov", FILENAME, *args)


    #@GET SLICE:GET('export to dx')
    def export_to_dx(self, FILENAME, *args):
        """Export a slice to OpenDX.

Synopsis: Slice.export_to_dx( string FILENAME, ...)
Following the file name, you may use any of the following options:

    - if 'ascii' is not used, the file will contain binary data   (non
portable, but fast).    - if 'edges' is used, the edges of the original mesh
will be   written instead of the slice content.    - if 'append' is used, the
opendx file will not be   overwritten, and the new data will be added at the
end of the   file.  More than one dataset may be written, just list them. Each
dataset consists of either:

    - a field interpolated on the slice (scalar, vector or   tensor), followed
by an optional name.    - a mesh_fem and a field, followed by an optional
name."""
        return self.get("export to dx", FILENAME, *args)


    #@GET SLICE:GET('memsize')
    def memsize(self):
        """Return the amount of memory (in bytes) used by the slice object."""
        return self.get("memsize")


    #@SET SLICE:SET('pts')
    def set_pts(self, P):
        """Replace the points of the slice.

Synopsis: Slice.set_pts( mat P)
The new points P are stored in the columns the matrix. Note that you can use
the function to apply a deformation to a slice, or to change the dimension of
the slice (the number of rows of P is not required to be equal to
Slice.dim())."""
        return self.set("pts", P)


class Spmat:
    """Getfem sparse matrix."""
    def __init__(self, *args):
        """
General constructor for getfem sparse matrices .

  These sparse matrix can be stored as CSC (compressed column sparse), which
is the format used by Matlab, or they can be stored as WSC (internal format to
getfem). The CSC matrices are not writable (it would be very inefficient), but
they are optimized for multiplication with vectors, and memory usage. The WSC
are writable, they are very fast with respect to random read/write operation.
However their memory overhead is higher than CSC matrices, and they are a
little bit slower for matrix-vector multiplications.

  By default, all newly created matrices are build as WSC matrices. This can
be changed later with Spmat.to_csc(), or may be changed automatically by
getfem (for example linsolve_() converts the matrices to CSC).

  The matrices may store REAL or COMPLEX values.

  * Spmat('empty', @tint m [, @tint n])
 Create a new empty (i.e. full of zeros) sparse matrix, of dimensions m x n.
If n is omitted, the matrix dimension is m x m.

  * Spmat('identity', @tint n)
 Create a n x n identity matrix.

  * Spmat.diag( mat D [, ivec E [, int n [,int m]]]) Create a diagonal matrix.
If E is given, D might be a matrix and each column of E will contain the sub-
diagonal number that will be filled with the corresponding column of D.

  * Spmat('copy', @spmat K [,I [,J]])
  Duplicate a matrix K (which might be a gfSpmat ). If I and/or J are given,
the matrix M will be a submatrix of K. For example
  M = gf_spmat('copy', sprand(50,50,.1), 1:40, [6 7 8 3 10])
 will return a 40x5 matrix.

  * Spmat('mult', @spmat A, @spmat B)
 Create a sparse matrix as the product of the sparse matrices A and B.  It
requires that A and B be both real or both complex, you may have to use
Spmat.to_complex()

  * Spmat('add', @spmat A, @spmat B)
 Create a sparse matrix as the sum of the sparse matrices A and B. Adding a
real matrix with a complex matrix is possible.

  * Spmat('load','hb'|'harwell-boeing', filename)
 Read a sparse matrix from an Harwell-Boeing file.

  * Spmat('mm', filename)
 * Spmat('matrix-market', filename)
 Read a sparse matrix from a Matrix-Market file.
        """
        generic_constructor(self,'spmat',*args)
    def __del__(self):
        generic_destructor(self,destructible=True)
    def __str__(self):
        return self.get('info')
    def __repr__(self):
        return '<getfem.Spmat '+str(self)+'>'
    def __getitem__(self, key):
        return getfem('spmat_get',self.id, 'full',*key)
    def __setitem__(self, key, keyval):
        getfem('spmat_set', self.id, 'assign', key[0], key[1], keyval)
    def __neg__(self):
        m=Spmat('copy',self)
        m.scale(-1)
        return m
    def __add__(self, other):
        return Spmat('add',self,other)
    def __sub__(self, other):
        return Spmat('add',self,other.__neg__())
    def __mul__(self, other):
        """Multiplication of a Spmat with another Spmat or a vector or a scalar.

        The result is another Spmat object.
        """
        if (isinstance(other,(int,float,complex))):
            m=Spmat('copy',self)
            m.set('scale',other)
        elif (isinstance(other,list) or isinstance(other, numarray.NDArray)):
            m=self.mult(other)
        else:
            m=Spmat('mult',self,other)
        return m;
    def __rmul__(self, other):
        if (isinstance(other,(int,float,complex))):
            m=Spmat('copy',self)
            m.set('scale',other)
        elif (isinstance(other,list) or isinstance(other, numarray.NDArray)):
            m=self.tmult(other)
        else:
            m=Spmat('mult',other,self)
        return m;
    def get(self, *args):
        return getfem('spmat_get',self.id, *args)
    def set(self, *args):
        return getfem('spmat_set',self.id, *args)

    #@GET SPMAT:GET('size')
    def size(self):
        """Return a vector [ni, nj] where ni and nj are the dimensions of the matrix."""
        return self.get("size")


    #@GET SPMAT:GET('nnz')
    def nnz(self):
        """Return the number of non-null values stored in the sparse matrix."""
        return self.get("nnz")


    #@GET SPMAT:GET('is_complex')
    def is_complex(self):
        """Return 1 if the matrix contains complex values."""
        return self.get("is_complex")


    #@GET SPMAT:GET('storage')
    def storage(self):
        """Return the storage type currently used for the matrix.

Synopsis: Spmat.storage()
The storage is returned as a string, either 'CSC' or 'WSC'."""
        return self.get("storage")


    #@GET SPMAT:GET('full')
    def full(self, *args):
        """Return a full (sub-)matrix of M.

Synopsis: Spmat.full([,I [,J]])
The optional arguments I, are the sub-intervals for the rows and columns that
are to be extracted."""
        return self.get("full", *args)


    #@GET SPMAT:GET('mult')
    def mult(self, V):
        """Product of the sparse matrix M with a vector V.

Synopsis: Spmat.mult( V)
For matrix-matrix multiplications, see Spmat('mult')"""
        return self.get("mult", V)


    #@GET SPMAT:GET('tmult')
    def tmult(self, V):
        """Product of M transposed (conjugated if M is complex) with the vector V."""
        return self.get("tmult", V)


    #@GET SPMAT:GET('diag')
    def diag(self, E=None):
        """Return the diagonal of M as a vector.

Synopsis: Spmat.diag([, E])
If E is used, return the sub-diagonals whose ranks are given in E."""
        return self.get("diag", E)


    #@GET SPMAT:GET('csc_ind')
    def csc_ind(self):
        """Return the two usual index arrays of CSC storage.

Synopsis: [JC,IR]=Spmat.csc_ind()
If M is not stored as a CSC matrix, it is converted into CSC."""
        return self.get("csc_ind")


    #@GET SPMAT:GET('csc_val')
    def csc_val(self):
        """Return the array of values of all non-zero entries of M.

Synopsis: [V]=Spmat.csc_val()
If M is not stored as a CSC matrix, it is converted into CSC."""
        return self.get("csc_val")


    #@GET SPMAT:GET('dirichlet nullspace')
    def dirichlet_nullspace(self, R):
        """Solve the dirichlet conditions MU=R.

Synopsis: [N,U0]=Spmat.dirichlet_nullspace( vec R)
A solution U0 which has a minimum L2-norm is returned, with a sparse matrix N
containing an orthogonal basis of the kernel of the (assembled) constraints
matrix M (hence, the PDE linear system should be solved on this subspace): the
initial problem

  KU = B with constraints MU=R

  is replaced by

    (N'*K*N)*UU = N'*B
             U = N*UU + U0"""
        return self.get("dirichlet nullspace", R)


    #@GET SPMAT:GET('info')
    def info(self):
        """Return a string contains a short summary on the sparse matrix (dimensions,
filling, ..)."""
        return self.get("info")


    #@GET SPMAT:GET('save')
    def save(self, format, filename):
        """Export the sparse matrix.

Synopsis: Spmat.save( string format, string filename)
the format of the file may be 'hb' for Harwell-Boeing, or 'mm' for Matrix-
Market."""
        return self.get("save", format, filename)


    #@SET SPMAT:SET('clear')
    def clear(self, *args):
        """Erase the non-zero entries of the matrix.

Synopsis: Spmat.clear([, I[, J]])
The optional arguments I and J may be specified to clear a sub-matrix instead
of the entire matrix."""
        return self.set("clear", *args)


    #@SET SPMAT:SET('scale')
    def scale(self, V):
        """Multiplies the matrix by a scalar value V."""
        return self.set("scale", V)


    #@SET SPMAT:SET('transpose')
    def transpose(self):
        """Transpose the matrix."""
        return self.set("transpose")


    #@SET SPMAT:SET('conjugate')
    def conjugate(self):
        """Conjugate each element of the matrix."""
        return self.set("conjugate")


    #@SET SPMAT:SET('transconj')
    def transconj(self):
        """Transpose and conjugate the matrix."""
        return self.set("transconj")


    #@SET SPMAT:SET('to_csc')
    def to_csc(self):
        """Convert the matrix to CSC storage.

Synopsis: Spmat.to_csc()
CSC storage is recommended for matrix-vector multiplications."""
        return self.set("to_csc")


    #@SET SPMAT:SET('to_wsc')
    def to_wsc(self):
        """Convert the matrix to WSC storage.

Synopsis: Spmat.to_wsc()
Read and write operation are quite fast with WSC storage."""
        return self.set("to_wsc")


    #@SET SPMAT:SET('to_complex')
    def to_complex(self):
        """Store complex numbers."""
        return self.set("to_complex")


    #@SET SPMAT:SET('diag')
    def diag(self, D, E=None):
        """Change the diagonal (or sub-diagonals) of the matrix.

Synopsis: Spmat.diag( mat D [, ivec E])
If E is given, D might be a matrix and each column of E will contain the sub-
diagonal number that will be filled with the corresponding column of D."""
        return self.set("diag", D, E)


    #@SET SPMAT:SET('assign')
    def assign(self, I, J, V):
        """Copy V into the sub-matrix M(I,J).

Synopsis: Spmat.assign( ivec I, ivec J, V)
V might be a sparse matrix or a full matrix."""
        return self.set("assign", I, J, V)


    #@SET SPMAT:SET('add')
    def add(self, I, J, V):
        """Add V to the sub-matrix M(I,J).

Synopsis: Spmat.add( I, J, V)
V might be a sparse matrix or a full matrix."""
        return self.set("add", I, J, V)


    
class Precond:
    """Getfem preconditioner."""
    def __init__(self, *args):
        """
General constructor for getfem preconditioners.

  The preconditioners may store REAL or COMPLEX values. They accept getfem
sparse matrices and Matlab sparse matrices.

  * Precond('identity')
 Create a REAL identity precondioner.

  * Precond('cidentity')
 Create a COMPLEX identity precondioner.

  * Precond('diagonal', @dcvec D)
 Create a diagonal precondioner.

  * Precond('ildlt', @spmat M)
 Create an ILDLT (Cholesky) preconditioner for the (symmetric) sparse matrix
M.
 This preconditioner has the same sparsity pattern than M (no fill-in).

  * Precond('ilu', @spmat M)
 Create an ILU (Incomplete LU) preconditioner for the sparse matrix M.
 This preconditioner has the same sparsity pattern than M (no fill-in).

  * Precond('ildltt', @spmat M [, int fillin [, scalar threshold]])
 Create an ILDLT (Cholesky with filling) preconditioner for the (symmetric)
sparse matrix M.
  The preconditioner may add at most 'fillin' additional non-zero entries on
each line. The default value for 'fillin' is 10, and the default threshold is
1e-7.

  * Precond('ilut', @spmat M [, int fillin [, scalar threshold]])
 Create an ILUT (Incomplete LU with filling) preconditioner for the sparse
matrix M.
  The preconditioner may add at most 'fillin' additional non-zero entries on
each line. The default value for 'fillin' is 10, and the default threshold is
1e-7.

  * Precond('superlu', @spmat M)
 Uses SuperLU to build an exact factorization of the sparse matrix M.  This
preconditioner is only available if the getfem-interface was built with
SuperLU support. Note that LU factorization is likely to eat all your memory
for 3D problems.
        """
        generic_constructor(self,'precond',*args)
    def __del__(self):
        generic_destructor(self,destructible=True)
    def get(self, *args):
        return getfem('precond_get',self.id, *args)
    def set(self, *args):
        return getfem('precond_set',self.id, *args)

    #@GET PRECOND:GET('mult')
    def mult(self, V):
        """Apply the preconditioner to the supplied vector."""
        return self.get("mult", V)


    #@GET PRECOND:GET('tmult')
    def tmult(self, V):
        """Apply the transposed preconditioner to the supplied vector."""
        return self.get("tmult", V)


    #@GET PRECOND:GET('type')
    def type(self):
        """Return a string describing the type of the preconditioner ('ilu', 'ildlt',..)."""
        return self.get("type")


    #@GET PRECOND:GET('size')
    def size(self):
        """Return the dimensions of the preconditioner."""
        return self.get("size")


    #@GET PRECOND:GET('is_complex')
    def is_complex(self):
        """Return 1 if the preconditioner stores complex values."""
        return self.get("is_complex")


    #@GET PRECOND:GET('info')
    def info(self):
        """Return a short informative string about the preconditioner."""
        return self.get("info")





#@FUNC ::LINSOLVE('gmres')
def linsolve_gmres(*args):
    """Solve MX=b with the generalized minimum residuals method, using P as a
preconditioner.

Synopsis: linsolve_('gmres', @spmat M, vec b [, int restart][, @precond
P][,'noisy'][,'res', r][,'maxiter', n])
The default value of the restart parameter is 50."""
    return getfem('linsolve','gmres', *args)

#@FUNC ::LINSOLVE('cg')
def linsolve_cg(*args):
    """Solve MX=b with the conjugated gradient method, using P as a preconditioner."""
    return getfem('linsolve','cg', *args)

#@FUNC ::LINSOLVE('bicgstab')
def linsolve_bicgstab(*args):
    """Solve MX=b with the bi-conjugated gradient stabilized method, using P as a
preconditioner."""
    return getfem('linsolve','bicgstab', *args)

#@FUNC ::LINSOLVE('lu')
def linsolve_lu(M, b, P=None):
    """Alias for linsolve_('superlu')"""
    return getfem('linsolve','lu', M, b, P)

#@FUNC ::LINSOLVE('superlu')
def linsolve_superlu(M, b, P=None):
    """Apply the SuperLU solver (sparse LU factorization). The condition number
estimate is returned with the solution."""
    return getfem('linsolve','superlu', M, b, P)


#@FUNC ::COMPUTE('L2 norm')
def compute_L2_norm(mf,U, mim, CVLST=None):
    """Compute the L2 norm of the (real or complex) field U.

Synopsis: compute_L2 norm(mf, U,  meshim mim [,CVLST])
If CVLST is indicated, the norm will be computed only on the listed convexes."""
    return getfem('compute',mf,U,'L2 norm', mim, CVLST)

#@FUNC ::COMPUTE('H1 semi norm')
def compute_H1_semi_norm(mf,U, mim, CVLST=None):
    """Compute the L2 norm of grad(U).

Synopsis: N=compute_H1 semi norm(mf, U,  meshim mim [,CVLST])
If CVLST is given, the norm will be computed only on the listed convexes."""
    return getfem('compute',mf,U,'H1 semi norm', mim, CVLST)

#@FUNC ::COMPUTE('H1 norm')
def compute_H1_norm(mf,U, mim, CVLST=None):
    """Compute the H1 norm of U.

Synopsis: N = compute_H1 norm(mf, U,  meshim mim [,CVLST])
If CVLST is given, the norm will be computed only on the listed convexes."""
    return getfem('compute',mf,U,'H1 norm', mim, CVLST)

#@FUNC ::COMPUTE('H2 semi norm')
def compute_H2_semi_norm(mf,U, mim, CVLST=None):
    """Compute the L2 norm of D^2(U).

Synopsis: N = compute_H2 semi norm(mf, U,  meshim mim [,CVLST])
If CVLST is given, the norm will be computed only on the listed convexes."""
    return getfem('compute',mf,U,'H2 semi norm', mim, CVLST)

#@FUNC ::COMPUTE('H2 norm')
def compute_H2_norm(mf,U, mim, CVLST=None):
    """Compute the H2 norm of U.

Synopsis: N = compute_H2 norm(mf, U,  meshim mim [,CVLST])
If CVLST is given, the norm will be computed only on the listed convexes."""
    return getfem('compute',mf,U,'H2 norm', mim, CVLST)

#@FUNC ::COMPUTE('gradient')
def compute_gradient(mf,U, MFGRAD):
    """Compute the gradient of the field U defined on meshfem MFGRAD.

Synopsis: DU = compute_gradient(mf, U,  meshfem MFGRAD)
The gradient is interpolated on the mesh_fem MFGRAD, and returned in DU.  For
example, if U is defined on a P2 mesh_fem, DU should be evaluated on a
P1-discontinuous mesh_fem. MF and MFGRAD should share the same mesh.  U may
have any number of dimensions (i.e. this function is not restricted to the
gradient of scalar fields, but may also be used for tensor fields). However
the last dimension of U has to be equal to the number of dof of MF. For
example, if U is a 3x3xnbdof(MF) array, DU will be a Nx3x3[xQ]xnbdof(MFGRAD)
array, where N is the dimension of the mesh, and the optional Q dimension is
inserted if qdim(MF) != qdim(MFGRAD)."""
    return getfem('compute',mf,U,'gradient', MFGRAD)

#@FUNC ::COMPUTE('hessian')
def compute_hessian(mf,U, MFHESS):
    """Compute the hessian of the field U defined on meshfem MFHESS."""
    return getfem('compute',mf,U,'hessian', MFHESS)

#@FUNC ::COMPUTE('interpolate on')
def compute_interpolate_on(mf,U, *args):
    """Interpolate a field on another mesh_fem or a slice.

Synopsis: U2 = compute_interpolate_on(mf, U,  { meshfem MF2 | slice SL })
* interpolation on another mesh_fem MF2: MF2 has to be Lagrangian.  If MF and
MF2 share the same mesh object, the  interpolation will be much faster.

  * interpolation on a slice object: this is similar to interpolation on a
refined P1-discontinuous mesh, but it is much faster.  This can also be used
with Slice('points') to obtain field values at a given set of points.

See also asm_interpolation_matrix()"""
    return getfem('compute',mf,U,'interpolate on', *args)

#@FUNC ::COMPUTE('extrapolate on')
def compute_extrapolate_on(mf,U, MF2):
    """Extrapolate a field on another mesh_fem.

Synopsis: U2 = compute_extrapolate_on(mf, U,  meshfem MF2)
If the mesh of MF2 is stricly included in the mesh of MF, this function does
stricly the same job as compute_interpolate_on(mf, U, ). However, if the mesh
of MF2 is not exactly included in MF (imagine interpolation between a curved
refined mesh and a coarse mesh), then values which are slightly outside MF
will be extrapolated.

See also asm_extrapolation_matrix()"""
    return getfem('compute',mf,U,'extrapolate on', MF2)

#@FUNC ::COMPUTE('error estimate')
def compute_error_estimate(mf,U, MIM):
    """Compute an a posteriori error estimation.

Synopsis: E=compute_error_estimate(mf, U,  meshim MIM)
Currently there is only one which is available: for each convex, the jump of
the normal derivative is integrated on its faces."""
    return getfem('compute',mf,U,'error estimate', MIM)


#@FUNC ::ASM('volumic source')
def asm_volumic_source(mim, mf_u, mf_d, Fd):
    """Assembly of a volumic source term.

Synopsis: [vec V]=asm_volumic_source( meshim mim, meshfem mf_u, meshfem mf_d,
@dcvec Fd)
Output a vector V, assembled on the mesh_fem MF_U, using the data vector FD
defined on the data mesh_fem MF_D. FD may be real or complex-valued."""
    return getfem('asm','volumic source', mim, mf_u, mf_d, Fd)

#@FUNC ::ASM('boundary source')
def asm_boundary_source(boundary_num, mim, mf_u, mf_d, G):
    """Assembly of a boundary source term.

Synopsis: asm_boundary_source( boundary_num, meshim mim, meshfem mf_u, meshfem
mf_d, vec G)
G should be a [QDIM x N] matrix, where N is the number of degree of freedom of
mf_d, and QDIM=MESHFEM::GET{mf_u}('qdim')."""
    return getfem('asm','boundary source', boundary_num, mim, mf_u, mf_d, G)

#@FUNC ::ASM('mass matrix')
def asm_mass_matrix(mim, mf1, mf2=None):
    """Assembly of a mass matrix."""
    return getfem('asm','mass matrix', mim, mf1, mf2)

#@FUNC ::ASM('laplacian')
def asm_laplacian(mim, mf_u, mf_d, A):
    """Assembly of the Laplacian ( div(a(x)*grad U) with a scalar)."""
    return getfem('asm','laplacian', mim, mf_u, mf_d, A)

#@FUNC ::ASM('linear elasticity')
def asm_linear_elasticity(mim, mf_u, mf_d, lambda_d, mu_d):
    """Assembles of linear elasticity.

Synopsis: asm_linear_elasticity( meshim mim, meshfem mf_u, meshfem mf_d, vec
lambda_d, vec mu_d)
Lambda and mu are the usual Lam coefficients."""
    return getfem('asm','linear elasticity', mim, mf_u, mf_d, lambda_d, mu_d)

#@FUNC ::ASM('stokes')
def asm_stokes(mim, mf_u, mf_p, mf_data, visc):
    """Assembly of matrices for the Stokes problem { visc(x).Delta(U) - p = 0, div(U)
= 0 }.

Synopsis: [K,B]=asm_stokes( meshim mim, meshfem mf_u, meshfem mf_p, meshfem
mf_data, vec visc)
On output, K is the usual linear elasticity stiffness matrix with lambda=0 and
2*mu=visc. B is a sparse matrix corresponding to $\int p.div v$."""
    return getfem('asm','stokes', mim, mf_u, mf_p, mf_data, visc)

#@FUNC ::ASM('helmholtz')
def asm_helmholtz(mim, mf_u, mf_data, k2):
    """Assembly of the matrix for the Helmholtz problem { Laplacian(u) + k^2u = 0 }"""
    return getfem('asm','helmholtz', mim, mf_u, mf_data, k2)

#@FUNC ::ASM('bilaplacian')
def asm_bilaplacian(mim, mf_u, mf_data, a):
    """Assembly of the matrix for the Bilaplacian problem {
Laplacian(a(x)*Laplacian(u)) }"""
    return getfem('asm','bilaplacian', mim, mf_u, mf_data, a)

#@FUNC ::ASM('dirichlet')
def asm_dirichlet(boundary_num, mim, mf_u, mf_d, H, R, threshold=None):
    """Assembly of Dirichlet conditions of type 'h.u = r'.

Synopsis: [HH,RR]=asm_dirichlet( int boundary_num, meshim mim, meshfem mf_u,
meshfem mf_d, mat H, vec R [, threshold])
Handle 'h.u = r' where h is a square matrix (of any rank) whose size is equal
to MESHFEM:GET{mf_u}('qdim'). This matrix is stored in 'H', one column per dof
in 'mf_d', each column containing the values of the matrix 'h' stored in
fortran order: H(:,j) = [h11(x_j) h21(x_j) h12(x_j) h22(x_j)] if u is a 2D
vector field.

  Of course, if the unknown is a scalar field, you just have to set H =
ones(1, MESFEM:GET{mf_d}('nbdof')).

  This is basically the same than calling asm_boundary_qu_term() for H and
calling asm_neumann() for R, except that this function tries to produce a
'better' (more diagonal) constraints matrix (when possible).  see also
Spmat.Dirichlet_nullspace()."""
    return getfem('asm','dirichlet', boundary_num, mim, mf_u, mf_d, H, R, threshold)

#@FUNC ::ASM('boundary qu term')
def asm_boundary_qu_term(boundary_num, mim, mf_u, mf_d, q):
    """Q should be be a [Qdim x Qdim x N] array, where N is the number of degree of
freedom of mf_d, and QDIM is dimension of the unkown u (that is set when
creating the mesh_fem)."""
    return getfem('asm','boundary qu term', boundary_num, mim, mf_u, mf_d, q)

#@FUNC ::ASM('volumic')
def asm_volumic(CV=None, *args):
    """Generic assembly procedure for volumic assembly.

Synopsis: [...]=asm_volumic([,CVLST], expr [[, mesh_ims, mesh_fems, data...]])
The expression 'expr' is evaluated over the mesh_fems listed in the arguments
(with optional data) and assigned to the output arguments. For details about
the syntax of assembly expressions, please refer to the getfem user manual (or
look at the file getfem_assembling.h in the getfem++ sources).

  For example, the L2 norm of a field can be computed with

 compute_L2 norm(mf, U, ) or with:

  asm_volumic('u=data(#1);
V()+=u(i).u(j).comp(Base(#1).Base(#1))(i,j)',mim,mf,U)

  The Laplacian stiffness matrix can be evaluated with

 asm_Laplacian(mim, mf, A) or equivalently with:

 asm_volumic('a=data(#2);M(#1,#1)+=sym(comp(Grad(#1).Grad(#1).Base(#2))(:,i,:,
i,j).a(j))', mim,mf, A);"""
    return getfem('asm','volumic', CV, *args)

#@FUNC ::ASM('boundary')
def asm_boundary(*args):
    """Generic boundary assembly.

Synopsis: [...]=asm_boundary( @tint bnum, string expr [[, mesh_ims, mesh_fems,
data...]])
See the help for asm_volumic()."""
    return getfem('asm','boundary', *args)

#@FUNC ::ASM('interpolation matrix')
def asm_interpolation_matrix(MF1, MF2):
    """Build the interpolation matrix from a meshfem onto another meshfem.

Synopsis: M=asm_interpolation_matrix( MF1, MF2)
Return a sparse matrix M, such that V=M*U is equal to
::COMPUTE{MF1,U}('interpolate_on',MF2). Useful for repeated interpolations.
Note that this is just interpolation, no elementary integrations are involved
here, and MF2 has to be lagrangian. In the more general case, you would have
to do a L2 projection via the mass matrix."""
    return getfem('asm','interpolation matrix', MF1, MF2)

#@FUNC ::ASM('extrapolation matrix')
def asm_extrapolation_matrix(MF1, MF2):
    """Build the extrapolation matrix from a meshfem onto another meshfem.

Synopsis: M=asm_extrapolation_matrix( MF1, MF2)
Return a sparse matrix M, such that V=M*U is equal to
::COMPUTE{MF1,U}('extrapolate_on',MF2). Useful for repeated extrapolations."""
    return getfem('asm','extrapolation matrix', MF1, MF2)


#@FUNC ::UTIL('trace level')
def util_trace_level(level):
    """Set the verbosity of some getfem++ routines (typically the messages printed by
the model bricks), 0 means no trace message (default is 3)."""
    return getfem('util','trace level', level)

#@FUNC ::UTIL('warning level')
def util_warning_level(level):
    """Filter the less important warnings displayed by getfem. 0 means no warnings,
default level is 3."""
    return getfem('util','warning level', level)

class LevelSet:
    """Getfem Level-set."""
    def __init__(self, *args):
        """
General constructor for LevelSet objects.
        """
        generic_constructor(self,'levelset',*args)
    def __del__(self):
        generic_destructor(self,destructible=True)
    def get(self, *args):
        return getfem('levelset_get',self.id, *args)
    def set(self, *args):
        return getfem('levelset_set',self.id, *args)


def memstats():
    print "*** Getfem view of the workspace:"
    getfem('workspace','stats')
    print "*** Python view of the workspace:"
    for id,c in obj_count.iteritems():
        if (c):
            name=str(factory(id).__class__)
            print "%s class %d, id %d : instances=%d" % (name,id.classid,id.objid,c)


def linsolve(what, *args):
    return getfem('linsolve', what, *args)
def compute(mf, U, what, *args):
    return getfem('compute', mf, U, what, *args)
def asm(what, *args):
    return getfem('asm', what, *args)
def util(what, *args):
    return getfem('util', what, *args)


###
#register_types(Mesh,MeshFem,GeoTrans,Fem,Integ,Eltm,CvStruct,Poly,Slice)

def factory(id):
    # must be in the same order than enum getfemint_class_id in gfi_array.h
    t = (Mesh,MeshFem,MeshIm,MdBrick,MdState,GeoTrans,Fem,Integ,Eltm,CvStruct,Poly,Slice,Spmat,Precond,LevelSet)[id.classid]
    return t(id)

register_python_factory(factory)
