// -*- Mode: C++ -*-
//
//    xstring - An useful string class
//
//    Copyright (C) 2004/2005 Aldo Nicolas Bruno
//
//    Linux Users Group San Fidenzio
//

/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/ 

#ifndef __XSTRING_H__
#define __XSTRING_H__

#include <iostream>
#include <string>
#include <new>

using namespace std;

typedef char xchr;
#define  xstr xchr*

typedef int xpos;

// for safe dereferences. return "" if x is null
#define DREFXSTRING(x) ((x)?*(x):(xstring)"")

// for safe dereferences of string* vectors. return "" if x or x[i] is null
#define DREFXSTRINGI(x,i) ((x)?((x)[(i)]?DREFXSTRING((x)[(i)]):(xstring)""):(xstring)"")

class xstringvector;

class xstring
{
private:
  //data
  xstr s; 

  //allocated size
  xpos sz; 
  xpos g; //growth
  xchr nullchr; // always 0

  bool binary; // if we accept null characters in string
  xpos binarylen;
  
  //whether we have to free the buffer
  bool freebuff;

public:
  // default constructor
  xstring(); 

  // from char* ( strdup's str if dup=true)
  xstring(const xstr str, bool dup=true);

  // copy constructor
  xstring(const xstring& str);

  // allocate size
  xstring(xpos size);

  // fill with n characters
  xstring(xchr chr,xpos n);

  // assign n characters from str, set binary mode
  xstring(const xstr str,xpos n);
  
  
  // not implemeented yett!! ???????
  xstring(const xstring& str,xpos n);

  virtual ~xstring();

  // allocation stuff -- deletes the contents!!!!
  void alloc(xpos size);

  // preserves contents if not smaller
  void resize(xpos size);

  // free memory - deletes data!!!
  void free();
  
  // set growth 
  void growth(xpos n);

  // return allocated size
  xpos allocatedsize() const;
  
  // adjusts allocated size to the current length, if it doesn't overflow.
  void adjust();

  // strlen
  xpos size() const;
  xpos len() const;
  xpos length() const;

  // truncate string to the specified length
  void truncate(xpos len);

  // set length ( in binary mode )
  void setlength(xpos l);

  // whether is a binary string ie it contains '\0's
  bool isbinary() const;

  // set binary mode
  void setbinary(bool cond=true);

  // *** xchr stuff ***
  
  // get element at
  inline xchr& operator[](xpos index);

  inline const xchr& operator[](xpos index)const;

  xchr& at(xpos index);
  const xchr& at(xpos index) const;
 

  // get/set  chr
  xchr getat(xpos index) const;
  void setat(xpos index, xchr c);
  void setrange(xpos a,xpos b,xchr c);


  // fill from index a to index b with character c
  void fillrange(xpos a,xpos b, xchr c);
 
  // assign a string with n characters c
  void fill(xchr c, xpos n);

// *** xstr stuff ***

  // returns a new c string (free with xstring::xstrfree())
  xstr duplicate() const;

  // returns a std::string copy
  std::string string () const;

  // return the string buffer (char*) use with caution!
  xstr getbuffer();

  // get character string (const char*)
  // warning: the xstring object must exist for using this pointer!
  // otherwise use duplicate()
  const xstr gets() const;
  const xstr text() const;
  const xstr c_str() const;

  //xstr operator (xstr)();

  // delete character at pos `index'. Shift characters left
  void delat(xpos index);
  
  // delete characters in range [a,b). Shift characters left
  void delrange(xpos a, xpos b);
  
  // clear string data. set to ""
  void clear();

  // fill from index a to index b with string s
  void fillrange(xpos a, xpos b, const xstring& s);

  // fill with ns xstrings s
  void fill(const xstring& s, xpos ns);

  // fill with ns strings s
  void fill(const xstr s, xpos ns);

  // split characters in string contained in seps.
  // returns a vector with the splitted strings, without the separators
  xstringvector splitc(const xstring& seps) const;

  // split character sep.
  // returns a vector with the splitted strings, without the separators
  // if maxseps > 0, limits the split to the first maxseps tokens, the
  // remaining part of the string will be appended as the last element.
  // example:
  //  xstring a="a|b|c|d|e"
  //  xstringvector v = a.splitc('|',3);
  // v => ("a","b","c","d|e")
  xstringvector splitc(xchr sep, int maxseps=0) const;

  //  xstring** splits(const xstring**seps);

  // split string with sep.
  // returns a vector with the splitted strings, without the separators
  xstringvector splits(const xstring& sep) const;
 
  // split strings with seplist.
  // returns a vector with the splitted strings, without the separators
  // if maxseps > 0, limits the split to the first maxseps tokens, the
  // remaining part of the string will be appended as the last element.
  xstringvector splitss(const xstringvector &seplist, int maxseps=0) const;
  
  // mid
  // returns a substring from range (start,start+len)
  xstring mid (xpos start, xpos len=1) const;
  
  // returns a substring from range (0,len-1)
  xstring left (xpos len) const;

  // returns a substring from range (length()-len,length())
  xstring right (xpos len) const;

  void setrange(xpos a,xpos b,const xstr s);
  void setrange(xpos a,xpos b,const xstring& s);

  // comparisons
  int compare ( const xstring& s ) const;
  int compare ( const xstr s ) const;
  int comparenocase ( const xstring& s ) const;
  int comparenocase ( const xstr s ) const;

  /*
  int comparerange ( xpos a, xpos b, const xstring& s ) const;
  int comparerange ( xpos a, xpos b, const xstr s ) const;
  int comparerangenocase ( xpos a, xpos b, const xstring& s ) const;
  int comparerangenocase ( xpos a, xpos b, const xstr s ) const;
  */

  // operators

  xstring & operator= (const xstring& s);
  xstring & operator= (const xstr s);
  xstring & operator= (const std::string & s);

  bool operator== (const xstring& s) const;
  bool operator>= (const xstring& s) const;
  bool operator<= (const xstring& s) const;
  bool operator<  (const xstring& s) const;
  bool operator>  (const xstring& s) const;
  bool operator!= (const xstring& s) const;
  
  xstring operator+ (const xstring& t) const;
 
  xstring operator+ (const int t) const;
  xstring operator+ (const long t) const;
  xstring operator+ (const float t) const;
  xstring operator+ (const double t) const;

  // xstring operator- (const xstring& t) const;
  
  xstring& operator+= (const xstring& s);

  //  xstring& operator-= (const xstring& s);

  bool operator== (const xstr s) const;
  bool operator>= (const xstr s) const;
  bool operator<= (const xstr s) const;
  bool operator<  (const xstr s) const;
  bool operator>  (const xstr s) const;
  bool operator!= (const xstr s) const;
  
  xstring operator+ ( const xstr t) const;
  xstring operator- ( const xstr t) const;
  xstring& operator+= (const xstr s);
  xstring& operator-= (const xstr s);

  xstring& operator<< (const xstring& s);
  xstring& operator<< (const xstr s);
  xstring& operator<< (xchr s);
  xstring& operator<< (long n);
  xstring& operator<< (float f);
  xstring& operator<< (double d);

  // empty
  bool operator! () const;
  bool isempty() const;
  bool notempty()const;

  // security:
  bool overflows() const;

  // conversions
  void toupper();
  void tolower();

  void trim();
  void trimr();
  void triml();
  
  void escape_cstyle();
  //  void escape_sstyle();
  void escape_sql();

// conversions copy
  xstring gettoupper() const;
  xstring gettolower() const;

  xstring gettrim() const;
  xstring gettrimr() const;
  xstring gettriml() const;
  
  xstring getescape_cstyle() const;
  
  int getint() const;
  long getlong() const;
  float getfloat() const;
  double getdouble() const;

  
  void frombinary(const xstr, xpos l);
  
  // searching
  xpos search (const xstring& s, int occurrence=0, bool overlap=true) const;
  xpos* searchall (const xstring& s,bool overlap=true) const;
  xpos search (const xstr s, int occurrence=0,bool overlap=true) const;
  xpos* searchall (const xstr s,bool overlap=true) const;

  xpos search (xchr c, int occurrence=0) const;
  xpos* searchall (xchr c) const;

  xpos occurrences(const xstring& s, bool overlap=true) const;
  xpos occurrences(const xstr s, bool overlap=true) const;
  xpos occurrences( xchr c) const;

  // replace
  int replace(const xstring& s,const xstring& r, int n=0);
  int replace(xchr c, xchr r, int n=0);
  int replace(const xstring& s,xchr r, int n=0);
  int replace(xchr c, const xstring& r, int n=0);

  int replace(const xstr s,const xstr r, int n=0);
  int replace(const xstr s,xchr r, int n=0);
  int replace(xchr c, const xstr r, int n=0);

  // delete
  xpos delstr (const xstring& s);
  xpos delchr (xchr c);

  // format!!
  xpos printf(const xstr format, ...);
  xpos format(const xstr format, ...);
  static xstring sprintf(const xstr format, ...);
  xpos appendformat(const xstr format, ...);

  // statics
  static xpos xstrlen(const xstr t, xpos allocated=-1);
  static xpos xstrcpy( xstr d,const xstr s);
  static xpos xstrncpy( xstr d,const xstr s, xpos n);
  static xpos xstrcat( xstr str, const xstr cat);
  static xpos xstrncat( xstr str, const xstr cat,xpos n);
  static int xstrncmp(const xstr a, const xstr b, xpos n);
  static int xstrcmp(const xstr a, const xstr b);

  static xstr xstrnew(xpos l);
  static void xstrfree(xstr t);
  static void freevect( xstring* v);
  static void freevect( xstring** v);

  static xstring fromint (int n);
  static xstring fromlong (long l);
  static xstring fromfloat (float f);
  static xstring fromdouble (double d);
  static xstring fromxstr (const xstr s);
  static xstring fromxchr (xchr c);

  friend std::ostream& operator << (std::ostream& o, const xstring& s);
};

std::ostream& operator << (std::ostream& o, const xstring& s);

inline const xchr& xstring::operator[](xpos index) const
{
  if (sz>0 && s && index>=0 && index <sz/*&& index<length()*/)
    return s[index];
  std::cerr<< "Error! xstring::operator[] index out of bounds!!" << std::endl;
  abort(); // FIXME: is this necessary???
}

inline xchr& xstring::operator[](xpos index)
{
  if (sz>0 && s && index>=0 && index <sz/*&& index<length()*/)
    return s[index];

  std::cerr<< "Error! xstring::operator[] index out of bounds!!" << std::endl;
  abort();// FIXME: is this necessary???
}

xstring operator+ (const char *s, const xstring& t);

//xstring& operator+ (xstring& a, const char* b);

#endif // __XSTRING_H__
