// -*- Mode: C++ -*-
//
//    ObjectList - a list of objects
//
//    Copyright (C) 2005 Aldo Nicolas Bruno
//
//    Linux Users Group San Fidenzio
//

/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/ 

#include "objectlist.h"

ObjectList::Item::Item( Object* value)
  : my_value(0)
{
  set(value);
  my_next=my_prev=NULL;
}

ObjectList::Item::Item( const ObjectList::Item& copy)
{
  set(copy.value());
  my_next=my_prev=NULL;
}

ObjectList::Item::~Item()
{
  if (my_value)
    my_value->dec();
}

ObjectList::Item* ObjectList::Item::next()
{
  return my_next;
}

const ObjectList::Item* ObjectList::Item::next() const
{
  return my_next;
}
/*
ObjectList::Item* ObjectList::Item::prev()
{
  return my_prev;
}

const ObjectList::Item* ObjectList::Item::prev() const
{
  return my_prev;
}
*/
Object* ObjectList::Item::value () const
{
  return my_value;
}

void ObjectList::Item::set(Object* value)
{
  value->inc();
  my_value=value;
}

void ObjectList::Item::setnext(ObjectList::Item* i)
{
  my_next=i;
}

ObjectList::ObjectList ()
{
  my_head=0;
}

ObjectList::ObjectList( Object* values,size_t n)
{
  my_head=0;
  fromvector(values,n);
}

ObjectList::~ObjectList()
{
  clear();
}


void ObjectList::fromvector(Object* values,size_t n)
{
  clear();
  Object* p=values;
  size_t i=0;
  for(;i<n;++i)
    {
      additem(&p[i]);
    }
}

ObjectList::Item* ObjectList::head()
{
  return my_head;
}

const ObjectList::Item* ObjectList::head() const
{
  return my_head;
}

ObjectList::Item* ObjectList::tail()
{
  Item *p = head();
  while( p && p->next() )
    {
      p=p->next();
    }
  return p;
}

const ObjectList::Item* ObjectList::tail() const
{
  const Item *p = head();
  while( p && p->next() )
    {
      p=p->next();
    }
  return p;
}


ObjectList::Item* ObjectList::getitem(size_t index) 
{
  Item *p=head();
  size_t i=0;
  while(p)
    {
      if(i==index)
	return p;

      ++i;
      p=p->next();
    }
  return NULL;
}

const ObjectList::Item* ObjectList::getitem(size_t index) const
{
  const Item *p=head();
  size_t i=0;
  while(p)
    {
      if(i==index)
	return p;

      ++i;
      p=p->next();
    }
  return NULL;
}


Object* ObjectList::getvalue(size_t index) const
{
  const Item* i = getitem(index);
  
  if(i)
    return i->value();

  return 0; //NULL
}

size_t ObjectList::getsize() const
{
  const Item *p=head();
  size_t i=0;
  
  while(p)
    {
      ++i;
      p=p->next();
    }
  
  return i;
}


void ObjectList::additem(ObjectList::Item* o)
{
  if(my_head)
    o->setnext(my_head);
  else
    o->setnext(NULL);
  
  my_head=o;
}

bool ObjectList::hasitem (ObjectList::Item * i) const
{
  if(!i) return false;
  const Item *p=head();
  while(p)
    {
      if(p==i)
	return true;
      p=p->next();
    }
  return false;  
}

bool ObjectList::hasitem (size_t index) const
{
  return index < getsize();
}

void ObjectList::additem( Object* value)
{
  additem( new Item(value) );
}
 
void ObjectList::removeitem (size_t index)
{
  Item * p = getitem(index);
  removeitem(p);
}

void ObjectList::removeitem (ObjectList::Item * i)
{
  bool found = hasitem(i);

  if (!found) return;

  // it's the first element??
  if( i == my_head)
    {
      Item * t = my_head->next();
      delete my_head;
      my_head=t;
    }
  else 
    {
      // find the previous element
      Item * p = my_head;
      while (p)
	{
	  if ( p->next() == i )
	    {
	      //cool
	      break;
	    }
	  p=p->next();
	}
      // p now is the i->prev() element
      // setup all...
      p->setnext(i->next());
      delete i;
    }
  // Done
  
}

void ObjectList::clear()
{
  Item *i=head();
  Item *p=NULL;
  while(i)
    {
      p=i->next();
      delete i;
      i=p;
    }
  my_head=NULL;
  // Done
}

