// -*- Mode: C++ -*-
// Copyright (C) 2005 Aldo Nicolas Bruno

/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <netdb.h>
#include <memory.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/poll.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netinet/ip.h>
#include <errno.h>
#include <asm/errno.h>

#include "common.h"
#include "inetsocket.h"

INETSocket::INETSocket()
  :Object("INETSocket"),FD(-1),Socket(-1),h(0),hn(NULL),port(0)
{
  // create the socket
  fd = socket ( AF_INET, SOCK_STREAM,0);//SOCK_RAW,  IPPROTO_TCP);
  if (fd < 0)
    {
      checkval =fd;
      perror("socket");
      return;
    }

  int f=1;
  setsockopt(fd,0,SO_REUSEADDR,&f,sizeof(f));
  f=1;
  
  /* bind any port number */
  memset (&local,0,sizeof(local));
  local.sin_family = AF_INET;
  local.sin_addr.s_addr = htonl(INADDR_ANY);
  local.sin_port = htons(0);

  int r = ::bind(fd, (struct sockaddr *) &local, sizeof(local));
  if ( r < 0)
    {
      cout  <<  "Error in bind to port " << (u_short)port << endl;
      perror("bind");
      checkval=r;
    }
}

INETSocket::INETSocket(int  filedescriptor, struct sockaddr_in local, struct sockaddr_in remote)
  :Object("INETSocket"),FD(filedescriptor),Socket(filedescriptor),hn(NULL)
{
  this->local=local;
  this->remote=remote;

  /*  long fflags=fcntl (fd, F_GETFL,0);
  fflags|=O_NONBLOCK;
  fcntl(fd,F_SETFL,fflags);
  */
}

INETSocket::~INETSocket()
{
  if(hn) delete [] hn;  
}

int INETSocket::bind(const char* host, int port)
{
  if(host && host[0])
    {
      h=gethostbyname(host);

      if ( ! h ) 
	{
	  perror ("gethostbyname");
	  close();
	  return -1;
	}
    
      if(hn) delete []hn;
      hn=new char[strlen(host)+2];
      strcpy(hn,host);
    }
  else
    {
      if(hn) delete []hn;
      hn=NULL;
    }

  this->port=port;

  memset (&local,0,sizeof(local));
  
  local.sin_family = AF_INET; //h->h_addrtype;
  
  if(hn)
    {
      memcpy((char *) &local.sin_addr.s_addr, h->h_addr_list[0], h->h_length);
    }
  else
    {
      local.sin_addr.s_addr = htonl(INADDR_ANY); 
    }
  
  local.sin_port = htons(port);
 
  if (fd == -1)
    {
      checkval=-1;
      fprintf (stdout,"serversocket::bind(): bad file descriptor\n");
      return -1;
    } 
  
  int n=0; 
  const int maxtimes=16;
  int r=-1;
  while (r<0&&n<=maxtimes)
    {
      cout << "binding..." <<endl;
      r=::bind(fd, (struct sockaddr *) &local, sizeof(local));
      if ( r < 0)
	{
	  if(errno==EINTR) continue; // ignore
	  else if(errno==EADDRINUSE)
	    {
	      n=n+1;
	      cout  <<"Address already in use (" << n << "/" << maxtimes << ") .. " << endl;
	      sleep(1); // sleep 1 secon
	      
	      checkval=r;
	      continue; // return r;
	    }
	  else
	    perror("bind");
	  return -1;
	}
    }
  return  r;  
}



int INETSocket::connect(const char* host_, int port_)
{
  if (check_integrity()<0) return -1;

  h=gethostbyname(host_);
  if ( ! h ) 
    {
      perror ("gethostbyname");
      close();
      return -1;
    }
  if (hn) delete []hn;
  hn =new char[strlen(host_)+2];
  strcpy (hn,host_);
  this->port=port;

  char buff[256];
  getip(buff,*((struct in_addr*)h->h_addr));
  cout << "remote address: " << buff << endl;

  remote.sin_family = h->h_addrtype;

  //memcpy((char *) &remote.sin_addr.s_addr, h->h_addr_list[0], h->h_length);
  //remote.sin_addr.s_addr = inet_addr(h->h_addr);

  inet_aton(buff,&remote.sin_addr);

  remote.sin_port = htons(port_);

  cout << "Connecting  ... ";
  if (::connect (fd,(sockaddr*) &remote, sizeof(remote)) == -1)
    {
      perror("connect");
      return checkval=-1;
    }
  cout <<  " ok\n";
  return checkval=0;
}


void INETSocket::getip (char *buf, struct in_addr addr)
{
  if (!buf) return;
  unsigned int i = htonl(addr.s_addr); 
  sprintf(buf,"%d.%d.%d.%d",(i>>24)&0xff,(i>>16)&0xff,(i>>8)&0xff,i&0xff);
  return;
}

/*
int INETSocket::sendall(char*s, int n)
{
  if (check_integrity()<0) return -1;
  int i;
  if(n<0) n=strlen(s);
  i=::send(fd,s,n,0);
  return i;
}
*/


struct sockaddr_in* INETSocket::getlocaladdr()
{
  return &local;
}

struct sockaddr_in* INETSocket::getremoteaddr()
{
  return &remote;
}
