// -*- Mode: C++ -*-
// Copyright (C) 2005 Aldo Nicolas Bruno

/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <time.h>
#include <regex.h>
#include <stdlib.h>

#include <xstring.h>

#include "httptime.h"

const char* wkdays [] = {"Mon","Tue","Wed","Thu","Fri","Sat","Sun", 0};
const char* weekdays [] = {"Monday", "Tuesday", "Wednesday", "Thursday","Friday","Saturday","Sunday",0 };
const char* months [] = {"Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec",0 };


HTTPTime::HTTPTime (time_t time)
  : Object( "HTTPTime")
{
  t=time;
}

HTTPTime::HTTPTime (const xstring &timestring)
{
  fromstring(timestring);
}

void HTTPTime::fromstring (const xstring& timestring)
{
  if (!from_rfc1123(timestring))
    if (!from_rfc1036(timestring))
      if (!from_asctime(timestring))
	t=-1;
}

time_t my_timegm (struct tm *tm) {
  time_t ret;
  char *tz;

  tz = getenv("TZ");
  setenv("TZ", "", 1);
  tzset();
  ret = mktime(tm);
  if (tz)
    setenv("TZ", tz, 1);
  else
    unsetenv("TZ");
  tzset();
  return ret;
}

bool HTTPTime::from_rfc1123(const xstring& timestring)
{
  regex_t r;
  int p;

  p=regcomp(&r,	       
         "^(Mon|Tue|Wed|Thu|Fri|Sat|Sun)[,] [0-9]{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) [0-9]{4} [0-9]{2}:[0-9]{2}:[0-9]{2} GMT$"

	    ,REG_EXTENDED);
  if (p) 
    {
      printf("HTTPTime: Error while compiling regex for rfc1123\n");
      t=-1;
      return false;
    }
  if(regexec(&r,timestring.text(),0,NULL,0)==0)
    {
      regfree(&r);
      struct tm tv;
      int i,day,year,hour,min,sec,imonth=0,iwkday=0;
      char month[50],wkday[50];
      month[0]=wkday[0]=0;
      sscanf(timestring.text(),
	     "%s %2d %s %4d %2d:%2d:%2d GMT",
	     &wkday, &day, &month, &year, &hour, &min, &sec);
     
      for(i=0; months[i]; ++i)
	if (! strcmp(months[i],month) )
	  {
	    imonth=i;
	    break;
	  }
      memset (&tv,0,sizeof(struct tm));
      tv.tm_sec=sec;
      tv.tm_min=min;
      tv.tm_hour=hour;
      tv.tm_mday=day;
      tv.tm_mon=imonth;
      tv.tm_year=year-1900;
      tv.tm_zone="GMT";
      t=my_timegm(&tv);

      if (t<0)
	{
	  perror("timegm failed");
	 
	  return false;
	}
      
      return true;
    }
  regfree(&r);
  return false;
}

bool HTTPTime::from_rfc1036(const xstring& timestring)
{
  regex_t r;
  int p;

  p=regcomp(&r,
	    "^(Monday|Tuesday|Wednesday|Thursday|Friday|Saturday|Sunday)[,] "
	    "[0-9]{2}[-]"
	    "(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)[-]"
	    "[0-9]{2} [0-9]{2}:[0-9]{2}:[0-9]{2} GMT$"
	    ,REG_EXTENDED);
  if (p) 
    {
      printf("HTTPTime: Error while compiling regex for rfc1036\n");
      t=0;
      return false;
    }
  if(regexec(&r,timestring.text(),0,NULL,0)==0)
    {
      regfree(&r);
      struct tm tv;
      int i,day,year,hour,min,sec,imonth=0,iwkday=0;
      char month[50],wkday[50];
      month[0]=wkday[0]=0;
      // Sunday, 06-Nov-94 08:49:37 GMT ;
      sscanf(timestring.text(),
	     "%s %d%*c%3s%*c%2d %2d:%2d:%2d GMT",
	     &wkday, &day, &month, &year, &hour, &min, &sec);
     
      for(i=0; months[i]; ++i)
	if (! strcmp(months[i],month) )
	  {
	    imonth=i;
	    break;
	  }
      memset (&tv,0,sizeof(struct tm));
      tv.tm_sec=sec;
      tv.tm_min=min;
      tv.tm_hour=hour;
      tv.tm_mday=day;
      tv.tm_mon=imonth;
      tv.tm_year=(year>=70)?year:(year+100);
      tv.tm_zone="GMT";
      t=my_timegm(&tv);

      if (t<0)
	{
	  perror("timegm failed");
	  return false;
	}
      return true;
    }
  regfree (&r);
  return false;
}


bool HTTPTime::from_asctime(const xstring& timestring)
{
  regex_t r;
  int p;

  p=regcomp(&r,
	    "^(Mon|Tue|Wed|Thu|Fri|Sat|Sun) "
	    "(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) *"
	    "[0-9]{1,2} "
	    "[0-9]{2}:[0-9]{2}:[0-9]{2} [0-9]{4}"
	    ,REG_EXTENDED);
  if (p) 
    {
      printf("HTTPTime: Error while compiling regex for asctime format\n");
      t=0;
      return false;
    }
  if(regexec(&r,timestring.text(),0,NULL,0)==0)
    {
      regfree(&r);
      struct tm tv;
      int i,day,year,hour,min,sec,imonth=0,iwkday=0;
      char month[50],wkday[50];
      
      month[0]=wkday[0]=0;
  
      sscanf(timestring.text(),
	     "%s %s %d %d:%d:%d %d",
	     &wkday, &month, &day, &hour, &min, &sec, &year);
      
      
      for(i=0; months[i]; ++i)
	if (! strcmp(months[i],month) )
	  {
	    imonth=i;
	    break;
	  }
      memset (&tv,0,sizeof(struct tm));
      tv.tm_sec=sec;
      tv.tm_min=min;
      tv.tm_hour=hour;
      tv.tm_mday=day;
      tv.tm_mon=imonth;
      tv.tm_year=year-1900;
      tv.tm_zone="GMT";
      t=my_timegm(&tv);

      if (t<0)
	{
	  perror("timegm failed");
	  return false;
	}
      return true;
    }
  regfree (&r);
  return false;
}

xstring HTTPTime::get_rfc1123() const
{
  xstring x;
  struct tm gmt;
  if (!gmtime_r(&t,&gmt)) 
    {
      perror("gmtime():");
      return ("Wed, 31 Dec 1969 23:59:59 GMT");
    }

  // Sun, 06 Nov 1994 08:49:37 GMT
  x.format("%s, %.2d %s %.2d %.2d:%.2d:%.2d GMT",
	   wkdays[gmt.tm_wday==0?6:gmt.tm_wday-1], // TODO: check this!!!!!! buggy libc?
	   gmt.tm_mday,
	   months[gmt.tm_mon],
	   1900+gmt.tm_year,
	   gmt.tm_hour,
	   gmt.tm_min,
	   gmt.tm_sec
	   );
  return x;
}

// the same as rfc1123 with the variation that the date elements are separated with dashes instead of spaces
xstring HTTPTime::get_cookietime() const
{
  xstring x;
  struct tm gmt;
  if (!gmtime_r(&t,&gmt)) 
    {
      perror("gmtime():");
      return ("Wed, 31-Dec-1969 23:59:59 GMT");
    }

  // Sun, 06 Nov 1994 08:49:37 GMT
  x.format("%s, %.2d-%s-%.2d %.2d:%.2d:%.2d GMT",
	   wkdays[gmt.tm_wday==0?6:gmt.tm_wday-1], // TODO: check this!!!!!! buggy libc?
	   gmt.tm_mday,
	   months[gmt.tm_mon],
	   1900+gmt.tm_year,
	   gmt.tm_hour,
	   gmt.tm_min,
	   gmt.tm_sec
	   );
  return x;
}


time_t HTTPTime::gettime() const
{
  return t;
}
