// -*- Mode: C++ -*-
// Copyright (C) 2004-2005 Aldo Nicolas Bruno

/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <sys/types.h>
#include <sys/wait.h>

#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <unistd.h>
#include <limits.h>
#include <sys/mman.h>


#include "httpserver.h"
#include "httpconnection.h"
#include "httpservertoken.h"
#include "httprequestheader.h"
#include "httpgetparamlist.h"
#include "httpresponseheader.h"
#include "httprequest.h"
#include "httpresponse.h"
#include "syntax.h"
//#include "dbmodule.h"


#include "inputconn.h"
#include "worker.h"
#include "schedendpointlist.h"
#include "httpinternalproxy.h"



//#include "connectionscheduler.h"

#ifndef MIN
# define MIN(a,b)((a)>(b)?(b):(a))
#endif


#ifndef MAX
# define MAX(a,b) ((a)>(b)?(a):(b))
#endif


HTTPServer::HTTPServer()
  :Object("HTTPServer")
{
  server=new ServerSocket();
  
  firstres=lastres=0; // NULL
  docroot=".";
  uri="";
  port=0;
  hostname="";  
  iproxy=NULL;
}

HTTPServer::~HTTPServer()
{
  if(server)
    {
      server->shutdown();
      server->close();
      delete server;
    }
}

ServerSocket * HTTPServer::getsocket()
{
  return server;
}

int HTTPServer::bind(const xstring & host_, int port_)
{
  int r =server->bind(host_.text(),port_);

  if(r==0)
    {
      sockaddr_in *addr=server->getlocaladdr();
      
      struct hostent* h;
      if(addr->sin_addr.s_addr != INADDR_ANY)
	h=gethostbyaddr(addr,sizeof *addr,AF_INET);
      else
	h=NULL;
      
      hostname=host_;
      port=port_;
      
      if ( ! h ) 
	{
	  //perror ("gethostbyaddr");
	  ip="";
	}
      else
	{
	  char buff[256];
	  INETSocket::getip(buff,*((struct in_addr*)h->h_addr));
	  ip=buff;
	  uri = xstring("http://") +  buff + ":" + xstring::fromint(port_);
	}
    }
  return r;
}

int HTTPServer::listen()
{
  server->listen(5);
  return 0;
}

Socket* HTTPServer::acceptone()
{ 
  Socket* s=0;
 
  
  long r=0;
 
  fd_set w;
  struct timeval tv;
  FD_ZERO(&w);
  
  while (!s)
    {
      //tv.tv_sec = 5;
      //tv.tv_usec = 0;
      FD_SET(server->getfd(),&w);
      r=select(server->getfd()+1,&w,NULL,NULL,NULL);//&tv);
      
      if(r<=0)
	{
      	  perror("HTTPServer::acceptone(): select()");
	  return 0; //NULL
	}
      else if(r)
	{
	  s = server->accept();
		  
	  if(!s && errno==EAGAIN)
	    continue;
	  break;
	}
    } //while !r
    
  return s;
}

int HTTPServer::processrequestline(HTTPConnection* c,const xstring& startline)
{
  MYTRACE("processrequestline()");
  int keepalive=1;
  HTTPRequest req (c,this);
  HTTPRequestHeader rh(c);
  HTTPResponse resp(c,this);
  
  long ris=rh.readandprocessheader();
  if(ris>=0)
    {
      xstring hosth =rh.getitem("Host");
      if(!hosth.isempty())
	uri = "http://" + hosth;
      
      if(hosth.isempty())
	{
	  resp.fromerror (400);
	  
	  return 0;
	}
      
      xstring pconn=rh.getitem("Connection");
      if((pconn.comparenocase("Keep-Alive")!=0) 
	 &&
	 (pconn!=""))
	keepalive=0;
      
      req.process(startline,&rh);
      
      if (req.getversion()=="HTTP/1.0"
	  &&
	  pconn.comparenocase("Keep-Alive")!=0)
	keepalive=0;
      
      // oh.. if its a post, read also the POST data :-) !!!!
      req.readpostdata();
	  
#ifdef DEBUG
      fprintf(stdout,"Processing response from request..\n");
#endif

      long r=resp.fromrequest(&req);

#ifdef DEBUG
      fprintf(stdout,"..done\n");
#endif

      if (r<0) keepalive=0;
    }
  else
    {
      if (c->timeout())
	; //  resp.fromerror(408); // timed out
      else
	resp.fromerror(400,"Syntax error in headers");
      keepalive=0;
    }	  
  return keepalive;
}

int HTTPServer::handleone(Socket* x)
{  	  
  HTTPConnection c(x);
  HTTPServerToken t(&c);
 
      //  usleep(100000); // delay between response and next request
  
  fdatasync(x->getfd()); // sync!! this is really needed
  
  fprintf (stdout,"Waiting for request...\n");

#ifdef DEBUG
  perror("in handleone()");
#endif

  //  c.clear(); // clear read buffer
  int rka=0;
  xstring line ;
  HTTPRequestHeader rh(&c);

  int l = rh.readstartline(line,HTTPRequestHeader::CHECK_REQUESTLINE,true);

  if (l < 0)
    {
      return -1;
    }
  else if (l>0)
    {
      
    }
  else 
    {
      return -1;
    }
  
  rka= processrequestline(&c,line);
  if (rka)
    return 1;
  else
    return 0;

  return 1;
  

}



int HTTPServer::handle(int n, long secs)
{ 

  iproxy = new HTTPInternalProxy(this,n);

  return iproxy->run();

}


Worker* HTTPServer::createworker()
{
  MYTRACE("HTTPServer::createworker()");

  //if (killed) return NULL;
  
  int pair[2];
  // int pair2[2];
  
  int r = socketpair(PF_UNIX,SOCK_STREAM,0,pair);
  if (fcntl (pair[0],F_SETFL,O_NONBLOCK) <0)
    perror ("fcntl O_NONBLOCK");
  if (fcntl (pair[1],F_SETFL,O_NONBLOCK) <0)
    perror ("fcntl O_NONBLOCK");
  
  pid_t pid= fork();
  if (pid< 0)
    {
      perror ("HTTPServer::handle: Error in fork()");
    }
  else if(pid == 0)
    {
      
      /// IN CHILD PROCESS
#ifdef DEBUG
      cout << "in child!" << endl;
#endif
      signal (SIGPIPE, SIG_DFL);
      signal (SIGINT,SIG_DFL);
      signal (SIGCHLD,SIG_DFL);

      close (pair[0]);
      
      // TODO: close all the other sockets!!!... 
      //	  workers.closeall()
      
      if(server) server->close();
      
      int fd =pair[1];
      Socket a(fd);	
      
      fd_set set;
      
      int x = oninit(); // virtual
      
      int r=1;
      
      while ( r >0)
	{ 
	  FD_ZERO(&set);
	  FD_SET(fd,&set);
	  
	  r = select (fd+1,&set,NULL,NULL,NULL);
	  if (r<=0) break;
	  
	  
	  int n=NHBUFFERLEN;
	  // xstring s = a.recv(n);
	  // cout << "Recv: " << s << endl;
	  int i = handleone(&a);
	  if (i<0)
	    {
	      MYTRACE("Child: caught negative rval in handleone()\n");
	      
	      break;
	    }

	}
#ifdef DEBUG
      perror ("Child finishing");
#endif
      exit(0);
      
    }
  close (pair[1]);
  
  Worker *w = new Worker(pair[0],pid,iproxy);

  //      w->send("ciao...\n");
  return w;
}


int HTTPServer::oninit()
{
  return 0;
}

//### FIXME!!! FREE DATA WHEN DESTROYED
int HTTPServer::addres(const xstring& name, HTTPResource *r)
{
  httpresourcemap *rx=new httpresourcemap;

  rx->name=name;
  rx->res=r;
  rx->next=0; // NULL
  
  if (!firstres) 
    lastres=firstres=rx;
  else
    {
      lastres->next=rx;
      lastres=rx;
    }
  return 0;
}

void HTTPServer::setdocroot(const xstring& root)
{
  docroot=root;
#ifdef DEBUG
  fprintf (stdout,"Document root: %s\n", root.text());
#endif
}

xstring HTTPServer::getdocroot() const
{
  return docroot;
}

xstring HTTPServer::gethostname() const
{
  return hostname;
}

xstring HTTPServer::geturi() const
{
  return uri;
}

int HTTPServer::getport() const
{
  return port;
}

xstring HTTPServer::getip() const
{
  return ip;
}
