// -*- Mode: C++ -*-
// Copyright (C) 2004 Aldo Nicolas Bruno

/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/


#include <xstring.h>

#include "httpserver.h"
#include "httpconnection.h"
#include "httpservertoken.h"
#include "httpheader.h"

#include "httprequestheader.h"
#include "syntax.h"


HTTPRequestHeader::HTTPRequestHeader(HTTPConnection *c )
  : Object("HTTPRequestHeader"),HTTPHeader(c), status(Idle),endoftoken(4)
{
  
  regex_stat=regcomp(&headerregex,"^" regextoken ":" field_content "$",0);

  if (regex_stat) 
    {
      std::cerr <<  "HTTPRequestHeader(): error in regcomp()\n";
    }
  int r = regcomp(&requestlineregex,"^[A-Za-z0-9]+ .+ HTTP/[0-9]+[.][0-9]+$",REG_EXTENDED);
  if (r) 
    {
      std::cerr <<  "HTTPRequestHeader(): error in regcomp(requestline)\n";
    }
  r = regcomp(&statuslineregex,"^HTTP/[0-9]+[.][0-9]+ [1-5][0-9]{2} .+$",REG_EXTENDED);
  if (r) 
    {
      std::cerr <<  "HTTPRequestHeader(): error in regcomp(statusline)\n";
    }

  endoftoken.append(CRLF);
  endoftoken.append(LF);
  endoftoken.append(CR);
  last=NULL;
}

HTTPRequestHeader::~HTTPRequestHeader()
{
  if (regex_stat == 0)
    regfree (&headerregex);
  regfree (&requestlineregex);
  regfree (&statuslineregex);
 
}

int HTTPRequestHeader::readstartline(xstring & startline, int check, bool blocking)
{
  int i = readline(startline,blocking);
  if (i<0)
    return i;

#ifdef DEBUG
  fprintf (stdout,"HTTPRequestHeader: start-line: %s\n",startline.text());
#endif
  if (startline.isempty())
    return -1;

  switch (check)
    {
    case CHECK_REQUESTLINE:
      if (regexec(&requestlineregex,startline.text(),0,NULL,0)==0)
	{
#ifdef DEBUG
	  fprintf (stdout, "Valid request line\n");
#endif
	  return 1;
	}
      break;
    case CHECK_STATUSLINE:
      if (regexec(&statuslineregex,startline.text(),0,NULL,0)==0)
	{
#ifdef DEBUG
	  fprintf (stdout, "Valid status line\n");
#endif
	  return 1;
	}
      break;
    default:
      return -1;
    }

  return -1;
}

bool HTTPRequestHeader::validateline(const xstring& s)
{
  //return true;
  bool x=false;

  if (regex_stat) 
    {
      std::cerr <<  "HTTPRequestHeader::validateline(): error in regcomp()\n";
      return false;
    }
  
  if (regexec(&headerregex,s.text(),0,NULL,0)==0)
    {
#ifdef DEBUG
      fprintf (stdout, "HTTPRequestHeader: Valid header: %s\n",s.text());
#endif 
      x=true;
    }
  else
    {
#ifdef DEBUG
  fprintf (stdout, "HTTPRequestHeader: Invalid header: %s\n",s.text());
#endif
    }
  return x;
}

int HTTPRequestHeader::readline(xstring & read,bool blocking)
{
  xstringvector linetoken(3);
  linetoken.append(CRLF);
  linetoken.append(CR);
  linetoken.append(LF);

  HTTPServerToken t(connection);
  xstring token;
  xstring tmp;
  int r =t.gettoken(linetoken,tmp,
			 blocking ? HTTPServerToken::Blocking
			 : HTTPServerToken::NonBlocking,
			 &token);
  read=tmp;
  return r;
}

int HTTPRequestHeader::sched(bool blocking)
{ 
  //  clear();
  xstring tmp;
 
  long y=0;
  while (1) // TODO: Limit connections!! 
    {
      int r=readline(tmp,blocking);
      //  printf ("tmp: %sx\n",tmp.text());
      //  printf ("token: %sx\n",token.text());
      if (connection->timeout()||r<0) 
	return status=Error;

      if (r>0)
	{
	  
	      //fprintf (stdout, "HTTPRequestHeader:\n");
	  status=NotComplete;
	  return status;
	}
      else if (tmp.isempty())
	{
	  if (last) goto stage2;
	  else goto lerror;
	}
      // Blank space
      else if (tmp[0]=='\t' || tmp[0]==0x20)
	{
	  if (!last) goto lerror;
	  tmp.triml();
	  last->set(last->value()+tmp);
	  if(!validateline(last->value()))
	    goto lerror; 
	}
      else
	{
	  if (!validateline(tmp))
	    goto lerror;
	  
	  last = new xstringlistitem(tmp);
	  hl.additem(last);
	}
      y++;
    }

 lerror:
  if (connection->timeout())
    cout <<  "HTTPRequestHeader: Timed out\n";
  else
    cout <<  "HTTPRequestHeader: Syntax error\n";
 
  return Error;

 stage2: 
  //  fprintf( stdout, "headers: %s\n", headers.text());  
  
  //  fprintf (stdout,"headers, stage2\n" );
  
  xstringlistitem * q=hl.head();
  
  for ( ; q; q=q->next())
    {
      xstringvector z = q->value().splitc(':',1);
      if(z.n()>0)
      	{
	  
	  if(z.n()>1)
	    {
	      z[1].triml();
	      setitem(z[0],z[1]);
	    }
	  else
	    {
	      setitem(z[0],"");
	    }
	
#ifdef DEBUG
	  cout << "Header: name: " << z[0] 
	       <<        " value: " << ((z.n()>1)?z[1]:"") 
	       << endl;    
#endif

	}
    }    
  hl.clear();
  last=NULL;
  return Complete;

}

int HTTPRequestHeader::readandprocessheader ()
{
  clear();
  if (sched ( /*blocking = */ true )== Complete)
      return 0;
  return -1;
}
