// -*- Mode: C++ -*-
// Copyright (C) 2005 Aldo Nicolas Bruno

/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/


#include <sys/select.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>

#include <xstring.h>

#include "server_socket.h"

#include "httpserver.h"
#include "httpconnection.h"

#include "httpheader.h"
#include "httpresponseheader.h"
#include "httprequestheader.h"
#include "httpgetparamlist.h"
#include "httprequest.h"
#include "httpresponse.h"
#include "httpresource.h"


HTTPConnection::HTTPConnection(Socket *sokt)
  :Object("HTTPConnection")
{
  socket = sokt; buff=""; pos=0;
  timeout_usecs=(30000000);  retry_usecs=(500000); timeoutct =(time(NULL));
  int t=1;
  pos=0;
  // setsockopt(socket->getfd(),SOL_IP,IP_RECVERR,&t,sizeof(int));
  long fflags=fcntl (socket->getfd(), F_GETFL,0);
  fflags|=O_NONBLOCK;
  fcntl(socket->getfd(),F_SETFL,fflags);

}

int HTTPConnection::handle()
{
  return -1;
}

void HTTPConnection::clear()
{
 if(pos!=0)
    {
      int r=pos;
      buff=buff.right(buff.length()-pos);
      pos=0;
#ifdef DEBUG
      fprintf (stdout,"buffer: '%s', pos=%d, was:%d\n",buff.text(),pos,r);
#endif
    }
}

int HTTPConnection::disconnect()
{// TODO: should send Connection: Close???
  int r=socket->shutdown();
  r|=socket->close();
  return r != 0;
}

// non-blocking read
long HTTPConnection::read(int n)
{
  // buffer size TODO: optimize?
  char *s = socket->recv(n,MSG_WAITALL);
  if (n>0) timeoutct=time(NULL);
  else if(n==0)
    {
      delete []s;
      timeoutct=0;
#ifdef DEBUG
  std::cout << "read(): return "<< n << endl;
#endif

      return 0;
    }
  else 
    {
      delete []s;
      if(errno==EAGAIN);

      else
	timeoutct=0;
#ifdef DEBUG
      
      perror("HTTPConnection::read(): ");
#endif
	  
      return n;
    }

  xstring q(s,n);
  buff+=q;
  delete [] s;
  s=NULL;

#ifdef DEBUG
  std::cout << "read(): return "<< n << endl;
#endif
  return n;
}

const xstring& HTTPConnection::buffer() const
{
  return buff;
}

xstring HTTPConnection::buffernew() const
{ 
  if (pos==0)
    return buff;
  return buff.mid(pos,buff.length()-pos);
}

long HTTPConnection::length() const
{
  return buff.length();
}

long HTTPConnection::position() const
{
  return pos;
}

void HTTPConnection::position (long p)
{
  pos = p;
}

// do not use this!!!
bool HTTPConnection::dataready() 
{  
  /*
  struct pollfd pfd;
  memset(&pfd,0,sizeof(struct pollfd));
  pfd.fd=socket->getfd();
  pfd.events=POLLIN;
  
  int p=poll(&pfd,1,100);
  */
  fd_set x;
  
  FD_ZERO(&x);  
  FD_SET(socket->getfd(),&x);
  
  struct timespec tv;
  tv.tv_sec=0;//(timeoutsec>=0?timeoutsec:0);
  tv.tv_nsec=retry_usecs*1000;

  //  sigset_t sig;
  int r= pselect (socket->getfd()+1,&x,NULL,NULL,&tv,NULL);
  if (r<0)
    {
      timeoutct=0;
      perror("HTTPConnection::dataready(): select()");
    }
  else if (r)
    {
      if (FD_ISSET(socket->getfd(),&x))
	{	  
	  timeoutct=time(NULL);
	  return true;
	}
    }
  else
    {
      if (!FD_ISSET(socket->getfd(),&x))
	{
	  timeoutct=0;
	  return false;
	}
    } // no data

  return false;
}

bool HTTPConnection::bufferready() const
{
  return (pos<buff.length());
}

#if 0
long HTTPConnection::readblocking() 
{
  long r=0;
  while (!timeout())
    {
      r=read();
      fprintf (stdout,"read %d\n",r);
      if (!r)
	{ 
	  timeoutct=0;
	  return 0;
	}
      else if (r>0) return r;
      if (!(errno&EAGAIN))
	{
	  timeoutct=0;
	  return r;
	}
       usleep(100000);
    }
  usleep(retry_usecs);
  return 0;
}
#endif

long HTTPConnection::readblockingn(long l)
{
  if(l<=0) return 0;
  while (! timeout())
    {
      if(length() - position() >= l)
	{
	  //postdata=conn->buffernew().mid(0,l);
	  //conn->position(conn->position()+l);
	  return length()-position();
	}
      readblocking();
    }
  return -1; 
}

xstring HTTPConnection::popn(long l)
{
  int r = readblockingn(l);
  if(r<=0) return "";
  xstring data=buffernew().mid(0,l);
  position(position()+l);
  return data;
}

long HTTPConnection::readblocking()
{
  long r=0;
  
  fd_set w;
  struct timeval tv;
  FD_ZERO(&w);
  
  while (r<=0)
    {
      //tv.tv_sec = 5;
      //tv.tv_usec = 0;
      FD_SET(socket->getfd(),&w);
      r=select(socket->getfd()+1,&w,NULL,NULL,NULL);//&tv);
      
      if(r<=0)
	{
	 
	  perror("HTTPConnection::readblocking(): select()");
	  return -1;
	}
      else if(r)
	{
	  r=read();
	  if(r<0 && errno==EAGAIN)
	    continue;
	  if(r<=0)
	    {
	      timeoutct=0;
	      r=-1;
	    }
	  break;
	}
    }
  //  usleep(100000);
#ifdef DEBUG
  fprintf(stdout,"recv %d bytes.\n",r);
#endif
  return r;
}

long HTTPConnection::send(const xstring& data)
{
  long r=0;
  
  fd_set rd,w;
  struct timeval tv;
  FD_ZERO(&w);
  FD_ZERO(&rd);
  
  while (r<=0)
    {
      //tv.tv_sec = 5;
      //tv.tv_usec = 0;
      FD_SET(socket->getfd(),&w);
      FD_SET(socket->getfd(),&rd);
      r=select(socket->getfd()+1,&rd,&w,NULL,NULL);//&tv);
      
      if(r<=0)
	{
	 
	  perror("HTTPConnection::send(): select()");
	  return -1;
	}
      
      else if(r)
	{
	  if(FD_ISSET(socket->getfd(),&rd))
	    readblocking();
	  
	  if(FD_ISSET(socket->getfd(),&w))
	    {
	      r=socket->send(data.text(),data.length());
	      if(r<0 && errno==EAGAIN)
		continue;
	      if(r<=0)
		{
		  timeoutct=0;
		  r=-1;
		}
	      break;
	    }
	}
    }
#ifdef DEBUG
  fprintf(stdout,"sent %d of %d bytes.\n",r,data.length());
#endif

  if (r<=0)
    return r;

  if (r<data.length()) // pretty hack :-) beware of loops!!
    r=send(data.mid(r,data.length()-r));

  return r;
}

bool  HTTPConnection::timeout()
{
  return timeoutct==0;
  return timeoutct+timeout_usecs/1000000 < time(NULL); 
}
