// $Id: main_thread.c,v 1.10 2006/03/19 20:17:14 khlut Exp $
// Copyright 2005 Elphel, Inc.
//
// This file is part of GenReS.
//
//    GenReS is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    GenReS is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with GenReS; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "genres.h"
#include <sys/wait.h>
#include <signal.h>
#include <time.h>

#define LICENSE "Copyright 2005 Elphel, Inc.<br>"\
"This plug-in is free software; you can redistribute it and/or modify"\
" it under the terms of the <a href=http://www.fsf.org/licensing/licenses/gpl.html>GNU General Public License</a> as published by"\
" <a href=http://www.fsf.org>the Free Software Foundation</a>; either version 2 of the License, or"\
" (at your option) any later version."

#define Pd ((PD*)instance->pdata)

NPNetscapeFuncs NPN;
NPIdentifier ready,	run;
int debug=1;
static int instance_cnt;

/*
** Convert any NPVariant to string
** result must be freed by free()
*/

char * NPVariant_to_string(const NPVariant * v)
{
    char r[128];
    switch(v->type) {
	case NPVariantType_Void: return strdup("(void)");
	case NPVariantType_Null: return strdup("0");
	case NPVariantType_Bool: return strdup(v->value.boolValue?"1":"0");
	case NPVariantType_Int32: snprintf(r, sizeof r, "%d", v->value.intValue);	break;
	case NPVariantType_Double: snprintf(r, sizeof r, "%g", v->value.doubleValue);	break;
	case NPVariantType_String: {
	    char *n=malloc(v->value.stringValue.utf8length+1);
	    if(n) {
		strncpy(n,v->value.stringValue.utf8characters,v->value.stringValue.utf8length);
		n[v->value.stringValue.utf8length]=0;
	    }
	    return n;
	}
	case NPVariantType_Object: return strdup("(Object)");
    }
    r[sizeof r - 1]=0;
    return strdup(r);
}

/*
** Get NPObject property as a string ( result must be freed by free() )
** object_type=NPNVWindowNPObject: owner window
** object_type=NPNVPluginElementNPObject: our EMBED or OBJECT htlm element
*/

char * objectprop(NPP instance, int object_type, const char *name)
{
    NPObject *a=0; char  *s0, *s1, *n;
    NPIdentifier id;
    NPVariant var={NPVariantType_Void};
    Debug("Objectprop");
    if(NPN.getvalue(instance, object_type, &a))	return 0;
    Debug(" %d\n",a->referenceCount);
    s0=n=strdup(name);
    do {
	s1=strchr(s0,'.');
	if(s1) *s1++=0;
	id=NPN.getstringidentifier(s0);
	Debug("  %s (%x)\n",s0,(int)id);
	if(!NPN.hasproperty(instance,a,id)) 	break;
	if(!NPN.getproperty(instance,a,id,&var))	break;
	NPN.releaseobject(a); a=0;
	Debug("  %d\n",var.type);
	if(var.type!=NPVariantType_Object)	break;
	a=var.value.objectValue;
	var.type=NPVariantType_Void;
	s0=s1;
    }while(s0);
    free(n);
    n=NPVariant_to_string(&var);
    if(a) NPN.releaseobject(a);
    NPN.releasevariantvalue(&var);
    return n;
}

/*
** Variable manipulation
*/
int do_send_value(PD*pd, char *key, char *value) {
  Debug("%p send %s=%s\n",pd,key,value);
  return fprintf(pd->slavectrl,"%s=%s\n",key,value);
}
void send_value(PD *pd, char *key, char *value) //send variable to slave
{
  PR_Lock(pd->lock);
  if(( (!pd->slavectrl) || do_send_value(pd,key,value)<=0 ) && pd->status==Running // try to reanimate slave
     && ( strcmp(key,"run") || strcmp(value,"0") )) { 	// ignore run=0 when slave is nonfunctional
    if(debug)perror("write to pipe");
   
    PR_Unlock(pd->lock);
    wait_slave(pd);
    pd->status=Init;
    run_slave(pd,pd->mime,1);	// run slave parallel process
    
    // slave observer thread
    if(!( pd->observer=PR_CreateThread(PR_USER_THREAD, (void (*)(void*))&observer_thread,
                                  pd, PR_PRIORITY_HIGH,
                                  PR_GLOBAL_THREAD, PR_JOINABLE_THREAD, 0) )) return;
    // send all current variables
    int i;
    for(i=0;i<pd->vars->size;i++) {
	if(pd->vars->var[i].key==run || pd->vars->var[i].key==ready) continue;
	char *sv=NPVariant_to_string(&(pd->vars->var[i].value));
	do_send_value(pd, NPN.utf8fromidentifier(pd->vars->var[i].key), sv);
	free(sv);
    }
    pd->xid && fprintf(pd->slavectrl,"xid=%x\n",(unsigned)pd->xid);
    do_send_value(pd,key,value);
    pd->status=Running;
    
    return;
    
  }else if(pd->status==Running && !strcmp(key,"run") && !strcmp(value,"0") && pd->slavectrl) {
    PR_Unlock(pd->lock);
    wait_slave(pd);
    pd->status=Running;

    return;
      
  }
  PR_Unlock(pd->lock);  
}

void send_value_hex(PD *pd, char *key, int value)
{
  char buf[22];
  sprintf(buf,"%x",value);
  send_value(pd,key,buf);
}

/*
** Netscape Scriptable class functions exportable through NPObject
*/

NPObject * NPAllocate(NPP npp, NPClass *aClass);
void NPDeallocate(NPObject *npobj);
void NPInvalidate(NPObject *npobj);
bool NPHasMethod(NPObject *npobj, NPIdentifier name);
bool NPInvoke(NPObject *npobj, NPIdentifier name, const NPVariant *args, uint32_t argCount, NPVariant *result);
bool NPInvokeDefault(NPObject *npobj, const NPVariant *args, uint32_t argCount, NPVariant *result);
bool NPHasProperty(NPObject *npobj, NPIdentifier name);
bool NPGetProperty(NPObject *npobj, NPIdentifier name, NPVariant *result);
bool NPSetProperty(NPObject *npobj, NPIdentifier name, const NPVariant *value);
bool NPRemoveProperty(NPObject *npobj, NPIdentifier name);

// undefined functions

#define	NPInvalidate 0
#define	NPInvoke 0
#define	NPInvokeDefault 0
#define NPRemoveProperty 0

static NPClass genresClass = {
    NP_CLASS_STRUCT_VERSION,
    NPAllocate,		NPDeallocate, 		NPInvalidate,
    NPHasMethod,	NPInvoke,		NPInvokeDefault,
    NPHasProperty,	NPGetProperty,		NPSetProperty,
    NPRemoveProperty
};
typedef struct {
    NPObject npo;
    NPP instance;
} PO;

NPObject * NPAllocate(NPP npp, NPClass *aClass)
{
    PO *po=PR_Malloc(sizeof(PO));
    Debug("Obj allocate %p\n", npp->pdata);
    if(po) {
	po->npo._class=&genresClass;
	po->npo.referenceCount=0;
	po->instance=npp;
    }
    return (NPObject*)po;
}
void NPDeallocate(NPObject *npobj)
{
    Debug("Obj deallocate %p\n", ((PO*)npobj)->instance->pdata);
    PR_Free(npobj);
}
bool NPHasMethod(NPObject *npobj, NPIdentifier name)
{
    Debug("Has method ");
    if(NPN.identifierisstring(name)) {
        char *s=NPN.utf8fromidentifier(name);
	Debug("string %s\n",s);
	PR_Free(s);
    }else{
	Debug("int %d\n",NPN.intfromidentifier(name));
    }
    return false;
}
bool NPHasProperty(NPObject *npobj, NPIdentifier name)
{
    NPP instance=((PO*)npobj)->instance;
    Debug("Has property ");
    if(NPN.identifierisstring(name)) {
        char *s=NPN.utf8fromidentifier(name);
	Debug("string %s ",s);
	PR_Free(s);
    }else{
	Debug("int %d ",NPN.intfromidentifier(name));
	return false;
    }
    
    return storage_get(Pd->vars, name)!=0;
}

bool NPGetProperty(NPObject *npobj, NPIdentifier name, NPVariant *result)
{
    NPP instance=((PO*)npobj)->instance;
    Debug("Get property");
    NPVariant *var=storage_get(Pd->vars, name);
    if(!var) {Debug(" no\n"); return false;}
    Debug(" yes\n");
    return NPVariant_cpy(result, var);
}

bool NPSetProperty(NPObject *npobj, NPIdentifier name, const NPVariant *value)
{
    NPP instance=((PO*)npobj)->instance;
    char *sv=NPVariant_to_string(value);
    Debug("Set property %s %s\n",NPN.utf8fromidentifier(name), sv);
    send_value(Pd, NPN.utf8fromidentifier(name), sv);
    free(sv);
    return true;
}

/*
** Global functions exported to browser
*/

char *NP_GetMIMEDescription(void)
{
  Debug("NP_GetMIMEDescription\n");
  return get_mime_list();
}

NPError NP_GetValue(void *future, NPPVariable variable, void *value)
{
  NPError err = NPERR_NO_ERROR;
  Debug("NP_GETVALUE\n");
  switch (variable) {
    case NPPVpluginNameString:
      *((char **) value) = "The GenReS Mozilla plugin V " RELEASE;
      break;
    case NPPVpluginDescriptionString:
      *((char **) value) = "<a href=https://savannah.nongnu.org/projects/genres>GENeric REconfigureable Scriptable plugin</a> for Mozilla/Firefox<br>"
    			   "Version " RELEASE "<br>"
			   "Author: Sergey Khlutchin<br>"
			   LICENSE;
      break;
    default:
      err = NPERR_GENERIC_ERROR;
  }
  return err;
}

NPError NP_Initialize(NPNetscapeFuncs * moz_funcs, NPPluginFuncs * plugin_funcs)
{
  void *ServiceManager=0;
  
  Debug("NP_Initialize\n");
  if (moz_funcs == NULL || plugin_funcs == NULL)
    return NPERR_INVALID_FUNCTABLE_ERROR;

  if ((moz_funcs->version >> 8) > NP_VERSION_MAJOR)
    return NPERR_INCOMPATIBLE_VERSION_ERROR;
  if (moz_funcs->size < sizeof (NPNetscapeFuncs))
    return NPERR_INVALID_FUNCTABLE_ERROR;
  if (plugin_funcs->size < sizeof (NPPluginFuncs))
    return NPERR_INVALID_FUNCTABLE_ERROR;

  NPN= *moz_funcs;
  ready=   NPN.getstringidentifier("ready");
  run=     NPN.getstringidentifier("run");
  instance_cnt=0;
  
  NPN.getvalue(0, NPNVserviceManager, &ServiceManager);
  init_qservice(ServiceManager);

  plugin_funcs->size = sizeof (NPPluginFuncs);
  plugin_funcs->version = (NP_VERSION_MAJOR << 8) + NP_VERSION_MINOR;
  plugin_funcs->newp = NPP_New;
  plugin_funcs->getvalue = NPP_GetValue;
  plugin_funcs->setwindow = NPP_SetWindow;
  plugin_funcs->destroy = NPP_Destroy;
  plugin_funcs->newstream = NPP_NewStream;
//	functions undefined
//  plugin_funcs->destroystream = NPP_DestroyStream;
//  plugin_funcs->setvalue = NPP_SetValue;
//  plugin_funcs->writeready = NPP_WriteReady;
//  plugin_funcs->write = NPP_Write;
//  plugin_funcs->asfile = NPP_StreamAsFile;
//  plugin_funcs->print = NPP_Print;
//  plugin_funcs->urlnotify = NPP_URLNotify;
//  plugin_funcs->event = NPP_HandleEvent;

  return NPERR_NO_ERROR;
}

NPError NP_Shutdown(void)
{
  Debug("NP_ShutDown\n");
  if(instance_cnt) {
    Debug("Have non-terminated instance, can not shut down!\n");
    return NPERR_GENERIC_ERROR;
  }
  shut_qservice();
  return NPERR_NO_ERROR;
}

/************************************************************
** Instance functions exported to browser by NP_Initialize **
************************************************************/

NPError NPP_New(NPMIMEType mime, NPP instance, uint16 mode, int16 argc, 
    char* argn[], char* argv[], NPSavedData* saved)
{
    int i; PD *pd;
    
    Debug("NPP_NEW %s %x %x %x %x\n", mime, (int)instance, (int)mode, (int)argc, (int)saved );
    
    if(!( pd=PR_MALLOC(sizeof *pd) ))	return NPERR_OUT_OF_MEMORY_ERROR;
    if(!( pd->vars=storage_new() ))	{ PR_Free(pd); return NPERR_OUT_OF_MEMORY_ERROR; }
    instance->pdata=pd;
    pd->instance=instance;
    pd->browser= PR_GetCurrentThread();
    pd->xid=0;
    pd->refcnt=1;
    pd->lock=PR_NewLock();
    pd->status=Init;
    pd->mime=strdup(mime);
    run_slave(pd,mime,0);	// run slave parallel process
    
    // slave observer thread
    if(!( pd->observer=PR_CreateThread( PR_USER_THREAD, (void (*)(void*))&observer_thread,
                                  pd, PR_PRIORITY_HIGH,
                                  PR_GLOBAL_THREAD, PR_JOINABLE_THREAD, 0) )) return NPERR_GENERIC_ERROR;
    instance_cnt++;

    // send variables to slave
    {
//     char *baseuri=objectprop(instance, NPNVWindowNPObject, "document.baseURI");
     char *baseuri=objectprop(instance, NPNVPluginElementNPObject, "baseURI");
     if(baseuri) {  send_value(pd,"baseURI",baseuri);  free(baseuri);  }
    }
    for(i=0; i<argc; i++) send_value(pd,argn[i],argv[i]);
    pd->status=Running;
    
    Debug("%p refcnt=%d\n",pd, Pd->refcnt);
    process_pending_events(Pd);
    Debug("%p refcnt=%d\n",pd, Pd->refcnt);
    process_pending_events(Pd);
    Debug("%p refcnt=%d\n",pd, Pd->refcnt);
    
    return NPERR_NO_ERROR;
}

void free_pd(PD *pd) { // free plugin data if reference counter zeroes
    if(--pd->refcnt)	{PR_Unlock(pd->lock); return;}
    Debug("free pd\n");
    storage_free(pd->vars);
    free(pd->mime);
    PR_Unlock(pd->lock);
    PR_DestroyLock(pd->lock);
    PR_Free(pd);
    instance_cnt--;
}

void wait_slave(PD * pd) { // wait for slave termination
    NPVariant *var;	int i;
    do{
        PR_Sleep(100);
        Debug("%p Process events\n", pd);
        process_pending_events(pd);
        var=storage_get(pd->vars, ready);
        if(!var || var->type!=NPVariantType_String) {
            Debug("variable \"ready\" not found\n");
            break;
        }
    }while(var->value.stringValue.utf8characters[0]=='1');

    Debug("%p refcnt=%d\n", pd, pd->refcnt);
    
    pd->status=Cancelling;
    PR_Sleep(100);
    process_pending_events(pd);
    Debug("wait %d\n",pd->slave);
    if(waitpid(pd->slave,&i,WNOHANG)<0) perror("wait");
    Debug("wait observer\n");
    if(pd->observer) {PR_JoinThread(pd->observer); pd->observer=0;}
    if(pd->slavectrl) {fclose(pd->slavectrl); pd->slavectrl=0;}
    Debug("Process events %p\n", pd);
    process_pending_events(pd);
    Debug("%p refcnt=%d\n", pd, pd->refcnt);
    process_pending_events(pd);
    Debug("%p refcnt=%d\n", pd, pd->refcnt);
}

NPError NPP_Destroy (NPP instance, NPSavedData** save)
{
    Debug("%p NPP_Destroy\n", instance->pdata);
    if(Pd) {
      if(Pd->observer && Pd->status==Running) {
	send_value_hex(Pd,"run", 0);
	wait_slave(Pd);
      }
      PR_Lock(Pd->lock);
        if(Pd->refcnt>1) { Debug("Have references!\n"); PR_Unlock(Pd->lock); return NPERR_GENERIC_ERROR; }
        Pd->instance=0;
      free_pd(Pd);
      instance->pdata=0;
    }
    return NPERR_NO_ERROR;
}

NPError NPP_GetValue(NPP instance, NPPVariable variable, void *value)
{
    Debug("%p NPP_GETVALUE %d %x\n", instance->pdata, (int) variable, (int)value);
    switch(variable) {
    case NPPVpluginScriptableNPObject: {
	*((NPObject **) value) = NPN.createobject(instance, &genresClass);
	} break;
    case NPPVpluginNeedsXEmbed: 
	*((PRBool *) value) = PR_TRUE;
	break;

    default:
        return NPERR_INVALID_PARAM;
    }

    return NPERR_NO_ERROR;
}


NPError NPP_SetWindow(NPP instance, NPWindow* npwindow)
{
    Window xid= (Window)npwindow->window;
    
    Debug("%p NPP_SETWINDOW %x %x %x\n", instance->pdata, (int)xid,
           (int)npwindow->ws_info, (int)npwindow->type );
    
    if(Pd->xid!=xid) {
    //TODO: display, type
	send_value_hex(Pd,"xid", (int)xid);
	if(!Pd->xid) send_value_hex(Pd,"run", 1);
	Pd->xid=xid;
    }
    return NPERR_NO_ERROR;
}

NPError NPP_NewStream(NPP instance, NPMIMEType type, NPStream* stream, NPBool seekable, uint16* stype)
{
    Debug("%p NPP_NEWSTREAM %x\n", instance->pdata, (int)seekable);
    return NPERR_GENERIC_ERROR;
}

/*
** Receiving events from observer
*/

typedef struct {
    PLEvent e;
    char * str;
    PD *pd;
    NPP instance;
} ObserveEvent;

void * PR_CALLBACK HandleEvent(ObserveEvent* event);
void PR_CALLBACK DestroyEvent(ObserveEvent* event);

PLEvent * strevent( PD *pd, const char * str)  // make event from a string (is called from other thread)
{
    if (!pd) return 0;
    PR_Lock(pd->lock);
    if(!pd->instance)	{ PR_Unlock(pd->lock); return 0; }
    ObserveEvent *event = PR_MALLOC(sizeof(ObserveEvent));
    if (!event)		{ PR_Unlock(pd->lock); return 0; }

    if(strncmp(str,"debug=",6)==0) debug=str[6]-'0';
    Debug("%p %d newEvent %s\n", pd, time(0), str);
    
    PL_InitEvent( &event->e, pd, HandleEvent, DestroyEvent);
    event->str=PR_Malloc(strlen(str)+3);
    strcpy(event->str+2, str); // reserve 2 chars at the left for "on"
    pd->refcnt++;	event->pd=pd;
    assert(pd->refcnt>1);
    
    PR_Unlock(pd->lock);
    return &event->e;
}

void PR_CALLBACK DestroyEvent(ObserveEvent* event)
{
    if(event) {
	PR_Lock(event->pd->lock);
	Debug("%p DestroyEvent refcnt=%d\n", event->pd, event->pd->refcnt);
	free_pd(event->pd);
	if(event->str) PR_Free(event->str);
	PR_Free(event);
    }
}

void * PR_CALLBACK HandleEvent(ObserveEvent* event)
{
    Debug("%p HandleEvent \"%s\"\t", event->pd, event->str+2);
    size_t namelen;
    for( namelen=2 ; ; namelen++ ) {
    if( !event->str[namelen] ) { Debug("I\n");		return 0; }
    else if( event->str[namelen] == ':'  ||  event->str[namelen] == '=' ) break;
	else if( event->str[namelen] & 128 ) event->str[namelen]=' ';
    }
    event->str[namelen]=0;
    char *name=strrchr(event->str+2,' ');
    if(!name++) name=event->str+2;
    if(!*name) {
	Debug("I\n");		return 0;
    }
    NPVariant val;
    STRINGZ_TO_NPVARIANT( event->str+namelen+1, val );
    NPObject *obj; 
    if(event->pd->instance && NPN.getvalue(event->pd->instance, NPNVPluginElementNPObject, &obj)==NPERR_NO_ERROR) {
	if( storage_set( &(event->pd->vars), NPN.getstringidentifier(name), &val ) ) {
	    if(strcmp(name,"debug")==0) debug=event->str[namelen+1]-'0';
	    // invoke a callback function
	    name-=2; name[0]='o'; name[1]='n';
	    int i;
	    val.type=NPVariantType_Void;
	    NPIdentifier id0=NPN.getstringidentifier(name);
	    for(i=2;name[i];i++) if(name[i]>='A' && name[i]<='Z') name[i]+='a'-'A';
	    NPIdentifier id1=NPN.getstringidentifier(name);
	    
	    if(!(   NPN.invoke(event->pd->instance, obj, id0, &val, 0, &val)
	         || NPN.invoke(event->pd->instance, obj, id1, &val, 0, &val)
		)&&( Debug("getprop "), NPN.getproperty(event->pd->instance, obj, id0, &val) && NPVARIANT_IS_STRING(val)
	         || (NPN.releasevariantvalue(&val), NPN.getproperty(event->pd->instance, obj, id1, &val) && NPVARIANT_IS_STRING(val))
		)
	    ) {
		Debug("evaluate ");
		//In "evaluate" the instance can be desroyed
		//NPN.evaluate(event->pd->instance, obj, &(val.value.stringValue), 0);
		char *url=PR_Malloc(val.value.stringValue.utf8length+12);
		strcpy(url,"javascript:");
		strncpy(url+11,val.value.stringValue.utf8characters, val.value.stringValue.utf8length);
		url[val.value.stringValue.utf8length+11]=0;
		Debug("geturl '%s' ",url);
		NPN.geturl(event->pd->instance, url, "_self");
		PR_Free(url);
	    }
	    
	    NPN.releaseobject(obj);
	    NPN.releasevariantvalue(&val);
	    
	}
	Debug("S\n");
    }else Debug("E\n");
    return 0;
}
