/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

/*!
  \file snp.h
  \brief wrapper for GWAS marker metadata
 */
#ifndef __GENEZIP__SNP_H__
#define __GENEZIP__SNP_H__
#include <string>
#include <vector>
#include <stdexcept>
#include "genezip/config.h"
#include "genezip/helper_functions.h"
namespace genezip_utils {
  /*!
    \class snp
    \brief wrapper for GWAS marker metadata
  */
  class snp {
  public:
    /*!
      \brief constructor
    */
    snp() 
      : _rsid(""),
      _chromosome(""),
      _position(0),
      _gposition(0.0),
      _frequency(0.0),
      _missingness(0.0),
      _quality(0.0) {}
    /*!
      \brief copy constructor
      @param other existing object
    */
    snp(const snp &other)
      : _rsid(other._rsid),
      _chromosome(other._chromosome),
      _position(other._position),
      _gposition(other._gposition),
      _alleles(other._alleles),
      _frequency(other._frequency),
      _missingness(other._missingness),
      _quality(other._quality) {}
    /*!
      \brief destructor
    */
    ~snp() throw() {}
    /*!
      \brief get the label of the marker
      \return the label of the marker
    */
    inline const std::string &rsid() const {return _rsid;}
    /*!
      \brief get the chromosome of the marker (as a character string)
      \return the chromosome of the marker (as a character string)
    */
    inline const std::string &chromosome() const {return _chromosome;}
    /*!
      \brief get the chromosome of the marker (as a number)
      \return the chromosome of the marker (as a number)
      \warning if the code is not [1,22] or X/Y/XY/M (PLINK codes), will throw
    */
    unsigned chromosome_numeric() const {
      return chromosome_numeric(chromosome());
    }
    /*!
      \brief get the physical position of the marker
      \return the physical position of the marker
    */
    inline unsigned position() const {return _position;}
    /*!
      \brief get the genetic position of the marker
      \return the genetic position of the marker
      \warning many files contain placeholder 0 instead of actual 
      genetic positions
      \warning usually stored in centiMorgans, but periodically Morgans 
      are encountered
    */
    inline const double &gposition() const {return _gposition;}
    /*!
      \brief get the first allele for the marker
      \return the first allele for the marker
    */
    inline const std::string &allele_one() const {
      return allele_n(1);
    }
    /*!
      \brief get the second allele for the marker
      \return the second allele for the marker
    */
    inline const std::string &allele_two() const {
      return allele_n(2);
    }
    /*!
      \brief get the nth allele for a marker
      @param index the number of the desired allele on [1,n_alleles]
      \return the nth allele for the marker
    */
    const std::string &allele_n(unsigned index) const;
    //! get the snp frequency
    //! \return the snp frequency
    inline const double &frequency() const {return _frequency;}
    //! get the snp quality metric
    //! \return the snp quality metric
    inline const double &quality() const {return _quality;}
    //! get the per-snp missingness at this marker
    //! \return the per-snp missingness at this marker
    inline const double &missingness() const {return _missingness;}
    /*!
      \brief get const_iterator to the beginning of the allele vector,
      for iteration
      \return const_iterator to the beginning of the allele vector,
      for iteration
    */
    inline std::vector<std::string>::const_iterator allele_begin() const {
      return _alleles.begin();
    }
    /*!
      \brief get const_iterator to the allele vector end, for iteration
      \return const_iterator to the allele vector end, for iteration
    */
    inline std::vector<std::string>::const_iterator allele_end() const {
      return _alleles.end();
    }
    /*!
      \brief set the marker label
      @param s new marker label
    */
    inline void rsid(const std::string &s) {_rsid = s;}
    /*!
      \brief set the marker chromosome (as a character string)
      @param s new marker chromosome (as a character string)
    */
    inline void chromosome(const std::string &s) {
      _chromosome = s;
    }
    /*!
      \brief set the marker chromosome (from an integer)
      @param u new marker chromosome (as an integer)
    */
    inline void chromosome(unsigned u) {
      _chromosome = genezip_utils::to_string<unsigned>(u);
    }
    /*!
      \brief set the marker physical position
      @param s new marker physical position
    */
    inline void position(const std::string &s) {
      _position = genezip_utils::from_string<unsigned>(s);
    }
    /*!
      \brief set the marker genetic position
      @param s new marker genetic position
    */
    inline void gposition(const std::string &s) {
      _gposition = genezip_utils::from_string<double>(s);
    }
    /*!
      \brief set the first allele for the marker
      @param allele new first allele for the marker
    */
    inline void allele_one(const std::string &allele) {
      allele_n(allele, 1);
    }
    /*!
      \brief set the second allele for the marker
      @param allele new second allele for the marker
    */
    inline void allele_two(const std::string &allele) {
      allele_n(allele, 2);
    }
    /*!
      \brief set the nth allele for the marker
      @param allele new nth allele for the marker
      @param index the number of the new allele on [1,n_alleles]
    */
    void allele_n(const std::string &allele, unsigned index);
    //! set the snp frequency
    //! \param d the snp's new frequency
    inline void frequency(const double &d) {
      _frequency = d;
    }
    //! set the snp quality metric
    //! \param d the snp's new quality metric
    inline void quality(const double &d) {
      _quality = d;
    }
    //! set the per-snp missingness at this marker
    //! \param d the snp's new missingness rate
    inline void missingness(const double &d) {
      _missingness = d;
    }
    //! equality operator
    //! \param other object to which to compare current object
    //! \return whether current marker is equivalent to compared marker
    //! \warning comparison based exclusively on marker name
    //! \sa rsid()
    inline bool operator==(const snp &other) const {
      return !rsid().compare(other.rsid());
    }
    //! less than operator
    //! \param other object to which to compare current object
    //! \return whether current marker is less than compared marker
    //! \warning comparison based exclusively on marker name
    //! \sa rsid()
    inline bool operator<(const snp &other) const {
      return rsid().compare(other.rsid()) < 0;
    }
    //! lte operator
    //! \param other object to which to compare current object
    //! \return whether current marker is lte the compared marker
    //! \warning comparison based exclusively on marker name
    //! \sa rsid()
    //! \sa operator<
    //! \sa operator==
    inline bool operator<=(const snp &other) const {
      return *this < other || *this == other;
    }
    //! greater than operator
    //! \param other object to which to compare current object
    //! \return whether current marker is greater than the compared marker
    //! \warning comparison based exclusively on marker name
    //! \sa rsid()
    //! \sa operator<=
    inline bool operator>(const snp &other) const {
      return !(*this <= other);
    }
    //! gte operator
    //! \param other object to which to compare current object
    //! \return whether current marker is gte the compared marker
    //! \warning comparison is based exclusively on marker name
    //! \sa rsid()
    //! \sa operator>
    inline bool operator>=(const snp &other) const {
      return !(*this < other);
    }
  private:
    /*!
      \brief convert a chromosome code string to a number
      \param s the chromosome code
      \return the numeric chromosome of the marker
      \warning if the code is not [1,22] or X/Y/XY/M (PLINK codes), will throw
    */
    unsigned chromosome_numeric(const std::string &s) const;
    std::string _rsid;//!< snp/marker label
    std::string _chromosome;//!< chromosome of snp
    unsigned _position;//!< physical position of snp
    double _gposition;//!< genetic position of snp
    std::vector<std::string> _alleles;//!< snp alleles
    double _frequency;//!< snp (A1) frequency
    double _missingness;//!< per-snp missingness
    double _quality;//!< quality metric (ie MACH r2)
  };
}
#endif //__SNP_H__
