/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

/*!
  \file smart_pointer_array.h
  \brief a small implementation of a smart pointer, to an array

  Certain systems segfault if the wrong delete[] syntax is used.
  Also using the opportunity to overload array indexing/arithmetic.
 */
#ifndef __GENEZIP__SMART_POINTER_ARRAY_H__
#define __GENEZIP__SMART_POINTER_ARRAY_H__

#include <stdexcept>
namespace genezip_utils {
  /*!
    \class smart_pointer_array
    \brief a small implementation of a smart pointer, to an array
    @tparam value_type type of object whose pointer the object will manage
  */
  template <class value_type>
    class smart_pointer_array {
  public:
    /*!
      \brief constructor
    */
    smart_pointer_array() : _ptr(0), _count(0) {}
    /*!
      \brief constructor: set managed pointer
      @param ptr pointer to be managed (should not be deleted by user)
    */
    smart_pointer_array(value_type *ptr);
    /*!
      \brief copy constructor
      @param other existing smart pointer
    */
    smart_pointer_array(const smart_pointer_array &other);
    /*!
      \brief destructor
    */
    ~smart_pointer_array() throw();
    //! assignment operator: copy existing object into this one
    //! @param other existing object
    //! \return reference to this object
    smart_pointer_array &operator=(const smart_pointer_array &other);
    /*!
      \brief get the managed pointer.  control of pointer remains with object
      \return managed pointer (or NULL/0)
    */
    value_type *get() const {return _ptr;}
    /*!
      \brief set the managed pointer.  control of pointer now with object.
      Existing pointer cleared.
      @param ptr new pointer to be managed.
    */
    void set(value_type *ptr);
    /*!
      \brief pointer arithmetic
      @param offset value to be added to pointer (NOT to referenced object)
      \return incremented pointer
    */
    value_type *operator+(unsigned offset) const;
    /*!
      \brief array indexing
      @param offset index of array to be accessed
      \return reference to object at array location
      \warning obviously, there is no error checking of the incrementation
    */
    value_type &operator[](unsigned offset) const;
    /*!
      \brief get the number of copies of the managed pointer
      \return the number of copies of the managed pointer (including *this)
    */
    unsigned get_count() const {return _count ? (*_count) : 0;}
    //! get the pointer to the count variable for this pointer
    //! \return pointer to the count variable for this pointer
    unsigned *get_count_pointer() const {return _count;}
  private:
    /*!
      \brief allocate the pointer counter
      @param initial_count initial value of the counter (presumably always 1)
    */
    void allocate_count(unsigned initial_count);
    /*!
      \brief clear the internal pointers.  Delete if count==1
    */
    void clear_internals();
    //! the managed pointer
    value_type *_ptr;
    //! a shared count of the number of instances of this pointer
    unsigned *_count;
  };



  template <class value_type>
    smart_pointer_array<value_type>::smart_pointer_array(value_type *ptr) 
    : _ptr(ptr), 
    _count(0) {
    if (_ptr) allocate_count(1);
  }

  template <class value_type>
    smart_pointer_array<value_type>::smart_pointer_array
    (const smart_pointer_array &other) 
    : _ptr(other._ptr), 
    _count(other._count) {
    if (_count) ++(*_count);
  }

  template <class value_type>
    smart_pointer_array<value_type>::~smart_pointer_array() throw() {
    clear_internals();
  }
  template <class value_type>
    smart_pointer_array<value_type> &smart_pointer_array
    <value_type>::operator=(const smart_pointer_array &other) {
    if (this != &other) {
      clear_internals();
      _ptr = other.get();
      _count = other.get_count_pointer();
      if (_count) ++(*_count);
    }
  }

  template <class value_type>
    void smart_pointer_array<value_type>::set(value_type *ptr) {
    if (_ptr) clear_internals();
    if (_ptr = ptr) allocate_count(1);
  }

  template <class value_type>
    value_type *smart_pointer_array<value_type>::operator+(unsigned offset)
    const {
    if (_ptr) return _ptr + offset;
    throw std::domain_error("smart_pointer_array::operator+: arithmetic on "
			    "nonexistent pointer");
  }

  template <class value_type>
    value_type &smart_pointer_array<value_type>::operator[](unsigned offset)
    const {
    if (_ptr) return *(_ptr + offset);
    throw std::domain_error("smart_pointer_array::operator[]: indexing on "
			    "nonexistent pointer");
  }

  template <class value_type>
    void smart_pointer_array<value_type>::allocate_count
    (unsigned initial_count) {
    if (_count) throw std::domain_error("smart_pointer_array::allocate_count:"
					" count exists");
    _count = new unsigned;
    *_count = initial_count;
  }

  template <class value_type>
    void smart_pointer_array<value_type>::clear_internals() {
    if (_ptr) {
      if (_count && (*_count) == 1) {
	delete [] _ptr;
	delete _count;
      } else if (_count) {
	--(*_count);
      }
      _ptr = 0;
      _count = 0;
    }
  }
}
#endif //__SMART_POINTER_ARRAY_H__
