/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/


#include "genezip/multilevel_huffman_hash.h"


void genezip_utils::multilevel_huffman_hash::populate
(const std::vector<std::pair<unsigned,
 std::pair<unsigned, unsigned> > > &huffman_data,
 unsigned tranche_one_threshold,
 unsigned max_bit_length) {
  multilevel_huffman_hash *newhash = 0;
  try {
    _tranche_one_threshold = tranche_one_threshold;
    clear_table();
    _lookup_table.resize(1 << _tranche_one_threshold,
			 std::pair<unsigned, 
			 multilevel_huffman_hash *>(0, NULL));
    std::map<unsigned, 
	     std::vector<std::pair<unsigned,
				   std::pair<unsigned,
					     unsigned> > > > 
      spawned_sublists;
    std::map<unsigned,
	     std::vector<std::pair<unsigned,
				   std::pair<unsigned, 
					     unsigned> > > >::iterator
      sublist_finder;
    for (std::vector<std::pair<unsigned, 
	   std::pair<unsigned, 
	   unsigned> > >::const_iterator iter = huffman_data.begin();
	 iter != huffman_data.end(); ++iter) {
      //check to make sure the value is consistent with the max bit
      //threshold provided
      if (iter->second.first >> max_bit_length)
	throw std::domain_error("genezip_utils::multilevel_huffman_hash"
				"::populate: value provided contains nonzero "
				"bits in unexpected places");

      if (iter->second.second == _tranche_one_threshold) {
	_lookup_table.at(iter->second.first).first = iter->first;
      } else if (iter->second.second < _tranche_one_threshold) {
	unsigned max_addition = 1 << (_tranche_one_threshold - 
				      iter->second.second);
	for (unsigned i = 0; i < max_addition; ++i) {
	  _lookup_table.at((iter->second.first << (_tranche_one_threshold -
						   iter->second.second)) 
			   + i).first = iter->first;
	}
      } else { //iter->second.second > _tranche_one_threshold
	if ((sublist_finder = 
	     spawned_sublists.find(iter->second.first >> 
				   (iter->second.second - 
				    _tranche_one_threshold))) == 
	    spawned_sublists.end()) {
	  sublist_finder = spawned_sublists.insert(std::make_pair
						   (iter->second.first >> 
						    (iter->second.second - 
						     _tranche_one_threshold),
						    std::vector<std::pair<unsigned, std::pair<unsigned, unsigned> > >())).first;
	}
	sublist_finder->second.push_back(*iter);
	unsigned good_mask = (1 << (iter->second.second -
				    _tranche_one_threshold)) - 1;
	sublist_finder->second.rbegin()->second.first &= good_mask;
	sublist_finder->second.rbegin()->second.second -= 
	  _tranche_one_threshold;
      }
    }
    //deal with the fact that a bunch of sublists now exist
    for (sublist_finder = spawned_sublists.begin(); sublist_finder != 
	   spawned_sublists.end(); ++sublist_finder) {
      newhash = new multilevel_huffman_hash;
      newhash->populate(sublist_finder->second, max_bit_length - 
			_tranche_one_threshold, max_bit_length - 
			_tranche_one_threshold);
      _lookup_table.at(sublist_finder->first).second = newhash;
      newhash = 0;
    }
  } catch (...) {
    if (newhash) {
      delete newhash;
      newhash = 0;
    }
    throw;
  }
}

unsigned genezip_utils::multilevel_huffman_hash::translate(unsigned huffcode, 
							   unsigned length) 
  const {
  unsigned lookup_val = huffcode;
  multilevel_huffman_hash *ptr = 0;
  if (length < _tranche_one_threshold) {
    lookup_val <<= _tranche_one_threshold - length;
  } else if (length > _tranche_one_threshold) {
    lookup_val >>= length - _tranche_one_threshold;
  }
  if ((ptr = _lookup_table.at(lookup_val).second)) {
    unsigned good_mask = (1 << (length - _tranche_one_threshold)) - 1;
    return ptr->translate(huffcode & good_mask, length -
			  _tranche_one_threshold);
  } else {
    return _lookup_table.at(lookup_val).first;
  }
}

void genezip_utils::multilevel_huffman_hash::clear_table() {
  for (std::vector<std::pair<unsigned, 
	 multilevel_huffman_hash *> >::iterator iter = _lookup_table.begin();
       iter != _lookup_table.end(); ++iter) {
    if (iter->second) {
      delete iter->second;
      iter->second = 0;
    }
  }
  _lookup_table.clear();
}
