/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

#include "genezip/huffman_encoded_stream_generator.h"

void genezip_utils::huffman_encoded_stream_generator::add_token(const vectorlist_search_result &token) {
  switch (token.get_result_type()) {
  case vectorlist_search_result::NONE:
    throw std::domain_error("genezip_utils::huffman_encoded_stream_generator"
			    "::add_token: null token provided to stream");
  case vectorlist_search_result::ENDOFBLOCK:
    {
      huffman_token _token;
      _token.set_type(huffman_token::LITERAL);
      _token.set_value(GENEZIP_LITERAL_VALUE_UPPER_BOUND);
      ++_lengthlit_counts.at(GENEZIP_LITERAL_VALUE_UPPER_BOUND);
      _tokens.push_back(_token);
      return;
    }
  case vectorlist_search_result::LITERAL:
    {
      huffman_token _token;
      _token.set_type(huffman_token::LITERAL);
      _token.set_value(token.get_literal());
      if (token.get_literal() >= _lengthlit_counts.size())
	throw std::domain_error("literal out of bounds: \""
				+ to_string<unsigned>(token.get_literal())
				+ "\"");
      ++_lengthlit_counts.at(token.get_literal());
      _tokens.push_back(_token);
      return;
    }
  case vectorlist_search_result::MATCH:
    {
      //need to handle the distance pointer and match length separately
      huffman_token _token;
      //match length first
      _token.set_type(huffman_token::LENGTH);
      std::pair<unsigned, std::pair<unsigned, unsigned> > encoded_data = 
	encode_length(token.get_match().second);
      _token.set_value(encoded_data.first);
      _token.set_additional(encoded_data.second);
      //add to appropriate huffman counter
      ++_lengthlit_counts.at(encoded_data.first);
      _tokens.push_back(_token);
      //then distance
      _token.clear();
      _token.set_type(huffman_token::DISTANCE);
      encoded_data = encode_distance(token.get_match().first);
      _token.set_value(encoded_data.first);
      _token.set_additional(encoded_data.second);
      //add to appropriate huffman counter
      ++_distance_counts.at(encoded_data.first);
      _tokens.push_back(_token);
      return;
    }
  default:
    throw std::domain_error("genezip_utils::huffman_encoded_stream_generator"
			    "::add_token: unrecognized search result type");
  }
}
void genezip_utils::huffman_encoded_stream_generator::flush_data(std::vector<bool> &target,
								 huffman_code &litlen_code,
								 huffman_code &distance_code) {
  litlen_code.generate_code(_lengthlit_counts, 9);
  distance_code.generate_code(_distance_counts, 6);
  unsigned huffed_value = 0, huffed_length = 0;
  //run through the enqueued tokens and add their bits to the vector
  for (std::vector<huffman_token>::const_iterator iter = _tokens.begin();
       iter != _tokens.end(); ++iter) {
    if (iter->get_type() == huffman_token::LENGTH ||
	iter->get_type() == huffman_token::LITERAL) {
      litlen_code.encode(iter->get_value(), huffed_value, huffed_length);
    } else if (iter->get_type() == huffman_token::DISTANCE) {
      distance_code.encode(iter->get_value(), huffed_value, huffed_length);
    } else throw std::domain_error("genezip_utils::huffman_encoded_stream"
				   "_generator::flush_data: unrecognized "
				   "token type");
    add_value_to_vector(huffed_value,
			huffed_length,
			iter->get_additional_content(),
			iter->get_number_additional_bits(),
			target);
  }
}
void genezip_utils::huffman_encoded_stream_generator::add_value_to_vector(unsigned value, unsigned value_length, unsigned additional, unsigned additional_bits, std::vector<bool> &target) {
  //target.resize(target.size() + value_length + additional_bits);
  //the huffman code gets added msb first
  for (int i = (value_length - 1); i >= 0; --i) {
    target.push_back(value & (1 << i)); 
  }
  //everything else gets added lsb first
  for (unsigned i = 0; i < additional_bits; ++i) {
    target.push_back(additional & (1 << i));
  }
}
std::pair<unsigned, std::pair<unsigned, unsigned> >
genezip_utils::huffman_encoded_stream_generator::encode_length(unsigned length)
  const {
  if (length < 2 || length > 257)
    throw std::domain_error("genezip_utils::huffman_encoded_stream_generator"
			    "::encode_length: invalid length input: \""
			    + to_string<unsigned>(length) + "\"");
  unsigned true_length = length + 1;
  unsigned code = 0;
  unsigned nbit_content = 0;
  unsigned nbits = 0;
  if (true_length <= 10) {
    //remove 0 bits
    code = GENEZIP_LITERAL_VALUE_UPPER_BOUND + (true_length - 2);
    nbit_content = 0;
    nbits = 0;
  } else if (true_length <= 18) {
    code = (GENEZIP_LITERAL_VALUE_UPPER_BOUND + 9)/*265*/ + ((true_length - 11) >> 1);
    nbit_content = (true_length - 11) & ((1 << 1) - 1);
    nbits = 1;
  } else if (true_length <= 34) {
    code = (GENEZIP_LITERAL_VALUE_UPPER_BOUND + 13)/*269*/ + ((true_length - 19) >> 2);
    nbit_content = (true_length - 19) & ((1 << 2) - 1);
    nbits = 2;
  } else if (true_length <= 66) {
    code = (GENEZIP_LITERAL_VALUE_UPPER_BOUND + 17)/*273*/ + ((true_length - 35) >> 3);
    nbit_content = (true_length - 35) & ((1 << 3) - 1);
    nbits = 3;
  } else if (true_length <= 130) {
    code = (GENEZIP_LITERAL_VALUE_UPPER_BOUND + 21)/*277*/ + ((true_length - 67) >> 4);
    nbit_content = (true_length - 67) & ((1 << 4) - 1);
    nbits = 4;
  } else if (true_length <= 257) {
    code = (GENEZIP_LITERAL_VALUE_UPPER_BOUND + 25)/*281*/
      + ((true_length - 131) >> 5);
    nbit_content = (true_length - 131) & ((1 << 5) - 1);
    nbits = 5;
  } else {
    code = (GENEZIP_LITERAL_VALUE_UPPER_BOUND + 29)/*285*/;
    nbit_content = 0;
    nbits = 0;
  }
  return std::make_pair(code, std::make_pair(nbit_content, nbits));
}
std::pair<unsigned, std::pair<unsigned, unsigned> >
genezip_utils::huffman_encoded_stream_generator::encode_distance
(unsigned true_distance) const {
  if (true_distance < 1 || true_distance > GENEZIP_MAX_OFFSET_POINTER)
    throw std::domain_error("genezip_utils::huffman_encoded_stream_generator"
			    "::encode_distance: invalid distance input: \""
			    + to_string<unsigned>(true_distance) + "\"");
  unsigned code = 0;
  unsigned nbit_content = 0;
  unsigned nbits = 0;
  if (true_distance <= 4) {
    code = true_distance - 1;
    nbit_content = 0;
    nbits = 0;
  } else {
    unsigned starting_index = 8;
    unsigned counter = 1;
    for ( ; starting_index <= GENEZIP_MAX_OFFSET_POINTER; starting_index <<= 1, ++counter) {
      if (true_distance <= starting_index) {
	nbits = counter;
	code = ((counter + 1) << 1) + 
	  (((true_distance - 1) & (1 << counter)) ? 1 : 0);
	nbit_content = (true_distance - 1) & ((1 << counter) - 1);
	break;
      }
    }
  }
  return std::make_pair(code, std::make_pair(nbit_content, nbits));
}
