/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

/*!
  \file gzfile_block_data.h
  \brief decompress a stream of compressed binary data using existing
  huffman codes
 */
#ifndef __GENEZIP__GZFILE_BLOCK_DATA_H__
#define __GENEZIP__GZFILE_BLOCK_DATA_H__

#include <fstream>
#include <string>
#include <iostream>
#include <stdexcept>
#include <algorithm>
#include "genezip/binary_buffer.h"
#include "genezip/helper_functions.h"
#include "genezip/huffman_code.h"
#include "genezip/output_buffer.h"

namespace genezip_utils {
  /*!
    \typedef std::pair<unsigned, unsigned> literal_length_disambiguation
    \brief convenience typedef; first value is actual decoded value, second 
    is type flag
  */
  typedef std::pair<unsigned, unsigned> literal_length_disambiguation;
  /*!
    \class gzfile_block_data
    \brief decompress a stream of compressed binary data using existing 
    huffman codes.

    This class originally contained full DEFLATE/gzip file decompressor code, 
    including handling of different kinds of blocks.
  */
  class gzfile_block_data {
  public:
    /*!
      \enum LENGTH_LITERAL_TYPE
      \brief flag indicating the type of value decoded from the binary stream
    */
    typedef enum {
      //! interpret the value as an ASCII code
      RAW_CHARACTER,
      //! interpret the value as an end-of-block code
      END_OF_BLOCK,
      //! interpret the value as a match length
      LENGTH
    } LENGTH_LITERAL_TYPE;
    /*!
      \brief constructor
    */
    gzfile_block_data() {}
    /*!
      \brief destructor
    */
    ~gzfile_block_data() throw() {}
    /*!
      \brief decode a compressed binary vector into a character stream
      @param buffer wrapper around input binary vector
      @param litlen_code existing huffman code for literals and reverse 
      reference match lengths
      @param distance_code existing huffman code for reverse reference offsets
      @param obuffer destinatiion buffer for uncompressed literals
    */
    inline void parse_vector(binary_buffer      &buffer,
			     const huffman_code &litlen_code,
			     const huffman_code &distance_code,
			     output_buffer      &obuffer) {
      process_dynamic_compressed_block(buffer, 
				       litlen_code, 
				       distance_code, 
				       obuffer);
    }
  private:
    /*!
      \brief process a block with a dynamic huffman code (as opposed to 
      constant code in gzip specification)
      @param buffer wrapper around input binary vector
      @param litlen_code existing huffman code for literals and reverse 
      reference match lengths
      @param distance_code existing huffman code for reverse reference offsets
      \warning this is the actual workhorse for parse_vector.  This call 
      structure is a remnant of a previous, more complicated build
    */
    void process_dynamic_compressed_block(binary_buffer      &buffer,
					  const huffman_code &litlen_code,
					  const huffman_code &distance_code,
					  output_buffer      &obuffer);
    /*!
      \brief convert a decompressed huffman code to a literal, match length,
      or end-of-block code
      @param initial_attempt raw value returned from huffman decoding
      @param buffer wrapper around binary vector; needed if additional bits 
      must be arithmetically added to decoded value
      \return decoded value, and flag indicating type of value
    */
    literal_length_disambiguation interpret_literal_length(unsigned 
							   initial_attempt, 
							   binary_buffer &
							   buffer) const;
    /*!
      \brief convert a decompressed huffman code to a match offset
      @param stock_distance raw value returned from huffman decoding
      @param buffer wrapper around binary vector; needed if additional bits
      must be arithmetically added to decoded value
      \return decoded offset
    */
    unsigned process_distance(unsigned stock_distance, binary_buffer &buffer)
      const;
  };
}
#endif //__GZFILE_BLOCK_DATA_H__
