/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

/*!
  \file general_input_hash.h
  \brief an implementation of custom_hash_function.h.  Used for 
  compressor hashing.
 */
#ifndef __GENEZIP__GENERAL_INPUT_HASH_H__
#define __GENEZIP__GENERAL_INPUT_HASH_H__

namespace genezip_utils {
  /*!
    \class general_input_hash
    \brief implementation of custom_hash_function.  Used for compressor
    hashing.
  */
  class general_input_hash : public custom_hash_function {
  public:
    /*!
      \brief default constructor
    */
    general_input_hash() : custom_hash_function() {}
    /*!
      \brief destructor
    */
    ~general_input_hash() throw() {}
    /*!
      \brief hash two integers on [0,1000]
      \warning only a perfect hash if integers are on [0,1000]; will
      function regardless, but not perfectly
      @param u1 first integer to be hashed
      @param u2 second integer to be hashed
      \return hashed value; modded to fit in domain specified by max_hash()
    
      GENEZIP_LITERAL_VALUE_UPPER_BOUND is currently a global
      macro that allows the flexible definition of a literal.
      Eventually this will be properly parametrized, so the compressor can 
      operate on runtime-specified probabilities, dosages, or genotypes
    */
    virtual inline unsigned hashify(unsigned u1, unsigned u2) const {
      return (u1 * GENEZIP_LITERAL_VALUE_UPPER_BOUND + u2) % 
	GENEZIP_GENERAL_INPUT_HASH_MAX_VALUE;
    }
    /*!
      \brief return the maximum hash value possibly returned by this hash, +1
      \return maximum hash value possibly returned by this hash, +1
    */
    virtual inline unsigned max_hash() const {
      return GENEZIP_GENERAL_INPUT_HASH_MAX_VALUE;
    }
  private:
  };
}
#endif //__GENERAL_INPUT_HASH_H__
