/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

#ifndef __GENEZIP__FILE_IO_HANDLER_H__
#define __GENEZIP__FILE_IO_HANDLER_H__

#include <fstream>
#include <string>
#include <stdexcept>
#include <iostream>
#include "genezip/config.h"
#ifdef HAVE_LIBZ
#include "zlib.h"
#endif //HAVE_LIBZ
namespace genezip_utils {
  //! a file reading class, wrapping fstream and zlib in a common interface
  class file_io_handler {
  public:
    //! enum describing read or write mode for an instance of the class
    typedef enum {
      READ, WRITE, READGZ, WRITEGZ, NA
    } io_type;
    //! default constructor, handling zlib pointer
    file_io_handler();
    //! constructor, opening file based on read/write mode provided
    file_io_handler(const std::string &filename, io_type mode);
    //! destructor, closing the file stream if needed
    ~file_io_handler() throw() {close();}
    //! test the "open" status of the internal stream
    //! \return whether the object has an open stream
    bool is_open() const;
    //! open a file in a given read/write mode
    //! \param filename name of file to open
    //! \param mode enum describing read/write mode to be used
    void open(const std::string &filename, io_type mode);
    //! close any open stream
    void close();
    //! read a line from an open read stream
    //! \param target where the read line should be stored
    //! \return whether a line was successfully read (false ~= EOF)
    bool readline(std::string &target);
    //! write a line to an open write stream
    //! \param line the line to be written
    //! \return whether line was successfully written (false ~= no disk space)
    bool writeline(const std::string &line);
    //! set which character(s) are used as "newline"
    //! \param newline_characters characters to be used for "newline"
    inline void set_newline_characters(const std::string &newline_characters) {
      _newline_chars = newline_characters;
    }
    //! get which character(s) are used as "newline"
    //! \return characters used for "newline"
    inline const std::string &get_newline_characters() const {
      return _newline_chars;
    }
  private:
    //! determine whether the object is in read mode
    //! \return whether the object is in read mode
    inline bool is_reading() const {
      return _readmode;
    }
    //! flip the object from read->write, or write->read, depending on state
    inline void swap_reading() {_readmode = !_readmode;}
    //! set the read state of the current object
    //! \param val whether the new state should be "read"
    inline void set_reading(bool val) {_readmode = val;}
    //! set the object's newline characters to the default for the system
    void default_newline();
    bool _readmode; //!< whether the object is in read mode
    std::string _newline_chars; //!< characters used for newline
    std::fstream _input; //!< fstream read/write stream
#ifdef HAVE_LIBZ
    gzFile _gzinput; //!< zlib read/write stream
#endif //HAVE_LIBZ
  };
}
#endif //__FILE_IO_HANDLER_H__
