/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

#include "genezip/file_io_handler.h"
genezip_utils::file_io_handler::file_io_handler()
  : _readmode(true) {
#ifdef HAVE_LIBZ
  _gzinput = 0;
#endif //HAVE_LIBZ
  default_newline();
}
genezip_utils::file_io_handler::file_io_handler(const std::string &filename, 
						io_type            mode)
  : _readmode(true) {
#ifdef HAVE_LIBZ
  _gzinput = 0;
#endif //HAVE_LIBZ
  default_newline();
  open(filename, mode);
}
bool genezip_utils::file_io_handler::is_open() const {
  bool res = _input.is_open();
#ifdef HAVE_LIBZ
  res |= (_gzinput != 0);
#endif //HAVE_LIBZ
  return res;
}
void genezip_utils::file_io_handler::open(const std::string &filename,
					  io_type            mode) {
  if (is_open())
    throw std::domain_error("genezip_utils::file_io_handler::open: "
			    "called with preexisting stream open");
  if (mode == READ || mode == WRITE) {
    _input.open(filename.c_str(), mode == READ ? std::ios_base::in 
		: std::ios_base::out);
    if (!_input.is_open())
      throw std::domain_error("genezip_utils::file_io_handler::open: unable "
			      "to open file \"" + filename + "\"");
    set_reading(mode == READ);
  } else if (mode == READGZ || mode == WRITEGZ) {
#ifdef HAVE_LIBZ
    if (!(_gzinput = gzopen(filename.c_str(), mode == READGZ ? "rb" : "wb")))
      throw std::domain_error("genezip_utils::file_io_handler::open: unable "
			      "to open file \"" + filename + "\"");
    set_reading(mode == READGZ);
#else
    throw std::domain_error("genezip_utils::file_io_handler::open: not "
			    "compiled with zlib support enabled");
#endif //HAVE_LIBZ
  } else {
    throw std::domain_error("genezip_utils::file_io_handler::open: "
			    "unrecognized open mode");
  }
}
void genezip_utils::file_io_handler::close() {
  if (_input.is_open()) {
    _input.close();
    _input.clear();
  }
#ifdef HAVE_LIBZ
  if (_gzinput) {
    gzclose(_gzinput);
    _gzinput = 0;
  }
#endif //HAVE_LIBZ
}
bool genezip_utils::file_io_handler::readline(std::string &target) {
  if (!is_reading())
    throw std::domain_error("genezip_utils::file_io_handler::readline: "
			    "stream is write-only");
  if (_input.is_open()) {
    if (_input.peek() != EOF) {
      getline(_input, target);
      return true;
    } else return false;
  }
#ifdef HAVE_LIBZ
  unsigned orig_len = 0, buffer_size = 100000;
  if (!buffer_size) throw std::domain_error("genezip_utils::file_io_handler"
					    "::readline: invalid buffer size");
  target = "";
  char buffer[buffer_size];
  if (_gzinput) {
    while (true) {
      //TODO: this end condition is completely wrong
      if (gzgets(_gzinput, buffer, buffer_size) != Z_NULL) {
	orig_len = target.size();
	target += std::string(buffer);
	if (target.size() - orig_len != buffer_size - 1 ||
	    *target.rbegin() == '\n')
	  break;
      } else break;
    }
    return !target.empty();
  }
#endif //HAVE_LIBZ
  throw std::domain_error("genezip_utils::file_io_handler::readline: "
			  "stream is not open");
}
bool genezip_utils::file_io_handler::writeline(const std::string &line) {
  if (is_reading())
    throw std::domain_error("genezip_utils::file_io_handler::writeline: "
			    "stream is read-only");
  if (_input.is_open()) {
    return (_input << line << get_newline_characters());
  }
#ifdef HAVE_LIBZ
  if (_gzinput) {
    return (gzputs(_gzinput, line.c_str()) < static_cast<int>(line.size()) ||
	    gzputs(_gzinput, get_newline_characters().c_str()) < 
	    static_cast<int>(get_newline_characters().size()));
  }
#endif //HAVE_LIBZ
  throw std::domain_error("genezip_utils::file_io_handler::writeline: "
			  "stream is not open");
}
void genezip_utils::file_io_handler::default_newline() {
#ifdef _WIN32
  //std::cout << "windows" << std::endl;
  _newline_chars = "\r\n";
#elif defined macintosh
  //std::cout << "old mac" << std::endl;
  _newline_chars = "\r";
#else
  //std::cout << "unix" << std::endl;
  _newline_chars = "\n";
#endif //
}
