/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

/*!
  \file compression_handler.cc
 */

#include "genezip/compression_handler.h"

void genezip_utils::compression_handler::enqueue_data
(const std::vector<unsigned> &preproc_data) {
  if (_finalized)
    throw std::domain_error("compression_handler: cannot compress"
			    " more data once dataset has been"
			    " finalized");
  if (_vectors_in_current_bin >= max_bin_size()) {
    compress_current_data();
  }
  if (!_vectors_in_current_bin) {
    if (!_compressor)
      _compressor = new gzfile_compress(_current_data.bits_per_element());
    smart_pointer<huffman_code> litlen_ptr(new huffman_code);
    smart_pointer<huffman_code> distance_ptr(new huffman_code);
    smart_pointer<gtype_vector> gtype_ptr(new gtype_vector);
    _compressed_types.push_back(gtype_ptr);
    _literal_length_codes.push_back(litlen_ptr);
    _distance_codes.push_back(distance_ptr);    
  }
  //unsigned current_index = _current_data.size();
  //_current_data.reserve(_current_data.size() + preproc_data.size());
  for (unsigned i = 0; i < preproc_data.size(); ++i) {
    _compressor->compress_current_value(preproc_data.at(i),
					**_compressed_types.rbegin(),
					**_literal_length_codes.rbegin(),
					**_distance_codes.rbegin());
    //_current_data.push_back(preproc_data.at(i));
  }
  ++_vectors_in_current_bin;
}

void genezip_utils::compression_handler::compress_current_data() {
  //if (!_current_data.empty()) {
    /*
    gzfile_compress compressor;
    smart_pointer<huffman_code> litlen_ptr(new huffman_code);
    smart_pointer<huffman_code> distance_ptr(new huffman_code);
    smart_pointer<gtype_vector> gtype_ptr(new gtype_vector);
    */
  _compressor->finish_value_stream(**_compressed_types.rbegin(),
				   **_literal_length_codes.rbegin(),
				   **_distance_codes.rbegin());
  //std::cout << "deallocating compressor" << std::endl;
  _compressor->clear();
  //_compressor = NULL;
  //_current_data.clear();
  _vectors_in_current_bin = 0;
  //}
}

void genezip_utils::compression_handler::finalize() {
  //if (!_current_data.empty()) {
  //  if (_vectors_in_current_bin)
  compress_current_data();
    //}
  _finalized = true;
}

void genezip_utils::compression_handler::get_vector_of_data(unsigned index,
							    unsigned size,
							    std::vector
							    <unsigned> &target)
{
  //calculate the bin to which the requested index belongs
  unsigned requested_bin = index / max_bin_size();
  //bounds of the data within the bin
  unsigned data_start = (index % max_bin_size()) * size;
  unsigned data_end = data_start + size;
  if (requested_bin >= number_stored_bins())
    throw std::domain_error("compression_handler: requested bin "
			    "out of bounds (requested="
			    + to_string<unsigned>(requested_bin) +
			    ", stored=" +
			    to_string<unsigned>(number_stored_bins()) + ")");
  //determine if the requested vector is in the currently extracted bin
  if (_extracted_bin.empty() ||
      _extracted_bin_index != requested_bin) {
    _extracted_bin_index = requested_bin;
    //TODO: the actual extraction
    gzfile_decompress decompressor;
    _extracted_bin.clear();
    decompressor.decompress_to_vector(*(_compressed_types.at(requested_bin)),
				      *(_literal_length_codes.at(requested_bin)),
				      *(_distance_codes.at(requested_bin)),
				      _extracted_bin,
				      max_bin_size() * size);
  }
  //make sure there are actually enough entries
  if (data_end > _extracted_bin.size())
    throw std::domain_error("compression_handler: requested bin "
			    "does not contain sufficient entries "
			    "to contain your vector!");
  //copy the data out to result vector
  target.resize(size);
  for (unsigned i = 0; i < size; ++i) {
    _extracted_bin.at(i + data_start, target.at(i));
  }
}
